package edu.harvard.med.countway.dl.servlet;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.naming.NamingException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;

import edu.harvard.med.countway.auth.ldap.HarvardLdapAuthorizer;
import edu.harvard.med.countway.auth.ldap.HarvardLdapAuthorizer.UserClass;
import edu.harvard.med.countway.config.CountwayConfig;
import edu.harvard.med.countway.config.LoginType;
import edu.harvard.med.countway.config.CountwayConfig.PropertyKey;
import edu.harvard.med.countway.dl.dao.EcommonsUserDAO;
import edu.harvard.med.countway.tools.MailUtil;

public class HuidauthzrequestServlet extends AbstractServlet
{
    private static final long serialVersionUID = 276398509918120902L;
    private static final Logger log = Logger.getLogger(HuidauthzrequestServlet.class);
    
    private static final String loginUrl = "login.html?url=huidauthzrequest.html";
    private static final String jspUrl = "huidauthzrequest.jspx";
    private static final String errorUrl = "error.jspx";
    
    public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws IOException, ServletException
    {
        try
        {
            setDummySessionAttributes(request);
            HttpSession session = request.getSession();
            EcommonsUserDAO ecuserDao = new EcommonsUserDAO();
            // session vars
            String uid = (String) session.getAttribute("uid");
            LoginType loginType = (LoginType) session.getAttribute("loginType");
            String email = (String) session.getAttribute("email");
            // request params
            String huid = request.getParameter("huid");
            String submit = request.getParameter("submit");
            // user messages
            List<String> messages = new ArrayList<String>();
            // redirect to login if not logged in
            if (uid == null || uid.trim().equals(""))
            {
                response.sendRedirect(loginUrl);
                return;
            }
            // check for ecid login
            if (!loginType.equals(LoginType.ecid))
            {
                messages.add("This request mechanism does not apply to you since you are not logged in with an eCommons ID");
                request.setAttribute("messages", messages);
                request.setAttribute("printform", false);
                request.getRequestDispatcher(jspUrl).forward(request, response);
                return;
            }
            // check for email
            if (email == null || email.trim().equals(""))
            {
                messages.add("This request can't be completed because an e-mail address was not associated with your login session");
                request.setAttribute("messages", messages);
                request.setAttribute("printform", false);
                request.getRequestDispatcher(jspUrl).forward(request, response);
                return;
            }
            // check for authorized huid associated with this ecid
            // TODO: this could also be done by just checking the dlaccess session attribute
            if (!CountwayConfig.getProperty(PropertyKey.DUMMY_SESSION).equals("true"))
            {
                String associatedHuid = ecuserDao.getHuidFromEcid(uid);
                if (associatedHuid != null)
                {
                    HarvardLdapAuthorizer authz = new HarvardLdapAuthorizer(associatedHuid);
                    if (authz.isAuthorized())
                    {
                        messages.add("The HUID ("+associatedHuid+") associated with the eCommons ID ("+uid+") you used to login is already authorized for access to licensed electronic library resources.  As such you should already have access to licensed resources.  If you don't seem to have access, please <a href=\"logout.html\">logout</a> and then <a href=\"login.html\">login</a> with this HUID to access licensed resources.");
                        request.setAttribute("messages", messages);
                        request.setAttribute("printform", false);
                        request.getRequestDispatcher(jspUrl).forward(request, response);
                        return; 
                    }  
                }         
            }
            // handle form submission
            if (submit != null)
            {
                // check for empty huid
                if (huid == null || huid.trim().equals(""))
                {
                    messages.add("A valid HUID is required to proceed");
                    request.setAttribute("messages", messages);
                    request.setAttribute("printform", true);
                    request.getRequestDispatcher(jspUrl).forward(request, response);
                    return;                
                }
                // check if huid is already associated with another ecid
                if (!CountwayConfig.getProperty(PropertyKey.DUMMY_SESSION).equals("true"))
                {
                    String associatedEcid = ecuserDao.getEcidFromHuid(huid);
                    if (associatedEcid != null)
                    {
                        messages.add("The HUID you entered is already associated with another eCommons ID");
                        request.setAttribute("messages", messages);
                        request.setAttribute("printform", true);
                        request.getRequestDispatcher(jspUrl).forward(request, response);
                        return;                
                    }
                }
                // check if huid is authorized
                HarvardLdapAuthorizer authz = new HarvardLdapAuthorizer(huid);
                if (!authz.isAuthorized())
                {
                    messages.add("The HUID you entered is either invalid or not authorized for access to licensed electronic library resources");
                    request.setAttribute("messages", messages);
                    request.setAttribute("printform", true);
                    request.getRequestDispatcher(jspUrl).forward(request, response);
                    return; 
                }
                // get attributes and construct e-mail message
                UserClass userClass = authz.getUserClass();
                HashMap<String,Object> ecAttrs = ecuserDao.selectHuidAuthzAttributes(uid);
                HashMap<String,ArrayList<String>> huAttrs = authz.getAttributes();
                StringBuffer body = new StringBuffer();
                body.append("A Countway user with eCommons ID:\n\n");
                body.append("  "+uid+"\n\n");
                body.append("has requested that the following DL-authorized Harvard ID be associated with their eCommons account:\n\n");
                body.append("  "+huid+"\n\n");
                body.append("This Harvard ID has the following HUL/OIS eligible user class:\n\n");
                body.append("  "+userClass+"\n\n");
                body.append("ECOMMONS ATTRIBUTES\n\n");
                appendEcommonsAttribute("addr_num",body,ecAttrs);
                appendEcommonsAttribute("ldap_username",body,ecAttrs);
                appendEcommonsAttribute("harvard_id",body,ecAttrs);
                appendEcommonsAttribute("full_name",body,ecAttrs);
                appendEcommonsAttribute("institution",body,ecAttrs);
                appendEcommonsAttribute("prime_dept_desc50",body,ecAttrs);
                appendEcommonsAttribute("academic_title",body,ecAttrs);
                appendEcommonsAttribute("administrative_title",body,ecAttrs);
                appendEcommonsAttribute("line1",body,ecAttrs);
                appendEcommonsAttribute("line2",body,ecAttrs);
                appendEcommonsAttribute("line3",body,ecAttrs);
                appendEcommonsAttribute("city",body,ecAttrs);
                appendEcommonsAttribute("state",body,ecAttrs);
                appendEcommonsAttribute("telephone",body,ecAttrs);
                appendEcommonsAttribute("email",body,ecAttrs);
                appendEcommonsAttribute("data_source",body,ecAttrs);
                appendEcommonsAttribute("data_source_detail",body,ecAttrs);
                appendEcommonsAttribute("is_active",body,ecAttrs);
                appendEcommonsAttribute("created_dt",body,ecAttrs);
                appendEcommonsAttribute("updated_dt",body,ecAttrs);
                appendEcommonsAttribute("inactivated_dt",body,ecAttrs);
                body.append("\n");
                body.append("HU-LDAP ATTRIBUTES\n\n");
                appendHarvardLdapAttribute("harvardeduidnumber",body,huAttrs);
                appendHarvardLdapAttribute("cn",body,huAttrs);
                appendHarvardLdapAttribute("title",body,huAttrs);
                appendHarvardLdapAttribute("edupersonaffiliation",body,huAttrs);
                appendHarvardLdapAttribute("harvardedudepartmentaffiliation",body,huAttrs);
                appendHarvardLdapAttribute("postalAddress",body,huAttrs);
                appendHarvardLdapAttribute("mail",body,huAttrs);
                final String to;
                if (!CountwayConfig.getProperty(PropertyKey.DEPLOY_TIER).equals("prod") &&
                    !CountwayConfig.getProperty(PropertyKey.DEPLOY_TIER).equals("stage"))
                {
                    to = CountwayConfig.getProperty(PropertyKey.GENERAL_NOTIFICATION_TO_EMAIL);
                }
                else
                {
                    to = CountwayConfig.getProperty(PropertyKey.HUID_AUTHZ_REQUEST_EMAIL);
                }
                MailUtil.sendMail(
                    email,
                    to,
                    "Countway request for association of Harvard ID ["+huid+"] with eCommons ID ["+uid+"]",
                    body.toString(),
                    CountwayConfig.getProperty(PropertyKey.GENERAL_NOTIFICATION_TO_EMAIL)
                );
                messages.add("Your request has been successfully submitted to the HMS IT help desk.  Someone will follow up with you shortly.");
                request.setAttribute("messages", messages);
                request.setAttribute("printform", false);
            }
            else
            {
                request.setAttribute("printform", true);
            }
            request.getRequestDispatcher(jspUrl).forward(request, response);
        }
        catch (SQLException e)
        {
            request.setAttribute("javax.servlet.error.exception", e);
            request.setAttribute("javax.servlet.jsp.jspException", e);
            request.getRequestDispatcher(errorUrl).forward(request, response);
        }
        catch (NamingException e)
        {
            request.setAttribute("javax.servlet.error.exception", e);
            request.setAttribute("javax.servlet.jsp.jspException", e);
            request.getRequestDispatcher(errorUrl).forward(request, response);
        }
    }
    
    private static void appendEcommonsAttribute(String attr, StringBuffer buffer, HashMap<String,Object> ecAttrs)
    {
        if (ecAttrs.containsKey(attr))
        {
            String value = ecAttrs.get(attr).toString();
            if (value != null && !value.trim().equals(""))
            {
                buffer.append(attr+": "+value+"\n");
            }
        }        
    }
    
    private static void appendHarvardLdapAttribute(String attr, StringBuffer buffer, HashMap<String,ArrayList<String>> huAttrs)
    {
        if (huAttrs.containsKey(attr))
        {
            ArrayList<String> values = huAttrs.get(attr);
            if (values.size() == 1)
            {
                buffer.append(attr+": "+values.get(0)+"\n");
            }
            else if (values.size() > 1)
            {
                buffer.append(attr+":\n");
                for (String value: values)
                {
                    buffer.append("  "+value+"\n");
                }                
            }
        }        
    }
}
