package edu.harvard.med.countway.dl.servlet;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;

import edu.harvard.med.countway.auth.AuthResponseCode;
import edu.harvard.med.countway.auth.ecommons.HmsaccessAuthenticator;
import edu.harvard.med.countway.auth.ecommons.HmsaccessCookie;
import edu.harvard.med.countway.auth.ecommons.HmsaccessCookieValue;
import edu.harvard.med.countway.auth.hul.HulaccessCookie;
import edu.harvard.med.countway.auth.pin.HarvardPinAuthenticator;
import edu.harvard.med.countway.config.CountwayConfig;
import edu.harvard.med.countway.config.LoginType;
import edu.harvard.med.countway.config.CountwayConfig.PropertyKey;

public class LoginServlet extends AbstractServlet
{
    private static final long serialVersionUID = 7379082040779683863L;
    private static final Logger log = Logger.getLogger(LoginServlet.class);
    private static final boolean DEBUG = log.isDebugEnabled();
    
    private static final String jspUrl = "login.jspx";
    
    public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws IOException, ServletException
    {
        setDummySessionAttributes(request);
        HttpSession session = request.getSession();
        String pinUrl = "https://www.pin1.harvard.edu/pin/authenticate?__authen_application="+CountwayConfig.getProperty(PropertyKey.PIN_APP_NAME);
        // session vars
        String uid = (String) session.getAttribute("uid");
        // request params
        String url = request.getParameter("url");
        String pinParams = request.getParameter("__authen_parameters");
        String pinPgpSig = request.getParameter("__authen_pgp_signature");
        // hmsaccess cookie
        Cookie hmsaccess = getHmsaccessCookie(request);
        // set url and pinUrl
        try
        {
            if (url != null && !url.trim().equals(""))
            {
                pinUrl += "&url="+URLEncoder.encode(url, "UTF-8");
            }
            else
            {
                pinUrl += "&url="+URLEncoder.encode("index.html", "UTF-8");
                url = "index.html";
            }
        }
        catch (UnsupportedEncodingException e)
        {
            throw new RuntimeException(e);
        }
        // already logged in
        if (uid != null && !uid.trim().equals(""))
        {
            response.sendRedirect(url);
            return;
        }
        // login via pin
        if (pinParams != null && !pinParams.trim().equals("") && pinPgpSig != null && !pinPgpSig.trim().equals(""))
        {
            HarvardPinAuthenticator authn = new HarvardPinAuthenticator(pinParams, pinPgpSig, request.getRemoteAddr());
            AuthResponseCode responseCode = authn.getResponseCode();
            Boolean dlaccess = authn.getDlaccess();
            LoginType loginType = authn.getLoginType();
            if (responseCode.equals(AuthResponseCode.INVALID_PIN_TOKEN) ||
                responseCode.equals(AuthResponseCode.EXPIRED_PIN_TOKEN) ||
                responseCode.equals(AuthResponseCode.INVALID_IP_ADDRESS))
            {
                // pin token problem, redirect to pin
                response.sendRedirect(pinUrl);
                return;
            }
            else if (responseCode.equals(AuthResponseCode.ECOMMONS_HUID_LOOKUP_FAILED) ||
                     responseCode.equals(AuthResponseCode.ECOMMONS_HUID_NOT_AVAILABLE) ||
                     responseCode.equals(AuthResponseCode.HUID_LDAP_AUTH_FAILED) ||
                     responseCode.equals(AuthResponseCode.DL_ACCESS_DENIED) ||
                     responseCode.equals(AuthResponseCode.SUCCESS))
            {
                // login
                session.setAttribute("uid", authn.getUid());
                session.setAttribute("loginType", authn.getLoginType());
                session.setAttribute("firstName", authn.getFirstName());
                session.setAttribute("lastName", authn.getLastName());
                session.setAttribute("name", authn.getName());
                session.setAttribute("email", authn.getEmail());
                session.setAttribute("dluid", authn.getDluid());
                session.setAttribute("dlaccess", dlaccess ? "true" : "false");
                response.addCookie(new Cookie("login","login"));
                if (dlaccess)
                {
                    response.addCookie(new HulaccessCookie(request.getRemoteAddr(),authn.getLoginType(),authn.getUid(),session.getId()));
                }
                if (loginType != null && loginType.equals(LoginType.ecid))
                {
                    response.addCookie(new HmsaccessCookie(authn.getUid(),request.getRemoteAddr()));
                }
                if (responseCode.equals(AuthResponseCode.ECOMMONS_HUID_LOOKUP_FAILED) ||
                    responseCode.equals(AuthResponseCode.ECOMMONS_HUID_NOT_AVAILABLE) ||
                    responseCode.equals(AuthResponseCode.HUID_LDAP_AUTH_FAILED) ||
                    responseCode.equals(AuthResponseCode.DL_ACCESS_DENIED))
                {
                    request.setAttribute("responseCode", responseCode);
                }
                else if (responseCode.equals(AuthResponseCode.SUCCESS))
                {
                    response.sendRedirect(url);
                    return;
                }
            }
            else
            {
                // USER_DB_UPDATE_ERROR and UNKNOWN
                request.setAttribute("responseCode", responseCode);
            }
            request.getRequestDispatcher(jspUrl).forward(request, response);
            return;
        }
        // login via hmsaccess cookie
        if (hmsaccess != null)
        {
            HmsaccessAuthenticator authn = new HmsaccessAuthenticator(hmsaccess, request.getRemoteAddr());
            AuthResponseCode responseCode = authn.getResponseCode();
            Boolean dlaccess = authn.getDlaccess();
            if (responseCode.equals(AuthResponseCode.INVALID_HMSACCESS_COOKIE) ||
                responseCode.equals(AuthResponseCode.EXPIRED_HMSACCESS_COOKIE) ||
                responseCode.equals(AuthResponseCode.INVALID_IP_ADDRESS))
            {
                // hmsaccess cookie problem, redirect to pin
                response.sendRedirect(pinUrl);
                return;
            }
            else if (responseCode.equals(AuthResponseCode.ECOMMONS_HUID_LOOKUP_FAILED) ||
                     responseCode.equals(AuthResponseCode.ECOMMONS_HUID_NOT_AVAILABLE) ||
                     responseCode.equals(AuthResponseCode.HUID_LDAP_AUTH_FAILED) ||
                     responseCode.equals(AuthResponseCode.DL_ACCESS_DENIED) ||
                     responseCode.equals(AuthResponseCode.SUCCESS))
            {
                // login
                session.setAttribute("uid", authn.getUid());
                session.setAttribute("loginType", authn.getLoginType());
                session.setAttribute("firstName", authn.getFirstName());
                session.setAttribute("lastName", authn.getLastName());
                session.setAttribute("name", authn.getName());
                session.setAttribute("email", authn.getEmail());
                session.setAttribute("dluid", authn.getDluid());
                session.setAttribute("dlaccess", dlaccess ? "true" : "false");
                response.addCookie(new Cookie("login","login"));
                if (dlaccess)
                {
                    response.addCookie(new HulaccessCookie(request.getRemoteAddr(),authn.getLoginType(),authn.getUid(),session.getId()));
                }
                response.addCookie(new HmsaccessCookie(authn.getUid(),request.getRemoteAddr()));
                if (responseCode.equals(AuthResponseCode.ECOMMONS_HUID_LOOKUP_FAILED) ||
                    responseCode.equals(AuthResponseCode.ECOMMONS_HUID_NOT_AVAILABLE) ||
                    responseCode.equals(AuthResponseCode.HUID_LDAP_AUTH_FAILED) ||
                    responseCode.equals(AuthResponseCode.DL_ACCESS_DENIED))
                {
                    request.setAttribute("responseCode", responseCode);
                }
                else if (responseCode.equals(AuthResponseCode.SUCCESS))
                {
                    response.sendRedirect(url);
                    return;
                }
            }
            else
            {
                // USER_DB_UPDATE_ERROR and UNKNOWN
                request.setAttribute("responseCode", responseCode);
            }
            request.getRequestDispatcher(jspUrl).forward(request, response);
            return;
        }
        // redirect to pin
        response.sendRedirect(pinUrl);
    }
    
    private static Cookie getHmsaccessCookie(final HttpServletRequest request)
    {
        final Cookie[] cookies = request.getCookies();
        
        if (cookies == null)
        {
            return null;
        }
        
        Cookie hmsaccess = null;
        for (final Cookie cookie: cookies)
        {
            // getDomain() and getPath() always return null so check getName() only
            final String name = cookie.getName();
            final String value = cookie.getValue();
            if (name != null && name.equals("hmsaccess") && value != null)
            {
                if (DEBUG)
                {
                    log.debug("hmsaccess encoded value: "+value);
                    log.debug("hmsaccess decoded value: "+HmsaccessCookieValue.decodeValue(value));
                }
                hmsaccess = cookie;
                break;
            }
        }
        
        return hmsaccess;
    }
}
