package edu.harvard.med.countway.dl.servlet;

import java.io.IOException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;

import edu.harvard.med.countway.config.CountwayConfig;
import edu.harvard.med.countway.config.CountwayConfig.PropertyKey;
import edu.harvard.med.countway.dl.dao.ClassDAO;
import edu.harvard.med.countway.dl.dao.ClassUserDAO;
import edu.harvard.med.countway.dl.model.Class;
import edu.harvard.med.countway.tools.IntegerUtil;
import edu.harvard.med.countway.tools.MailUtil;
import edu.harvard.med.countway.tools.TemplateUtil;

public class RegisterServlet extends AbstractServlet
{
    private static final long serialVersionUID = -6624861358513360679L;
    private static final Logger log = Logger.getLogger(RegisterServlet.class);
    
    private static final String loginUrl = "../../../login.html?url=menuNavigation/libraryServices/classes/register.html";
    private static final String cancelUrl = "../classes.html";
    private static final String jspUrl = "register.jspx";
    private static final String errorUrl = "../../../error.jspx";
    
    public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws IOException, ServletException
    {
        try
        {
            setDummySessionAttributes(request);
            HttpSession session = request.getSession();
            ClassUserDAO classUserDao = new ClassUserDAO();
            ClassDAO classDao = new ClassDAO();
            // session vars
            Integer dluid = (Integer) session.getAttribute("dluid");
            String uid = (String) session.getAttribute("uid");
            // request params
            // insert
            String insert = request.getParameter("insert");
            String classId = request.getParameter("classId");
            String firstName = request.getParameter("firstName");
            String lastName = request.getParameter("lastName");
            String affiliation = request.getParameter("affiliation");
            String emailprm = request.getParameter("email");
            // cancel
            String cancel = request.getParameter("cancel");
            // user messages
            List<String> messages = new ArrayList<String>();
            if (uid == null || uid.trim().equals(""))
            {
                response.sendRedirect(loginUrl+"?classId="+classId);
                return;
            }
            if (classId == null || classId.trim().equals("") || !IntegerUtil.isValidInteger(classId) || !classDao.classExists(Integer.parseInt(classId)))
            {
                messages.add("This class does not exist or is invalid!");
                request.setAttribute("messages", messages);
                request.getRequestDispatcher(jspUrl).forward(request, response);
                return;
            }
            if (cancel != null)
            {
                response.sendRedirect(cancelUrl);
                return;
            }
            Class clazz = classDao.selectClass(Integer.parseInt(classId));
            if (classUserDao.classUserExists(clazz.getId(),dluid))
            {
                final String msg = TemplateUtil.getClassAlreadyRegisteredMsg(clazz.getCourse().getTitle(), clazz.getTime(), clazz.getLocation().getConcat());
                messages.add(msg);
                request.setAttribute("messages", messages);
                request.getRequestDispatcher(jspUrl).forward(request, response);
                return;
            }        
            if (insert != null)
            {
                // firstName/lastName/affiliation not required
                if (emailprm == null || emailprm.trim().equals(""))
                {
                    messages.add("E-mail is required");
                    request.setAttribute("messages", messages);
                    request.setAttribute("class", clazz);
                    request.getRequestDispatcher(jspUrl).forward(request, response);
                    return;
                }
                if (classUserDao.insertClassUser(clazz.getId(),dluid,firstName,lastName,affiliation,emailprm) != null)
                {
                    String regAction = "registered";
                    if (classUserDao.classUserIsEnrolled(clazz.getId(),dluid))
                    {
                        regAction = "enrolled";
                    }
                    else if (classUserDao.classUserIsWaitlisted(clazz.getId(),dluid))
                    {
                        regAction = "waitlisted";
                    }
                    final String msg = TemplateUtil.getClassRegisterMsg(regAction, clazz.getCourse().getTitle(), clazz.getTime(), clazz.getLocation().getConcat());
                    messages.add(msg);
                    try
                    {
                        final String to;
                        if (!CountwayConfig.getProperty(PropertyKey.DEPLOY_TIER).equals("prod") &&
                            !CountwayConfig.getProperty(PropertyKey.DEPLOY_TIER).equals("stage"))
                        {
                            to = CountwayConfig.getProperty(PropertyKey.GENERAL_NOTIFICATION_TO_EMAIL);
                        }
                        else
                        {
                            to = emailprm;
                        }
                        final String emailmsg = TemplateUtil.getClassRegisterEmail(regAction, clazz.getCourse().getTitle(), clazz.getTime(), clazz.getDuration(), clazz.getLocation().getConcat(), request.getScheme(), request.getServerName());
                        MailUtil.sendMail(CountwayConfig.getProperty(PropertyKey.CLASS_NOTIFICATION_FROM_EMAIL), to, "Countway Class Registration Notification", emailmsg, CountwayConfig.getProperty(PropertyKey.GENERAL_NOTIFICATION_TO_EMAIL));
                        messages.add("A registration confirmation has been e-mailed to: "+to);
                    }
                    catch (IOException e)
                    {
                        messages.add("Error sending registration confirmation e-mail");
                    }
                    request.setAttribute("messages", messages);
                    request.getRequestDispatcher(jspUrl).forward(request, response);
                    return;
                }
                else
                {
                    messages.add("Couldn't register you for \""+clazz.getCourse().getTitle()+"\"");
                }
            }
            request.setAttribute("messages", messages);
            request.setAttribute("class", clazz);
            request.getRequestDispatcher(jspUrl).forward(request, response);
        }
        catch (SQLException e)
        {
            request.setAttribute("javax.servlet.error.exception", e);
            request.setAttribute("javax.servlet.jsp.jspException", e);
            request.getRequestDispatcher(errorUrl).forward(request, response);
        }
    }
}
