package edu.harvard.med.countway.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.util.Properties;

import org.apache.log4j.Logger;

public class FileUtil
{
    private static final Logger log = Logger.getLogger(FileUtil.class);
    
    public static Properties getProperties(final String fileName) throws IOException
    {
        return getProperties(getResourceStream(fileName));
    }
    
    public static Properties getProperties(final File file) throws IOException
    {
        final Properties props = new Properties();
        
        props.load(new FileInputStream(file));
        
        return props;
    }
    
    public static Properties getProperties(final InputStream fileStream) throws IOException
    {
        final Properties props = new Properties();
        
        props.load(fileStream);
        
        return props;
    }

    public static InputStream getResourceStream(final String fileName) throws IOException
    {
        final InputStream fileStream;
        
        final URL url = getResourceUrl(fileName);

        if (url.getProtocol().equals("jar"))
        {
            // resource in jar, get from classloader
            fileStream = getClassLoader().getResourceAsStream(fileName);
        }
        else
        {
            // real resource file
            fileStream = new FileInputStream(url.getPath());
        }
        
        return fileStream;
    }

    public static URL getResourceUrl(final String fileName) throws IOException
    {   
        return getClassLoader().getResource(fileName);
    }
    
    public static File getResourceFile(final String fileName) throws IOException
    {   
        final URL url = getResourceUrl(fileName);
        
        final File file = new File(url.getPath());
        
        log.debug("resource file: "+file);
        log.debug("resource file path: "+file.getPath());
        log.debug("resource file absolute path: "+file.getAbsolutePath());
        
        return file;
    }

    public static long getResourceLastModified(final String fileName) throws IOException
    {   
        final File file = getResourceFile(fileName);
        
        return file.lastModified();
    }
    
    public static String getResourceString(final String fileName) throws IOException
    {
        return read(getResourceStream(fileName));
    }
    
    public static String read(final InputStream stream) throws IOException
    {
        return read(new InputStreamReader(stream));
    }

    public static String read(final File filename) throws IOException
    {
        return read(new FileReader(filename));
    }
    
    public static String read(final Reader reader) throws IOException
    {
        int len;

        final StringBuilder contents = new StringBuilder();

        final char[] buf = new char[1024];

        while((len = reader.read(buf)) > 0)
        {
            contents.append(buf, 0, len);
        }

        return contents.toString();
    }
    
    private static final ClassLoader getClassLoader()
    {
        return FileUtil.class.getClassLoader();
    }
}
