package edu.harvard.med.countway.tools;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

public class SignatureUtil
{
    private static final Logger log = Logger.getLogger(SignatureUtil.class);
    
    private static final String certificateFormat = "X.509";
    private static final String publicKeyAlgorithm = "DSA";
    private static final String signatureAlgorithm = "SHA1withDSA";
    private static final String keystoreType = "JKS";
    private static final String keystoreFilename = "countway.keystore";
    private static final String keystorePassword = "changeit";
    
    public static final String privateKeyAlias = "countway";
    
    private static SignatureUtil instance;
    
    private final KeyStore keystore;
    private final PrivateKey privateKey;
    private final Map<String,Certificate> certMap;

    public static synchronized SignatureUtil getInstance()
    {
        if (instance == null)
        {
            instance = new SignatureUtil();
        }

        return instance;
    }

    private SignatureUtil()
    {
        try
        {
            keystore = KeyStore.getInstance(keystoreType);
            keystore.load(FileUtil.getResourceStream(keystoreFilename), keystorePassword.toCharArray()); 
            privateKey = (PrivateKey) keystore.getKey(privateKeyAlias, keystorePassword.toCharArray());
            certMap = new HashMap<String,Certificate>();
            final Enumeration<String> aliases = keystore.aliases();
            while (aliases.hasMoreElements())
            {
                String alias = aliases.nextElement();
                certMap.put(alias, keystore.getCertificate(alias));
            }
        }
        catch (KeyStoreException e)
        {
            throw new SignatureUtilException(e);   
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (CertificateException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (IOException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (UnrecoverableKeyException e)
        {
            throw new SignatureUtilException(e);
        }
    }
    
    public KeyStore getKeyStore()
    {
        return keystore;
    }
    
    public Map<String,Certificate> getCertMap()
    {
        return certMap;
    }
    
    public boolean verifySignature(String alias, String data, String signature)
    {
        try
        {
            final X509Certificate signer = (X509Certificate) certMap.get(alias);
            final Signature verifier = Signature.getInstance(signatureAlgorithm);
            verifier.initVerify(signer);
            verifier.update(data.getBytes());
            return verifier.verify(Base64Util.decodeBase64(signature.getBytes()));
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (InvalidKeyException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (SignatureException e)
        {
            throw new SignatureUtilException(e);
        }        
    }
    
    public String sign(String data)
    {
        try
        {
            final Signature signer = Signature.getInstance(signatureAlgorithm);
            signer.initSign(privateKey);
            signer.update(data.getBytes());
            return new String(Base64Util.encodeBase64(signer.sign()));
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (InvalidKeyException e)
        {
            throw new SignatureUtilException(e);
        }
        catch (SignatureException e)
        {
            throw new SignatureUtilException(e);
        }
    }
    
    @SuppressWarnings("serial")
    public static class SignatureUtilException extends RuntimeException
    {
        public SignatureUtilException(Throwable thr)
        {
            super(thr);
        }
    }
}
