/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.security.impl;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;

import jakarta.servlet.ServletRequest;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.security.AccessControl;
import net.shibboleth.shared.security.AccessControlService;

/** Simple implementation that uses an in-memory map of policies. */
public class BasicAccessControlService extends AbstractIdentifiableInitializableComponent
        implements AccessControlService {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(BasicAccessControlService.class);
    
    /** Map of named policies. */
    @Nonnull private Map<String,AccessControl> policyMap;
    
    /** Constructor. */
    public BasicAccessControlService() {
        policyMap = CollectionSupport.emptyMap();
    }
    
    /**
     * Set the policies to store.
     * 
     * @param map map of named policies
     */
    public void setPolicyMap(@Nullable final Map<String,AccessControl> map) {
        checkSetterPreconditions();
        
        if (map != null) {
            policyMap = new HashMap<>(map.size());
            
            for (final Map.Entry<String,AccessControl> entry : map.entrySet()) {
                final String trimmed = StringSupport.trimOrNull(entry.getKey());
                if (trimmed != null && entry.getValue() != null) {
                    policyMap.put(trimmed, entry.getValue());
                }
            }
        } else {
            policyMap = CollectionSupport.emptyMap();
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public AccessControl getInstance(@Nonnull final String name) {

        final AccessControl ac = policyMap.get(name);
        if (ac != null) {
            return ac;
        }
        
        log.warn("Access Control Service {}: No policy named '{}' found, returning default denial policy",
                getId(), name);
        
        return new AccessControl() {
            public boolean checkAccess(@Nonnull final ServletRequest request, @Nullable final String operation,
                    @Nullable final String resource) {
                return false;
            }
        };
    }

}