/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.scripting;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.script.Bindings;
import javax.script.Compilable;
import javax.script.CompiledScript;
import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;

import org.slf4j.Logger;

import com.google.common.io.Files;

import net.shibboleth.shared.annotation.constraint.NonnullAfterInit;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.resource.Resource;

/** This is a helper class that takes care of reading in, optionally compiling, and evaluating a script. */
public final class EvaluableScript extends AbstractInitializableComponent {

    /** The scripting language. */
    @Nonnull @NotEmpty private String scriptLanguage;

    /** The script to execute. */
    @NonnullAfterInit @NotEmpty private String script;

    /** The script engine to execute the script. */
    @NonnullAfterInit private ScriptEngine scriptEngine;

    /** The compiled form of the script, if the script engine supports compiling. */
    @Nullable private CompiledScript compiledScript;
    
    /** The log. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(EvaluableScript.class);

    /**
     * Constructor.
     */
    public EvaluableScript() {
        scriptLanguage = "javascript";
    }

    /**
     * Gets the script source.
     * 
     * @return the script source
     */
    @NonnullAfterInit @NotEmpty public String getScript() {
        return script;
    }

    /**
     * Sets the script source.
     *
     * @param what the script source
     */
    public void setScript(@Nonnull @NotEmpty final String what) {
        script = Constraint.isNotNull(StringSupport.trimOrNull(what), "Script must not be null or empty");
    }

    /**
     * Sets the script source.
     *
     * @param scriptSource how to get the script source
     * @throws IOException if there were issues reading the script
     */
    public void setScript(@Nonnull final InputStream scriptSource) throws IOException {

        Constraint.isNotNull(scriptSource, "Script source should not be null");

        script = StringSupport.inputStreamToString(
                Constraint.isNotNull(scriptSource, "Script source can not be null or empty"), null);
    }

    /**
     * Sets the script source.
     * 
     * @param scriptSource how to get the script source
     * @throws IOException if there were issues reading the script
     */
    public void setScript(@Nonnull final File scriptSource) throws IOException {

        Constraint.isNotNull(scriptSource, "Script source cannot be null");

        if (!scriptSource.exists()) {
            throw new IOException("Script source file " + scriptSource.getAbsolutePath() + " does not exist");
        }

        if (!scriptSource.canRead()) {
            throw new IOException("Script source file " + scriptSource.getAbsolutePath()
                    + " exists but is not readable");
        }

        script = Constraint.isNotNull(
                        StringSupport.trimOrNull(Files.asCharSource(scriptSource, Charset.defaultCharset()).read()),
                        "Script source cannot be empty");
    }

    /**
     * Sets the script source.
     *
     * @param scriptSource how to get the script source
     * @throws IOException if there were issues reading the script
     */
    public void setScript(@Nonnull final Resource scriptSource) throws IOException {

        Constraint.isNotNull(scriptSource, "Script source should not be null");

        try (final InputStream is = scriptSource.getInputStream()) {
            setScript(is);
        }
    }

    /**
     * Gets the script language.
     *
     * @return the script language
     */
    @Nonnull @NotEmpty public String getScriptLanguage() {
        return scriptLanguage;
    }

    /**
     * Sets the script language.
     *
     * @param what the script language
     */
    public void setEngineName(@Nonnull @NotEmpty final String what) {
        scriptLanguage = Constraint.isNotNull(StringSupport.trimOrNull(what),
                "Language must not be null or emoty");
    }

    /**
     * Evaluates this script against the given bindings.
     * 
     * @param scriptBindings the script bindings
     * 
     * @return the result of the script or null if the script did not return a result
     * 
     * @throws ScriptException thrown if there was a problem evaluating the script
     */
    @Nullable public Object eval(@Nonnull final Bindings scriptBindings) throws ScriptException {
        checkComponentActive();
        if (compiledScript != null) {
            return compiledScript.eval(scriptBindings);
        }
        return scriptEngine.eval(script, scriptBindings);
    }

    /**
     * Evaluates this script against the given context.
     * 
     * @param scriptContext the script context
     * 
     * @return the result of the script or null if the script did not return a result
     * 
     * @throws ScriptException thrown if there was a problem evaluating the script
     */
    @Nullable public Object eval(@Nonnull final ScriptContext scriptContext) throws ScriptException {
        checkComponentActive();
        if (compiledScript != null) {
            return compiledScript.eval(scriptContext);
        }
        return scriptEngine.eval(script, scriptContext);
    }

    /** {@inheritDoc}
     * Initializes the scripting engine and compiles the script, if possible.
     *
     * @throws ComponentInitializationException if the scripting engine supports 
     * compilation and the script does not compile
     */
    protected void doInitialize() throws ComponentInitializationException {

        if (script == null) {
            throw new ComponentInitializationException("Script cannot be null or empty");
        }

        final ScriptEngineManager engineManager = new ScriptEngineManager();
        scriptEngine = engineManager.getEngineByName(scriptLanguage);
        if (scriptEngine == null) {
            // fallback into our private implementations?
            log.trace("Native support for {} not found, trying shibboleth-{}", scriptLanguage, scriptLanguage);
            scriptEngine = engineManager.getEngineByName("shibboleth-" + scriptLanguage);
        }
        Constraint.isNotNull(scriptEngine, "No scripting engine associated with scripting language " + scriptLanguage);

        if (scriptEngine instanceof Compilable) {
            try {
                compiledScript = ((Compilable) scriptEngine).compile(script);
            } catch (final ScriptException e) {
                throw new ComponentInitializationException(e);
            }
        } else {
            compiledScript = null;
        }
    }
}
