package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetNewInstanceIDs;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Listgraphs;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Logout;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Online;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Query;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.UpdateInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Upload;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.WhoAmI;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.NOT_AUTHORIZED_MESSAGE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.NO_SESSION_MESSAGE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.UNAVAILABLE_MESSAGE;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.URI;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.contrib.ssl.EasySSLProtocolSocketFactory;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.User;
import org.eaglei.datatools.Workspace;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.provider.EIDataToolsProviderException;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProviderMessages;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIURI;
import org.eaglei.security.Session;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */

public final class RESTRepositoryProvider implements RepositoryProvider {

	protected static final String		READ_VIEW					= "user";
	// string sent as part of http requests
	private static final String			FORMAT_VALUE				= "application/xml";

	// format string used for all Jena calls
	private static final String			RDF_FORMAT					= "RDF/XML";
	private static final Log			log							= LogFactory.getLog(RESTRepositoryProvider.class);
	private static final boolean		isDebugEnabled				= log.isDebugEnabled();
	private static String				DEFAULT_REPOSITORY			= "";
	private static String				DEFAULT_NAMESPACE			= "";
	private static boolean				ACCEPT_SELF_SIGNED_CERTS	= false;
	private static HostConfiguration	hostConfiguration;

	/* FIXME:at the moment these two workspaces are excluded when filling the worspace list in User object*/
	private static String				WITDRAWN_WORKSPACE_URI		= "http://eagle-i.org/ont/repo/1.0/NG_Withdrawn";
	private static String				SANDBOX_WORKSPACE_URI		= "http://eagle-i.org/ont/repo/1.0/NG_Sandbox";

	enum RestCommands {
		GetNewInstanceIDs("repository/new"), WhoAmI("repository/whoami"), Logout("repository/logout/"), GetInstance("i"), UpdateInstance("repository/update"), Query("repository/sparql"), Online(""), Upload("repository/graph"), Workflow("repository/fakeworkflow"), Listgraphs("/repository/listGraphs");

		private RestCommands(final String propKey) {
			key = propKey;
		}

		public String getURL() {
			return DEFAULT_REPOSITORY + key;
		}

		private final String	key;
	}

	enum LoginFields {
		USERNAME(0), SESSION(1), USER_URI(2), ERROR_MESSAGE(3), WF_STATE(3);

		private final int	value;

		private LoginFields(final int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}
	}

	private final Map<String, HttpClient>	clientMap	= new HashMap<String, HttpClient>();

	/*
	 * Private constructor, must access via static INSTANCE field
	 */
	public RESTRepositoryProvider(final DatatoolsConfiguration config) throws IOException {
		if (config != null) {
			// TODO: at some point we need to get the auth information via
			// Sessions or other method
			DEFAULT_REPOSITORY = config.getDatatoolsRepositoryURL();
			DEFAULT_NAMESPACE = config.getDatatoolsRepositoryNamespace();
			ACCEPT_SELF_SIGNED_CERTS = config.getDatatoolsAcceptSelfSignedCerts();
		}
		if (ACCEPT_SELF_SIGNED_CERTS) {
			final Protocol easyhttps = new Protocol("https", new EasySSLProtocolSocketFactory(), 8443);
			final URI uri = new URI(DEFAULT_REPOSITORY, true);
			// use relative url only
			hostConfiguration = new HostConfiguration();
			hostConfiguration.setHost(uri.getHost(), uri.getPort(), easyhttps);
		}
	}

	private HttpClient makeHttpClient(final String username, final String password) {
		final HttpClient client = new HttpClient();
		client.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
		client.getState().setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(username, password));
		client.getParams().setAuthenticationPreemptive(true);

		return client;
	}

	private int executeMethod(final HttpClient client, final HttpMethod method) throws HttpException, IOException {
		if (ACCEPT_SELF_SIGNED_CERTS) {
			return client.executeMethod(hostConfiguration, method);
		} else {
			return client.executeMethod(method);
		}
	}

	/*
	 * private HttpClient getHttpClient(String session) { if
	 * (clientMap.get(session) == null) { return null; // or error indicating we
	 * need a new one? } return clientMap.get(session); }
	 */

	private HttpClient getHttpClient(final Session session) {
		if (!Session.isValid(session)) {
			return null;
		}
		if (clientMap.get(session.getSessionId()) == null) {
			return null; // or error indicating we need a new one?
		}

		return clientMap.get(session.getSessionId());
	}

	protected void setWriteParameters(final PostMethod method, final String id) {
		method.setParameter("uri", id);
		method.setParameter("format", FORMAT_VALUE);
		// As per conversation w Larry, repository/update doesn't need a
		// view/graph except for create
		// and removing it when unneeded avoids mismatch of views
		// will add it only in createInstance
		// method.setParameter("workspace",
		// "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace");
		method.setRequestHeader("charset", "UTF-8");
	}

	protected void setReadParameters(final PostMethod method) {
		method.setParameter("format", FORMAT_VALUE);
		method.setParameter("view", READ_VIEW);
		method.setParameter("noinferred", "true");
		method.setRequestHeader("charset", "UTF-8");
	}

	@Override
	public void createInstance(final Session session, final EIInstance instance, final EIEntity workspaceEntity) throws Exception {
		// fail early
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if (instance == null || instance.getInstanceURI() == null) {
			log.warn("Trying to create (duplicate) instance with null instance or null instance URI");
			return;
		}
		final String instanceUri = instance.getInstanceURI().toString();
		final String instanceAsString = EIInstanceFactory.INSTANCE.serialize(instance, RDF_FORMAT);
		createInstance(session, instanceAsString, instanceUri, workspaceEntity);
	}

	public void createInstance(final Session session, final String rdfString, final String instanceUri, final EIEntity workspaceEntity) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if (rdfString == null) {
			log.warn("Trying to create (duplicate) instance with null instance or null instance URI");
			return;
		}
		int status = 0;
		final PostMethod method = new PostMethod(UpdateInstance.getURL());
		setWriteParameters(method, instanceUri);
		if (workspaceEntity != null && !workspaceEntity.equals(EIEntity.NULL_ENTITY)) {
			log.info("creating instance in workspace " + workspaceEntity.getURI().toString());
			method.setParameter("workspace", workspaceEntity.getURI().toString());
		} else {
			log.info("creating instance in default workspace");
			method.setParameter("workspace", DEFAULT_WORKSPACE_ENTITY.getURI().toString());
		}

		method.setParameter("action", "create");

		if (isDebugEnabled) {
			log.debug("dump data: " + rdfString);
		}
		method.setParameter("insert", rdfString);
		if (isDebugEnabled) {
			log.debug("Trying to get create instance at " + instanceUri);
		}
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK || status == HttpStatus.SC_CREATED) {
				log.info("get instance succeded with status: " + status + " response: " + resp);
			} else {
				log.error("create instance failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("create", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	public EIInstance deepCopy(final Session session, final EIURI originalUri) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		final EIInstance original = getInstance(session, originalUri);
		if (isDebugEnabled) {
			log.debug("got original instance");
		}
		return duplicateInstance(session, original);
	}

	public EIInstance duplicateInstance(final Session session, final EIInstance original) throws Exception {
		final EIInstance duplicate = getEmptyEIInstance(session, original.getInstanceType().getURI());

		for (final EIEntity property : original.getDatatypeProperties().keySet()) {
			for (final String value : original.getDatatypeProperty(property)) {
				duplicate.addDatattypeProperty(property, value);
			}
		}

		for (final EIEntity property : original.getObjectProperties().keySet()) {
			for (final EIEntity value : original.getObjectProperty(property)) {
				duplicate.addObjectProperty(property, value);
			}
		}

		for (final EIEntity property : original.getNonOntologyLiteralProperties().keySet()) {
			for (final String value : original.getNonOntologyLiteralProperty(property)) {
				duplicate.addNonOntologyLiteralProperty(property, value);
			}
		}

		for (final EIEntity property : original.getNonOntologyResourceProperties().keySet()) {
			for (final EIEntity value : original.getNonOntologyResourceProperty(property)) {
				duplicate.addNonOntologyResourceProperty(property, value);
			}
		}

		if (isDebugEnabled) {
			log.debug("finished making new copy");
		}
		return duplicate;
	}

	// TODO there is still some code repetition between delete and update

	public void deleteInstance(final Session session, final EIURI instanceUri) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if (instanceUri == null || instanceUri.toString() == null || instanceUri.toString().length() == 0) {
			return;
		}
		final String token = getToken(session, instanceUri.toString());
		int status = 0;
		final PostMethod method = new PostMethod(UpdateInstance.getURL());
		setWriteParameters(method, instanceUri.toString());
		method.setParameter("action", "update");
		method.setParameter("token", token);
		method.setParameter("delete", createDeleteStatement(instanceUri));
		if (isDebugEnabled) {
			log.debug("Trying to get delete instance at " + instanceUri.toString());
		}
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("delete instance succeded with status: " + status + " response: " + resp);
			} else {
				log.error("delete instance failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("delete", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	public String updateInstance(final Session session, final EIInstance instance, final String token) throws Exception {
		// Fail early
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if (instance == null || instance.getInstanceURI() == null || instance.getInstanceURI().toString() == null || instance.getInstanceURI().toString().length() == 0) {
			log.warn("Trying to update instance with null instance or null instance URI");
			return null;
		}
		// clear read-only properties so we don't screw up the metadata
		// repo-side
		instance.setReadOnlyLiteralProperties(new HashMap<EIEntity, String>());
		final String instanceAsString = EIInstanceFactory.INSTANCE.serialize(instance, RDF_FORMAT);
		final String instanceUri = instance.getInstanceURI().toString();
		return updateInstance(session, instanceAsString, instanceUri, token);
	}

	public String updateInstance(final Session session, final String rdfString, final String instanceUri, final String token) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if (rdfString == null || rdfString.length() == 0) {
			log.warn("Trying to create (duplicate) instance with null instance or null instance URI");
			return null;
		}
		if (token == null) {
			return updateAndGetToken(session, rdfString, instanceUri);
		}
		int status = 0;
		final PostMethod method = new PostMethod(UpdateInstance.getURL());
		setWriteParameters(method, instanceUri);

		method.setParameter("insert", rdfString);
		method.setParameter("delete", createDeleteStatement(instanceUri));
		if (isDebugEnabled) {
			log.debug("Trying to get update instance at " + instanceUri);
		}
		method.setParameter("action", "update");
		method.setParameter("token", token);
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("update succeded with status: " + status + " response size: " + resp.length());
				return resp;
			} else if (status == HttpStatus.SC_CONFLICT) {
				log.error("Stale token. Update instance failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.STALE_TOKEN_MESSAGE);
			} else {
				log.error("update instance failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("update", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	// TODO:
	// This is a brute force approach to getting the repository to delete ALL
	// properties before inserting a new one:
	// a better future approach:
	// 1) obtain the CURRENT instance
	// 2) diff this with the "updated" instance
	// 3) only "delete" and "update" those new statements

	private String createDeleteStatement(final EIURI instanceUri) {
		return createDeleteStatement(instanceUri.toString());
	}

	private String createDeleteStatement(final String instanceUri) {
		String s = null;
		final String wildcard = "<" + instanceUri + "> <http://eagle-i.org/ont/repo/1.0/MatchAnything> <http://eagle-i.org/ont/repo/1.0/MatchAnything> .";
		log.info("Trying to create a wildcard graph to delete all the statements");
		final Model m = ModelFactory.createDefaultModel();
		final StringReader st = new StringReader(wildcard);
		m.read(st, null, "N-TRIPLE");
		final StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			m.write(sw, RDF_FORMAT);
			s = sw.toString();
			sw.flush();
		} finally {
			try {
				if (sw != null) {
					sw.close();
				}
			} catch (final IOException e) {
				log.warn("createDelete statement: " + e);
			}
		}
		// if(isDebugEnabled) log.debug("wildcard: " + s);
		return s;
	}

	protected String updateAndGetToken(final Session session, final EIInstance instance) throws Exception {
		final String instanceUri = instance.getInstanceURI().toString();
		final String instanceAsString = EIInstanceFactory.INSTANCE.serialize(instance, RDF_FORMAT);
		return updateAndGetToken(session, instanceAsString, instanceUri);
	}

	// For delete we only need the token
	protected String getToken(final Session session, final String instanceUri) throws Exception {
		return updateAndGetToken(session, null, instanceUri);
	}

	protected String updateAndGetToken(final Session session, final String rdfString, final String instanceUri) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		final PostMethod method = new PostMethod(UpdateInstance.getURL());
		setWriteParameters(method, instanceUri);
		if (rdfString != null && rdfString.length() > 0) {
			method.setParameter("insert", rdfString);
		}
		method.setParameter("action", "gettoken");
		int status = 0;
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				// check to see if we asked for token then return it
				// otherwise just return the response
				return parseResponseToken(resp);
			} else {
				log.error("get token failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get token", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	private String parseResponseToken(final String resp) {
		final ResultSet results = ResultSetFactory.fromXML(resp);
		final QuerySolution soln = results.nextSolution();
		final Resource r = soln.getResource("token");
		final String tokenid = r.getURI();
		log.info("Retrieved new token " + tokenid);
		return tokenid;
	}

	public EIInstance getInstance(final Session session, final EIURI instanceID) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if (instanceID == null || instanceID.toString() == null) {
			log.warn("Trying to get instance for null EIURI");
			return null;
		}
		int status = 0;
		final PostMethod method = new PostMethod(GetInstance.getURL());
		setReadParameters(method);
		method.setParameter("uri", instanceID.toString());
		final HttpClient client = getHttpClient(session);
		if (isDebugEnabled) {
			log.debug("Trying to get instance " + instanceID.toString());
		}
		try {
			status = executeMethod(client, method);
			if (status == HttpStatus.SC_OK) {
				if (isDebugEnabled) {
					log.debug("started with an OK status");
				}
				if (isDebugEnabled) {
					log.debug("reading in input stream as a model");
				}
				final String response = getStringFromInputStream(method.getResponseBodyAsStream());
				final EIInstance instance = EIInstanceFactory.INSTANCE.create(instanceID, response, RDF_FORMAT);
				if (isDebugEnabled) {
					log.debug("got instance contents: " + instance);
				}
				return instance;
			} else {
				log.error("get instance " + instanceID + " failed with status: " + status);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get instance", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	public List<EIURI> getNewInstanceID(final Session session, final int count) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		int status = 0;
		ResultSet results = null;
		final PostMethod method = new PostMethod(GetNewInstanceIDs.getURL());
		final List<EIURI> instances = new ArrayList<EIURI>(count);
		log.info("getting " + count + " instance IDs");
		try {
			method.setParameter("accept", "application/sparql-results+xml");
			method.setParameter("count", Integer.toString(count));
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			// We'll get this as a resource stream and convert into EIURIs
			// if(isDebugEnabled) log.debug("RESPONSE: " +
			// method.getResponseBodyAsString());
			final String response = getStringFromInputStream(method.getResponseBodyAsStream());

			if (status == HttpStatus.SC_OK) {
				results = ResultSetFactory.fromXML(response);
			} else { // TODO: check for an unathenticated session
				log.error("Error getting instance IDs; HTTP Status: " + status + " Response: " + response);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get new ID", status));
			}
		} finally {
			method.releaseConnection();
		}
		for (; results.hasNext();) {
			final QuerySolution soln = results.nextSolution();
			final Resource r = soln.getResource("new");
			final String id = r.getURI();
			instances.add(EIURI.create(id));
			log.info("Created new instance id: " + id);
		}
		if (isDebugEnabled) {
			log.debug("Created " + instances.size() + " ids");
		}
		return instances;
	}

	protected User getUserInformation(final Session session, final HttpClient client) throws Exception {
		//This method can take null session as argument, 
		//in which case it creates a new session 
		int status = 0;
		String responseBody = null;
		ResultSet results = null;
		final GetMethod method = new GetMethod(WhoAmI.getURL());

		try {
			if (client == null) {
				log.error("http Client is null");
			}
			status = executeMethod(client, method);
			responseBody = getStringFromInputStream(method.getResponseBodyAsStream());
		} catch (final Exception e) {
			log.error("problem getting user info " + WhoAmI.getURL() + " Message from repo: " + responseBody);
			throw new EIDataToolsProviderException(RepositoryProviderMessages.NOT_AUTHORIZED_MESSAGE);
		} finally {
			method.releaseConnection();
		}

		checkStatus(status);
		if (responseBody == null) {
			throw new EIDataToolsProviderException(NOT_AUTHORIZED_MESSAGE);
		}
		results = ResultSetFactory.fromXML(responseBody);

		if (results == null) {
			throw new EIDataToolsProviderException(NOT_AUTHORIZED_MESSAGE);
		}

		// we are authorized so let's return the user name
		final QuerySolution soln = results.nextSolution();
		final Literal username = soln.getLiteral("username");
		final Resource userURI = soln.getResource("uri");

		if (username == null || userURI == null) {
			throw new EIDataToolsProviderException(NOT_AUTHORIZED_MESSAGE);
		}

		log.info("Authenticated user: " + username.getString() + " " + userURI.getURI());
		String sessionId;
		if (!Session.isValid(session)) {
			sessionId = UUID.randomUUID().toString();
			clientMap.put(sessionId, client);
		} else {
			sessionId = session.getSessionId();
		}
		log.info("session ID: " + sessionId + "; new session? " + (Session.isValid(session) ? "yes" : "no"));
		Session newSession = new Session(sessionId, "", username.getString(), userURI.getURI());

		final List<EIURI> states = getWFStatesList(newSession);
		log.info("has number of wfsStates " + states.size());
		final List<Workspace> workspaceList = getWorkspaces(newSession);
		log.info("has number of workspaces " + workspaceList.size());

		final User user = new User(username.getString(), EIURI.create(userURI.getURI()), states, workspaceList, newSession);

		return user;

		// return states.toArray(new String[states.size()]);
	}

	private void checkStatus(final int status) throws EIDataToolsProviderException {
		if (status != HttpStatus.SC_OK) {
			if (status == HttpStatus.SC_NOT_FOUND) {
				log.error("Repo unavailable");
				throw new EIDataToolsProviderException(UNAVAILABLE_MESSAGE);
			}
			if (status == HttpStatus.SC_UNAUTHORIZED) {
				log.error("not authorized to get user information (login/whoami)");
				throw new EIDataToolsProviderException(NOT_AUTHORIZED_MESSAGE);
			}
			throw new EIDataToolsProviderException(RepositoryProviderMessages.NOT_AUTHORIZED_MESSAGE);
		}
	}

	@Override
	public List<Workspace> getWorkspaces(final Session session) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		final GetMethod method = new GetMethod(Listgraphs.getURL());
		final NameValuePair pair1 = new NameValuePair("format", "application/xml");
		final NameValuePair pair2 = new NameValuePair("type", "workspace");
		method.setQueryString(new NameValuePair[] { pair1, pair2 });
		try {

			final List<Workspace> workspaceList = new ArrayList<Workspace>();
			final HttpClient client = getHttpClient(session);
			log.info("executing " + Listgraphs.getURL() + " api call");
			final int status = executeMethod(client, method);
			checkStatus(status);
			final String responseBody = getStringFromInputStream(method.getResponseBodyAsStream());
			log.info(Listgraphs.getURL() + " api call has returned ,parsing the results");
			final ResultSet resultSet = ResultSetFactory.fromXML(responseBody);
			while (resultSet.hasNext()) {

				final QuerySolution solution = resultSet.next();
				final String workspaceName = solution.getLiteral(MetadataConstants.WORKSPACE_NAMED_GRAPH_LABEL).getString();
				final String workspaceURI = solution.getResource(MetadataConstants.WORKSPACE_NAMED_GRAPH_URI).getURI();
				/* FIXME:at the moment hardcode to exclude two workspaces */
				if (workspaceURI.equals(WITDRAWN_WORKSPACE_URI) || workspaceURI.equals(SANDBOX_WORKSPACE_URI)) {
					continue;
				}
				final String typeURI = solution.getResource(MetadataConstants.WORKSPACE_TYPE).getURI();
				final boolean canUserAdd = solution.getLiteral(MetadataConstants.WORKSPACE_ADD).getBoolean();
				final Workspace workspace = new Workspace(workspaceName, EIURI.create(workspaceURI), EIURI.create(typeURI), canUserAdd);
				workspaceList.add(workspace);

			}
			return workspaceList;

		} catch (final Exception e) {
			log.error(e.getMessage());
			log.error("problem getting workspaces " + Listgraphs.getURL() + " Message from repo: " + getStringFromInputStream(method.getResponseBodyAsStream()) + "; Exception " + e);
			throw new EIDataToolsProviderException(RepositoryProviderMessages.FAILED_MESSAGE);
		} finally {
			method.releaseConnection();
		}

	}

	@Override
	public User login(final String userName, final String password) throws Exception {
		final HttpClient client = makeHttpClient(userName, password);
		client.getParams().setParameter("accept", "application/sparql-results+xml");
		client.getParams().setAuthenticationPreemptive(true);

		if (isDebugEnabled) {
			log.debug("Trying to login at " + WhoAmI.getURL() + " with username " + userName);
		}
		final User user = getUserInformation(null, client);

		if (user.getUserName() == null) {
			log.error("Could not authenticate user: " + user.getUserName());
			throw new EIDataToolsProviderException(NOT_AUTHORIZED_MESSAGE);
		}

		return user;
	}

	public void logout(final Session session) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		int status = 0;
		final PostMethod method = new PostMethod(Logout.getURL());
		// now let's try to do a whoami if it succeeds then we are logged in
		// otherwise return null
		if (isDebugEnabled) {
			log.debug("Trying to logout at " + Logout.getURL());
		}
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
		} catch (final Exception e) {
			log.error("problem with logout  " + Logout.getURL() + " " + e);
			throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("logout", 0));
		} finally {
			method.releaseConnection();
		}
		// check to see if we succeeded
		if (status == HttpStatus.SC_OK) {
			if (isDebugEnabled) {
				log.debug("logout succeded");
			}
			clientMap.remove(session);
		} else if (status == 302) { // FOUND
			if (isDebugEnabled) {
				log.debug("logout semi-succeded: status 302");
			}
			clientMap.remove(session);
		} else {
			log.info("Could not logout user: HTTP Status: " + status);
			clientMap.remove(session);
			throw new EIDataToolsProviderException("Could not logout user: HTTP Status: " + status);
		}
	}

	@Override
	public User whoami(final Session session) throws Exception {
		if (!Session.isValid(session)) {
			log.info("Using null session.  Could not whoami user");
			return null;
		}

		final HttpClient client = getHttpClient(session);

		if (client == null) {
			log.info("Using stale session.  Could not whoami user");
			return null;
		}

		client.getParams().setParameter("accept", "application/sparql-results+xml");
		// now let's try to do a whoami if it succeeds then we are logged in
		// otherwise return null
		if (isDebugEnabled) {
			log.debug("Trying to whoami at " + WhoAmI.getURL());
		}

		final User sessionToResponse = getUserInformation(session, client);

		if (sessionToResponse.getUserName() == null) {
			log.info("Could not whoami user: http status:");
			return null;
		}

		return sessionToResponse;
	}

	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri, final EIEntity instanceEntity) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		return EIInstanceFactory.getInstance().createEmpty(classUri, instanceEntity);
	}

	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		final List<EIURI> ids = getNewInstanceID(session, 1);
		if (isDebugEnabled) {
			log.debug("New instance ID is: " + ids.get(0));
		}
		final EIInstance instance = EIInstanceFactory.getInstance().createEmpty(classUri, EIEntity.create(ids.get(0), ""));

		// TODO: debug
		log.info("got new instance of class " + classUri + "; null class ? " + (instance.getClass() == null ? "yes" : "no"));
		if (isDebugEnabled) {
			log.debug("New instance is: " + instance);
		}
		return instance;
	}

	@Override
	public boolean isOnline() {
		int status = 0;
		boolean online = false;
		final GetMethod method = new GetMethod(Online.getURL());
		final HttpClient client = new HttpClient();
		client.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
		if (isDebugEnabled) {
			log.debug("Trying to see if Repository is available: " + Online.getURL());
		}
		try {
			status = executeMethod(client, method);
			if (status == HttpStatus.SC_OK || status == HttpStatus.SC_UNAUTHORIZED) {
				online = true;
				if (isDebugEnabled) {
					log.debug("Repository is available: " + Online.getURL() + " is available with status: " + status);
				}
			} else {
				if (isDebugEnabled) {
					log.debug("Repository is unavailable: " + Online.getURL() + " is available with status: " + status);
				}
			}
		} catch (final Exception e) {
			log.error("problem checking online status of repository: " + Online.getURL() + " " + e);
		} finally {
			method.releaseConnection();
		}
		return online;
	}

	private synchronized String postQuery(final Session session, final String q) throws Exception {
		return postQuery(session, q, READ_VIEW);
	}

	private synchronized String postQuery(final Session session, final String q, final String view) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if (q == null) {
			log.warn("Null query!");
			return null;
		}
		int status = 0;
		final PostMethod method = new PostMethod(Query.getURL());
		if (isDebugEnabled) {
			log.debug("Trying to query: " + Query.getURL());
		}
		setReadParameters(method);
		method.setParameter("view", view);
		method.setParameter("query", q);
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				if (isDebugEnabled) {
					log.debug("query sparql endpoint succeded with status: " + status);
				}
				return resp;
			} else {
				log.error("query sparql endpoint  failed with status: " + status);
				log.error("Message from repository: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("query", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	// The return format is hardcoded to RDF/XML
	// FIXME should we allow the format to be specified?
	public String query(final Session session, final String sparql) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		return postQuery(session, sparql);
	}

	public List<EIInstanceMinimal> EIQuery(final Session session, final String sparql) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if (isDebugEnabled) {
			log.debug("Sparql: " + sparql);
		}
		final ResultSet resultSet = ResultSetFactory.fromXML(postQuery(session, sparql));
		return EIInstanceFactory.getInstance().create(resultSet);
	}

	public List<EIInstanceMinimal> getFilterQuery(final Session session, final EIURI classUri, final EIURI state, final EIURI lab) throws Exception {
		return getFilterQuery(session, classUri, state, lab, false);
	}

	public List<EIInstanceMinimal> getFilterQuery(final Session session, final EIURI classUri, final EIURI state, final EIURI lab, final boolean strictOwnerFilter) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		return EIQuery(session, SPARQLQueryUtil.getInstance().getFilterQuery(session.getUserURI(), classUri, state, lab, strictOwnerFilter));
	}

	@Override
	public List<EIInstanceMinimal> referencedByQuery(final Session session, final EIURI resourceUri, final boolean strictOwnerFilter) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		return EIQuery(session, SPARQLQueryUtil.getInstance().getReferencedByQuery(session.getUserURI(), resourceUri, strictOwnerFilter));
	}

	@Deprecated
	public void uploadInstances(final Session session, final String rdf) throws Exception {
		if (!Session.isValid(session)) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if (rdf == null) {
			log.warn("Trying to upload null rdf");
			return;
		}
		int status = 0;
		final PostMethod method = new PostMethod(Upload.getURL());
		method.setParameter("format", "text/plain");
		method.setParameter("name", "http://eagle-i.org/ont/repo/1.0/NG_Published");
		method.setParameter("type", "published");
		method.setParameter("action", "add");
		method.setParameter("content", rdf);
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("upload succeed with status: " + status + " response: " + resp);
			} else {
				log.error("upload sparql endpoint  failed with status: " + status + " response: " + resp);
			}
		} catch (final Exception e) {
			log.error("problem with upload endpoint at URL  " + Upload.getURL() + " because " + e);
		} finally {
			method.releaseConnection();
		}
	}

	public String claim(final Session session, final EIURI uri, final String claimant) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if (uri == null || claimant == null) {
			// TODO: other error handling
			log.warn("null claim parameter, returning null: " + (uri == null ? "uri" : claimant == null ? "claimant" : "huh?"));
			return null;
		}
		int status = 0;
		final PostMethod method = new PostMethod(RestCommands.Workflow.getURL());
		method.setParameter("uri", uri.toString());
		method.setParameter("format", FORMAT_VALUE);
		method.setParameter("claim", claimant);
		log.info("Trying to claim " + uri + " for " + claimant);
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("claim succeded with status: " + status + " response: " + resp);
				return resp; // Maybe response code only? Do we return it?
			} else {
				log.error("claim failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("claim", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	public String promote(final Session session, final EIURI uri, final EIURI newState) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if (uri == null) {
			log.warn("Trying to promote a null URI");
			return null;
		}

		int status = 0;
		final PostMethod method = new PostMethod(RestCommands.Workflow.getURL());
		method.setParameter("uri", uri.toString());
		method.setParameter("format", FORMAT_VALUE);
		method.setParameter("promote", newState.toString());
		// if(isDebugEnabled) log.debug("dump data: " +
		// EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
		if (isDebugEnabled) {
			log.debug("Trying to get promote " + uri + " to state " + newState);
		}
		try {
			final HttpClient client = getHttpClient(session);
			status = executeMethod(client, method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("promote succeded with status: " + status + " response: " + resp);
				return newState.toString();
			} else {
				log.error("promote failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("promote", status));
			}
		} finally {
			method.releaseConnection();
		}
	}

	public List<EIURI> getWFStatesList(final Session session) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		final String q = SPARQLQueryUtil.getInstance().getWFStatesQuery(session.getUserURI());
		if (isDebugEnabled) {
			log.debug("get wf states query: " + q);
		}
		final String result = postQuery(session, q, "all");
		final ResultSet resultSet = ResultSetFactory.fromXML(result);
		final List<EIURI> states = new ArrayList<EIURI>();
		while (resultSet.hasNext()) {
			final QuerySolution qs = resultSet.next();
			// state is the variable name in the query obtained from
			// SPARQLUtil
			if (qs.contains(STATE_VARIABLE)) {
				states.add(EIURI.create(qs.getResource(STATE_VARIABLE).getURI()));
			}
		}

		return states;

	}

	public String[] getWFStates(final Session session) throws Exception {
		final List<EIURI> states = getWFStatesList(session);
		return states.toArray(new String[states.size()]);
	}

	public Map<EIEntity, String> retrieveLabels(final Session session, final List<EIEntity> entities) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		// TODO: generate a UNION sparql query instead: this may be slow

		log.info("retrieving labels");

		final Map<EIEntity, String> results = new HashMap<EIEntity, String>(entities.size());
		for (final EIEntity entity : entities) {
			try {
				results.put(entity, retrieveLabel(session, entity.getURI()));
			} catch (final EIDataToolsProviderException e) {
				final int statusLoc = e.getMessage().indexOf("4");
				if (statusLoc < 0) {
					throw e;
				}

				final String statusString = e.getMessage().substring(statusLoc);
				if (!statusString.equals("400")) {
					throw e;
				}

				log.warn("could not retrieve label for " + entity.getURI() + "; skipping");
			}
		}

		final String q = SPARQLQueryUtil.getInstance().getRetrieveLabelsQuery(entities);
		if (isDebugEnabled) {
			log.debug("retrieve labels query: " + q);
		}
		final String labelsResult = postQuery(session, q, "all");
		if (isDebugEnabled) {
			log.debug("result " + labelsResult);
		}

		/*
		 * ResultSet resultSet = ResultSetFactory.fromXML(labelsResult); for
		 * (EIEntity entity : entityList) { results.put(entity, ); }
		 */

		return results;
	}

	public Map<EIURI, String> retrieveUriLabels(final Session session, final List<EIURI> uris) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		// TODO: generate a UNION sparql query instead: this may be slow

		log.info("retrieving labels");

		final Map<EIURI, String> results = new HashMap<EIURI, String>(uris.size());
		for (final EIURI uri : uris) {
			results.put(uri, retrieveLabel(session, uri));
		}

		final String q = SPARQLQueryUtil.getInstance().getRetrieveUriLabelsQuery(uris);
		if (isDebugEnabled) {
			log.debug("retrieve labels query: " + q);
		}
		final String labelsResult = postQuery(session, q, "all");
		if (isDebugEnabled) {
			log.debug("result " + labelsResult);
		}

		return results;

	}

	public String retrieveLabel(final Session session, final EIURI uri) throws Exception {
		if (!Session.isValid(session)) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		final String q = SPARQLQueryUtil.getInstance().getRetrieveLabelQuery(uri);
		if (isDebugEnabled) {
			log.debug("retrieve label query: " + q);
		}
		final String result = postQuery(session, q, "all");
		if (isDebugEnabled) {
			log.debug("result " + result);
		}
		final ResultSet resultSet = ResultSetFactory.fromXML(result);
		final List<String> labels = new ArrayList<String>();
		while (resultSet.hasNext()) {
			final QuerySolution qs = resultSet.next();
			// state is the variable name in the query obtained from
			// SPARQLUtil
			if (qs.contains(LABEL_VARIABLE)) {
				labels.add(qs.getLiteral(LABEL_VARIABLE).getString());
			}
		}
		// FIXME remove this once we're sure it works
		log.info("labels for " + uri.toString() + ": "); // TODO: back to
		// debug
		for (final String label : labels) {
			log.info(label);
		}
		return labels.isEmpty() ? "" : labels.get(0);

	}

	private String getStringFromInputStream(final InputStream in) throws IOException {
		if (in == null) {
			return null;
		}
		final StringWriter writer = new StringWriter();
		// encoding needs to be explicitly set
		IOUtils.copy(in, writer, "UTF-8");
		return writer.toString();
	}

}
