package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetNewInstanceIDs;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.ListTransitions;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Listgraphs;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Logout;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Online;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Query;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.UpdateInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Upload;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.WhoAmI;
import static org.eaglei.datatools.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.NOT_AUTHORIZED_MESSAGE;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.lang.CharEncoding;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.User;
import org.eaglei.datatools.WorkFlowTransition;
import org.eaglei.datatools.Workspace;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.provider.EIDataToolsProviderException;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.eaglei.model.jena.MetadataConstants;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.security.Session;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */

public final class RESTRepositoryProvider implements RepositoryProvider {

	protected static final String READ_VIEW = "user";
	// string sent as part of http requests
	private static final String FORMAT_VALUE = "application/xml";

	// format string used for all Jena calls
	private static final String RDF_FORMAT = "RDF/XML";
	private static final Log log = LogFactory.getLog( RESTRepositoryProvider.class );
	private static final boolean isDebugEnabled = log.isDebugEnabled();
	private static String DEFAULT_REPOSITORY = "";

	public static final String LABEL_QUERY_FAILED = "could not retrieve label";

	// Workflow Variable Names
	private static final String WORKFLOW_TRANSITION_SUBJECT = "subject";
	private static final String WORKFLOW_TRANSITION_LABEL = "label";
	private static final String WORKFLOW_TRANSITION_FROM = "initial";
	private static final String WORKFLOW_TRANSITION_FROM_LABEL = "initialLabel";
	private static final String WORKFLOW_TRANSITION_TO = "final";
	private static final String WORKFLOW_TRANSITION_TO_LABEL = "finalLabel";
	private static final String WORKFLOW_TRANSITION_DESCRIPTION = "description";
	private static final String WORKFLOW_TRANSITION_WORKSPACE = "workspace";
	private static final String WORKFLOW_TRANSITION_WORKSPACELABEL = "workspaceLabel";
	private static final String WORKFLOW_TRANSITION_ALLOWED = "allowed";
	/*
	 * FIXME:at the moment these two workspaces are excluded when filling the worspace list in User object
	 */
	private static String WITDRAWN_WORKSPACE_URI = "http://eagle-i.org/ont/repo/1.0/NG_Withdrawn";
	private static String SANDBOX_WORKSPACE_URI = "http://eagle-i.org/ont/repo/1.0/NG_Sandbox";

	enum RestCommands {

		GetNewInstanceIDs("repository/new"), WhoAmI("repository/whoami"), Logout("repository/logout/"), GetInstance("i"), UpdateInstance("repository/update"), Query("repository/sparql"), Online(""), Upload("repository/graph"), FakeWorkflow(
				"repository/fakeworkflow"), Listgraphs("repository/listGraphs"), Claim("repository/workflow/claim"), Release("repository/workflow/release"), Transition("repository/workflow/push"), ListTransitions("repository/workflow/transitions"), ListResources(
				"repository/workflow/resources");

		private RestCommands(final String propKey) {
			key = propKey;
		}

		public String getURL() {
			return DEFAULT_REPOSITORY + key;
		}

		private final String key;
	}

	enum LoginFields {
		USERNAME(0), SESSION(1), USER_URI(2), ERROR_MESSAGE(3), WF_STATE(3);

		private final int value;

		private LoginFields(final int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}
	}

	private final Map<String, HttpClient> clientMap = new HashMap<String, HttpClient>();

	private final Map<String, Map<EIURI, EIEntity>> repoUsersMap = new HashMap<String, Map<EIURI, EIEntity>>();

	private final JenaEIInstanceFactory instanceFactory;

	public RESTRepositoryProvider(final DatatoolsConfiguration config, final JenaEIInstanceFactory instanceFactory) throws IOException {
		if ( config != null ) {
			DEFAULT_REPOSITORY = config.getDatatoolsRepositoryURL();
		}
		this.instanceFactory = instanceFactory;
	}

	private HttpClient createHttpClient(final String username, final String password) {
		final HttpClient client = new HttpClient();
		client.setHttpConnectionManager( new MultiThreadedHttpConnectionManager() );
		client.getState().setCredentials( AuthScope.ANY, new UsernamePasswordCredentials( username, password ) );
		client.getParams().setAuthenticationPreemptive( true );
		return client;
	}

	private HttpClient getHttpClient(final Session session) {
		if ( !Session.isValid( session ) ) {
			return null;
		}
		if ( clientMap.get( session.getSessionId() ) == null ) {
			return null; // or error indicating we need a new one?
		}

		return clientMap.get( session.getSessionId() );
	}

	protected void setWriteParameters(final PostMethod method, final String id) {
		method.setParameter( "uri", id );
		method.setParameter( "format", FORMAT_VALUE );
		// As per conversation w Larry, repository/update doesn't need a
		// view/graph except for create
		// and removing it when unneeded avoids mismatch of views
		// will add it only in createInstance
		// method.setParameter("workspace",
		// "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace");
		method.setRequestHeader( "charset", "UTF-8" );
	}

	protected void setReadParameters(final PostMethod method) {
		method.setParameter( "format", FORMAT_VALUE );
		method.setParameter( "view", READ_VIEW );
		method.setParameter( "noinferred", "true" );
		method.setRequestHeader( "charset", "UTF-8" );
	}

	@Override
	public void createInstance(final Session session, final EIInstance instance, final EIEntity workspaceEntity) throws Exception {
		log.info( "the instance label is   " + instance.getInstanceLabel().toString() );
		ProviderUtils.validateSession( session );
		if ( instance == null || instance.getInstanceURI() == null ) {
			log.warn( "Trying to create (duplicate) instance with null instance or null instance URI" );
			return;
		}
		final String instanceUri = instance.getInstanceURI().toString();
		final String instanceAsString = instanceFactory.serialize( instance, RDF_FORMAT );
		createInstance( session, instanceAsString, instanceUri, workspaceEntity );
	}
	
	/* -------------------making this request multipart ----------------- */
	public void createInstance(final Session session, final String rdfString, final String instanceUri, final EIEntity workspaceEntity) throws Exception {

		ProviderUtils.validateSession( session );
		if ( rdfString == null ) {
			log.warn( "Trying to create (duplicate) instance with null instance or null instance URI" );
			return;
		}
		final PostMethod method = new PostMethod( UpdateInstance.getURL() );
		Part idPart = new StringPart( "uri", instanceUri, "utf-8" );

		Part workspacePart = null;
		if ( workspaceEntity != null && !workspaceEntity.equals( EIEntity.NULL_ENTITY ) ) {
			log.info( "creating instance in workspace " + workspaceEntity.getURI().toString() );
			/* multipart workspace request */
			workspacePart = new StringPart( "workspace", workspaceEntity.getURI().toString(), "UTF-8" );
		} else {
			log.info( "creating instance in default workspace" );
			workspacePart = new StringPart( "workspace", DEFAULT_WORKSPACE_ENTITY.getURI().toString(), "utf-8" );
		}
		/* multipart action request */
		if ( isDebugEnabled ) {
			log.debug( "creating the action part" );
		}
		Part actionPart = new StringPart( "action", "create" );
		/* multipart insert request */
		Part insertPart = new XmlStringPart( "insert", rdfString, CharEncoding.UTF_8 );
		if ( isDebugEnabled ) {
			log.debug( "the transfer encoding of insertpart  is " + insertPart.getTransferEncoding() );
			log.debug( "the charset of insertpart  is " + insertPart.getCharSet() );
			log.debug( "the content type of insertpart  is " + insertPart.getContentType() );
		}

		log.info( "making parts array from the created parts to send it a multipart request" );
		Part[] parts = { workspacePart, actionPart, insertPart, idPart };
		method.setRequestEntity( new MultipartRequestEntity( parts, method.getParams() ) );
		if ( isDebugEnabled ) {
			log.debug( "Trying to create instance at " + instanceUri );
		}

		ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	private String encodeToUTF8(final String rdfString) throws UnsupportedEncodingException {
		return new String( rdfString.getBytes( "UTF-8" ) );
	}

	public EIInstance deepCopy(final Session session, final EIURI originalUri) throws Exception {
		ProviderUtils.validateSession( session );

		final EIInstance original = getInstance( session, originalUri );
		if ( isDebugEnabled ) {
			log.debug( "got original instance" );
		}
		return duplicateInstance( session, original );
	}

	public EIInstance duplicateInstance(final Session session, final EIInstance original) throws Exception {
		final EIInstance duplicate = getEmptyEIInstance( session, original.getInstanceType().getURI() );

		for (final EIEntity property : original.getDatatypeProperties().keySet()) {
			for (final String value : original.getDatatypeProperty( property )) {
				duplicate.addDatattypeProperty( property, value );
			}
		}

		for (final EIEntity property : original.getObjectProperties().keySet()) {
			for (final EIEntity value : original.getObjectProperty( property )) {
				duplicate.addObjectProperty( property, value );
			}
		}

		for (final EIEntity property : original.getNonOntologyLiteralProperties().keySet()) {
			for (final String value : original.getNonOntologyLiteralProperty( property )) {
				duplicate.addNonOntologyLiteralProperty( property, value );
			}
		}

		for (final EIEntity property : original.getNonOntologyResourceProperties().keySet()) {
			for (final EIEntity value : original.getNonOntologyResourceProperty( property )) {
				duplicate.addNonOntologyResourceProperty( property, value );
			}
		}

		if ( isDebugEnabled ) {
			log.debug( "finished making new copy" );
		}
		return duplicate;
	}

	public void deleteInstance(final Session session, final EIURI instanceUri) throws Exception {
		ProviderUtils.validateSession( session );
		if ( instanceUri == null || instanceUri.toString() == null || instanceUri.toString().length() == 0 ) {
			return;
		}
		final String token = getToken( session, instanceUri.toString() );
		final PostMethod method = new PostMethod( UpdateInstance.getURL() );
		setWriteParameters( method, instanceUri.toString() );
		method.setParameter( "action", "update" );
		method.setParameter( "token", token );
		method.setParameter( "delete", createDeleteStatement( instanceUri ) );
		if ( isDebugEnabled ) {
			log.debug( "Trying to delete instance at " + instanceUri.toString() );
		}
		ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	public String updateInstance(final Session session, final EIInstance instance, final String token) throws Exception {
		ProviderUtils.validateSession( session );
		if ( instance == null || instance.getInstanceURI() == null || instance.getInstanceURI().toString() == null || instance.getInstanceURI().toString().length() == 0 ) {
			log.warn( "Trying to update instance with null instance or null instance URI" );
			return null;
		}
		// clear read-only properties so we don't screw up the metadata
		// repo-side
		instance.setReadOnlyLiteralProperties( new HashMap<EIEntity, String>() );
		final String instanceAsString = instanceFactory.serialize( instance, RDF_FORMAT );
		final String instanceUri = instance.getInstanceURI().toString();
		return updateInstance( session, instanceAsString, instanceUri, token );
	}

	// FIXME revise the interface - what do we do with this result?
	/* -------------------making this request multipart ----------------- */
	public String updateInstance(final Session session, final String rdfString, final String instanceUri, final String token) throws Exception {
		ProviderUtils.validateSession( session );
		Part idPart = new StringPart( "uri", instanceUri, CharEncoding.UTF_8 );
		if ( rdfString == null || rdfString.length() == 0 ) {
			log.warn( "Trying to create (duplicate) instance with null instance or null instance URI" );
			return null;
		}
		if ( token == null ) {
			return updateAndGetToken( session, rdfString, instanceUri );
		}
		final PostMethod method = new PostMethod( UpdateInstance.getURL() );

		/* multipart action request */
		Part actionPart = new StringPart( "action", "update" );

		/* multipart insert request */
		Part insertPart = new XmlStringPart( "insert", rdfString, CharEncoding.UTF_8 );

		/* multipart delete request */		
		Part deletePart = new XmlStringPart( "delete", createDeleteStatement( instanceUri ), CharEncoding.UTF_8 );

		/* multipart token request */
		Part tokenPart = new StringPart( "token", token );
		
		if ( isDebugEnabled ) {
			log.debug( "Trying to update instance at " + instanceUri );
		}
		log.info( "making parts array from the created parts to send it a multipart request" );
		Part[] parts = { idPart, actionPart, insertPart, deletePart, tokenPart };
		method.setRequestEntity( new MultipartRequestEntity( parts, method.getParams() ) );

		return ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	private String createDeleteStatement(final EIURI instanceUri) {
		return createDeleteStatement( instanceUri.toString() );
	}

	private String createDeleteStatement(final String instanceUri) {
		String s = null;
		final String wildcard = "<" + instanceUri + "> <http://eagle-i.org/ont/repo/1.0/MatchAnything> <http://eagle-i.org/ont/repo/1.0/MatchAnything> .";
		log.info( "Trying to create a wildcard graph to delete all the statements" );
		final Model m = ModelFactory.createDefaultModel();
		final StringReader st = new StringReader( wildcard );
		m.read( st, null, "N-TRIPLE" );
		final StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			m.write( sw, RDF_FORMAT );
			s = sw.toString();
			sw.flush();
		} finally {
			try {
				if ( sw != null ) {
					sw.close();
				}
			} catch (final IOException e) {
				log.warn( "createDelete statement: " + e );
			}
		}
		return s;
	}

	// For delete we only need the token
	private String getToken(final Session session, final String instanceUri) throws Exception {
		return updateAndGetToken( session, null, instanceUri );
	}

	private String updateAndGetToken(final Session session, final String rdfString, final String instanceUri) throws Exception {

		List<Part> partsList = new ArrayList<Part>();
		ProviderUtils.validateSession( session );

		final PostMethod method = new PostMethod( UpdateInstance.getURL() );
		Part idPart = new StringPart( "uri", instanceUri, CharEncoding.UTF_8 );
		partsList.add( idPart );

		Part actionPart = new StringPart( "action", "gettoken", CharEncoding.UTF_8 );
		partsList.add( actionPart );

		if ( rdfString != null && rdfString.length() > 0 ) {
			Part insertPart = new XmlStringPart( "insert", rdfString, CharEncoding.UTF_8 );
			partsList.add( insertPart );

		}
		method.setRequestEntity( new MultipartRequestEntity( partsList.toArray( new Part[partsList.size()] ), method.getParams() ) );
		final String resp = ProviderUtils.getHttpResponse( getHttpClient( session ), method );
		return parseResponseToken( resp );
	}

	private String parseResponseToken(final String resp) {
		final ResultSet results = ResultSetFactory.fromXML( resp );
		final QuerySolution soln = results.nextSolution();
		final Resource r = soln.getResource( "token" );
		final String tokenid = r.getURI();
		log.info( "Retrieved new token " + tokenid );
		return tokenid;
	}

	public EIInstance getInstance(final Session session, final EIURI instanceID) throws Exception {
		ProviderUtils.validateSession( session );

		if ( instanceID == null || instanceID.toString() == null ) {
			log.warn( "Trying to get instance for null EIURI" );
			return null;
		}
		final PostMethod method = new PostMethod( GetInstance.getURL() );
		setReadParameters( method );
		method.setParameter( "uri", instanceID.toString() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to get instance " + instanceID.toString() );
		}

		final String responseBody = ProviderUtils.getHttpResponse( getHttpClient( session ), method );
		return instanceFactory.create( instanceID, responseBody, RDF_FORMAT );
	}

	public List<EIURI> getNewInstanceID(final Session session, final int count) throws Exception {
		ProviderUtils.validateSession( session );

		final PostMethod method = new PostMethod( GetNewInstanceIDs.getURL() );
		method.setParameter( "accept", "application/sparql-results+xml" );
		method.setParameter( "count", Integer.toString( count ) );
		log.info( "getting " + count + " instance IDs" );
		final String responseBody = ProviderUtils.getHttpResponse( getHttpClient( session ), method );
		final ResultSet results = ResultSetFactory.fromXML( responseBody );
		final List<EIURI> instances = new ArrayList<EIURI>( count );
		while ( results.hasNext() ) {
			final QuerySolution soln = results.nextSolution();
			final Resource r = soln.getResource( "new" );
			final String id = r.getURI();
			instances.add( EIURI.create( id ) );
			log.info( "Created new instance id: " + id );
		}
		if ( isDebugEnabled ) {
			log.debug( "Created " + instances.size() + " ids" );
		}
		return instances;
	}

	private User getUserInformation(final Session session, final HttpClient client) throws Exception {
		// This method can take null session as argument,
		// in which case it creates a new session
		if ( client == null ) {
			log.error( "http Client is null" );
			return null;
		}
		final GetMethod method = new GetMethod( WhoAmI.getURL() );

		final String responseBody = ProviderUtils.getHttpResponse( client, method );
		if ( responseBody == null ) {
			throw new EIDataToolsProviderException( NOT_AUTHORIZED_MESSAGE );
		}

		final ResultSet results = ResultSetFactory.fromXML( responseBody );

		if ( results == null ) {
			throw new EIDataToolsProviderException( NOT_AUTHORIZED_MESSAGE );
		}

		final QuerySolution soln = results.nextSolution();
		final Literal username = soln.getLiteral( "username" );
		final Resource userURI = soln.getResource( "uri" );

		if ( username == null || userURI == null ) {
			throw new EIDataToolsProviderException( NOT_AUTHORIZED_MESSAGE );
		}

		log.info( "Authenticated user: " + username.getString() + " " + userURI.getURI() );
		String sessionId;
		if ( !Session.isValid( session ) ) {
			sessionId = UUID.randomUUID().toString();
			clientMap.put( sessionId, client );
		} else {
			sessionId = session.getSessionId();
		}
		log.info( "session ID: " + sessionId + "; new session? " + ( Session.isValid( session ) ? "yes" : "no" ) );
		final Session newSession = new Session( sessionId, "", username.getString(), userURI.getURI() );

		List<WorkFlowTransition> states = new ArrayList<WorkFlowTransition>();
		try {

			states = listWorkFlowTransitions( newSession, null );
			log.info( "has number of wfsStates " + states.size() );
		} catch (final Exception e) {
			log.warn( "failed to get fake workflow states" );
		}
		// FIXME see if we can skip this (we're not using workspaces and it's adding up time)

		// final List<Workspace> workspaceList = getWorkspaces( newSession );
		// log.info( "has number of workspaces " + workspaceList.size() );

		final List<Workspace> workspaceList = Collections.emptyList();
		final User user = new User( username.getString(), EIURI.create( userURI.getURI() ), states, workspaceList, newSession );

		return user;
	}

	@Override
	public List<Workspace> getWorkspaces(final Session session) throws Exception {
		ProviderUtils.validateSession( session );

		final PostMethod method = new PostMethod( Listgraphs.getURL() );
		method.setParameter( "format", FORMAT_VALUE );
		// FIXME we need to figure out how to restrict this
		// method.setParameter( "type", "workspace" );
		log.info( "executing " + Listgraphs.getURL() + " api call" );

		final String responseBody = ProviderUtils.getHttpResponse( getHttpClient( session ), method );
		log.info( Listgraphs.getURL() + " api call has returned ,parsing the results" );
		final List<Workspace> workspaceList = new ArrayList<Workspace>();
		final ResultSet resultSet = ResultSetFactory.fromXML( responseBody );
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			final String workspaceName = solution.getLiteral( MetadataConstants.WORKSPACE_NAMED_GRAPH_LABEL ).getString();
			final String workspaceURI = solution.getResource( MetadataConstants.WORKSPACE_NAMED_GRAPH_URI ).getURI();
			// FIXME:at the moment hard code to exclude two workspaces
			if ( workspaceURI.equals( WITDRAWN_WORKSPACE_URI ) || workspaceURI.equals( SANDBOX_WORKSPACE_URI ) ) {
				continue;
			}
			final String typeURI = solution.getResource( MetadataConstants.WORKSPACE_TYPE ).getURI();
			final boolean canUserAdd = solution.getLiteral( MetadataConstants.WORKSPACE_ADD ).getBoolean();
			final boolean canUserDelete = solution.getLiteral( MetadataConstants.WORKSPACE_REMOVE ).getBoolean();
			final Workspace workspace = new Workspace( workspaceName, EIURI.create( workspaceURI ), EIURI.create( typeURI ), canUserAdd, canUserDelete );
			workspaceList.add( workspace );

		}
		return workspaceList;
	}

	@Override
	public User login(final String userName, final String password) throws Exception {
		final HttpClient client = createHttpClient( userName, password );
		client.getParams().setParameter( "accept", "application/sparql-results+xml" );

		if ( isDebugEnabled ) {
			log.debug( "Trying to login at " + WhoAmI.getURL() + " with username " + userName );
		}

		final User user = getUserInformation( null, client );

		if ( user == null || user.getUserURI() == null ) {
			log.error( "Could not authenticate user: " + userName );
			throw new EIDataToolsProviderException( NOT_AUTHORIZED_MESSAGE );
		}
		return user;
	}

	@Override
	public void logout(final Session session) throws Exception {
		ProviderUtils.validateSession( session );
		final PostMethod method = new PostMethod( Logout.getURL() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to logout at " + Logout.getURL() );
		}
		try {
			ProviderUtils.getHttpResponse( getHttpClient( session ), method );
			clientMap.remove( session.getSessionId() );
			repoUsersMap.remove( session.getSessionId() );
		} catch (final Exception e) {
			clientMap.remove( session.getSessionId() );
			repoUsersMap.remove( session.getSessionId() );
			throw e;
		}
	}

	@Override
	public User whoami(final Session session) throws Exception {
		if ( !Session.isValid( session ) ) {
			log.info( "Using null session.  Could not whoami user" );
			return null;
		}

		final HttpClient client = getHttpClient( session );

		if ( client == null ) {
			log.info( "Using stale session.  Could not whoami user" );
			return null;
		}

		client.getParams().setParameter( "accept", "application/sparql-results+xml" );
		// now let's try to do a whoami if it succeeds then we are logged in
		// otherwise return null
		if ( isDebugEnabled ) {
			log.debug( "Trying to whoami at " + WhoAmI.getURL() );
		}

		final User sessionToResponse = getUserInformation( session, client );

		if ( sessionToResponse.getUserName() == null ) {
			log.info( "Could not whoami user: http status:" );
			return null;
		}

		return sessionToResponse;
	}

	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri, final EIEntity instanceEntity) throws Exception {
		ProviderUtils.validateSession( session );
		return instanceFactory.createEmpty( classUri, instanceEntity );
	}

	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri) throws Exception {
		ProviderUtils.validateSession( session );

		final List<EIURI> ids = getNewInstanceID( session, 1 );
		if ( isDebugEnabled ) {
			log.debug( "New instance ID is: " + ids.get( 0 ) );
		}
		final EIInstance instance = instanceFactory.createEmpty( classUri, EIEntity.create( ids.get( 0 ), "" ) );

		if ( isDebugEnabled ) {
			log.debug( "got new instance of class " + classUri + "; null class ? " + ( instance.getClass() == null ? "yes" : "no" ) );
			log.debug( "New instance is: " + instance );

		}
		return instance;
	}

	@Override
	public boolean isOnline() {
		int status = 0;
		boolean online = false;
		final GetMethod method = new GetMethod( Online.getURL() );
		final HttpClient client = new HttpClient();
		client.setHttpConnectionManager( new MultiThreadedHttpConnectionManager() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to see if Repository is available: " + Online.getURL() );
		}
		try {
			status = client.executeMethod( method );
			if ( status == HttpStatus.SC_OK || status == HttpStatus.SC_UNAUTHORIZED ) {
				online = true;
				if ( isDebugEnabled ) {
					log.debug( "Repository is available: " + Online.getURL() + " is available with status: " + status );
				}
			} else {
				if ( isDebugEnabled ) {
					log.debug( "Repository is unavailable: " + Online.getURL() + " is available with status: " + status );
				}
			}
		} catch (final Exception e) {
			log.error( "problem checking online status of repository: " + Online.getURL() + " " + e );
		} finally {
			method.releaseConnection();
		}
		return online;
	}

	private synchronized String postQuery(final Session session, final String q) throws Exception {
		return postQuery( session, q, READ_VIEW );
	}

	private synchronized String postQuery(final Session session, final String q, final String view) throws Exception {
		ProviderUtils.validateSession( session );

		if ( q == null ) {
			log.warn( "Null query!" );
			return null;
		}
		final PostMethod method = new PostMethod( Query.getURL() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to query: " + Query.getURL() );
		}
		setReadParameters( method );
		method.setParameter( "view", view );
		method.setParameter( "query", encodeToUTF8( q ) );
		return ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	// The return format is hardcoded to RDF/XML
	// FIXME should we allow the format to be specified?
	public String query(final Session session, final String sparql) throws Exception {
		ProviderUtils.validateSession( session );

		return postQuery( session, sparql );
	}

	public List<EIInstanceMinimal> EIQuery(final Session session, final String sparql) throws Exception {
		ProviderUtils.validateSession( session );
		if ( isDebugEnabled ) {
			log.debug( "Sparql: " + sparql );
		}
		final ResultSet resultSet = ResultSetFactory.fromXML( postQuery( session, sparql ) );
		return EIInstanceMinimalFactory.getInstance().create( resultSet );
	}

	@Deprecated
	public List<EIInstanceMinimal> getFilterQuery(final Session session, final EIURI classUri, final EIURI state, final EIURI lab) throws Exception {
		return getFilterQuery( session, classUri, state, lab, false );
	}

	@Deprecated
	public List<EIInstanceMinimal> getFilterQuery(final Session session, final EIURI classUri, final EIURI state, final EIURI lab, final boolean strictOwnerFilter) throws Exception {
		ProviderUtils.validateSession( session );
		return EIQuery( session, SPARQLQueryUtil.getInstance().getFilterQuery( session.getUserURI(), classUri, state, lab, strictOwnerFilter ) );
	}

	@Override
	@Deprecated
	public List<EIInstanceMinimal> referencedByQuery(final Session session, final EIURI resourceUri, final boolean strictOwnerFilter) throws Exception {
		ProviderUtils.validateSession( session );
		return EIQuery( session, SPARQLQueryUtil.getInstance().getReferencedByQuery( session.getUserURI(), resourceUri, strictOwnerFilter ) );
	}

	@Deprecated
	public void uploadInstances(final Session session, final String rdf) throws Exception {
		ProviderUtils.validateSession( session );

		if ( rdf == null ) {
			log.warn( "Trying to upload null rdf" );
			return;
		}
		final PostMethod method = new PostMethod( Upload.getURL() );
		method.setParameter( "format", "text/plain" );
		method.setParameter( "name", "http://eagle-i.org/ont/repo/1.0/NG_Published" );
		method.setParameter( "type", "published" );
		method.setParameter( "action", "add" );
		method.setParameter( "content", encodeToUTF8( rdf ) );
		ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	@Deprecated
	public String fakeWorkflowClaim(final Session session, final EIURI uri, final String claimant) throws Exception {
		ProviderUtils.validateSession( session );
		if ( uri == null || claimant == null ) {
			// TODO: other error handling
			log.warn( "null claim parameter, returning null: " + ( uri == null ? "uri" : claimant == null ? "claimant" : "huh?" ) );
			return null;
		}
		final PostMethod method = new PostMethod( RestCommands.FakeWorkflow.getURL() );
		method.setParameter( "uri", uri.toString() );
		method.setParameter( "format", FORMAT_VALUE );
		method.setParameter( "claim", claimant );
		log.info( "Trying to claim " + uri + " for " + claimant );
		return ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	@Deprecated
	public String fakeWorkflowPromote(final Session session, final EIURI uri, final EIURI newState) throws Exception {
		ProviderUtils.validateSession( session );

		if ( uri == null ) {
			log.warn( "Trying to promote a null URI" );
			return null;
		}

		final PostMethod method = new PostMethod( RestCommands.FakeWorkflow.getURL() );
		method.setParameter( "uri", uri.toString() );
		method.setParameter( "format", FORMAT_VALUE );
		method.setParameter( "promote", newState.toString() );

		if ( isDebugEnabled ) {
			log.debug( "Trying to promote " + uri + " to state " + newState );
		}
		return ProviderUtils.getHttpResponse( getHttpClient( session ), method );
	}

	@Deprecated
	public List<EIURI> fakeWorkflowGetWFStatesList(final Session session) throws Exception {
		ProviderUtils.validateSession( session );

		final String q = SPARQLQueryUtil.getInstance().getWFStatesQuery( session.getUserURI() );
		if ( isDebugEnabled ) {
			log.debug( "get wf states query: " + q );
		}
		final String result = postQuery( session, q, "all" );
		final ResultSet resultSet = ResultSetFactory.fromXML( result );
		final List<EIURI> states = new ArrayList<EIURI>();
		while ( resultSet.hasNext() ) {
			final QuerySolution qs = resultSet.next();
			// state is the variable name in the query obtained from
			// SPARQLUtil
			if ( qs.contains( STATE_VARIABLE ) ) {
				states.add( EIURI.create( qs.getResource( STATE_VARIABLE ).getURI() ) );
			}
		}

		return states;

	}

	@Deprecated
	public String[] fakeWorkflowGetWFStates(final Session session) throws Exception {
		final List<EIURI> states = fakeWorkflowGetWFStatesList( session );
		return states.toArray( new String[states.size()] );
	}

	public String retrieveLabel(final Session session, final EIURI uri) throws Exception {
		ProviderUtils.validateSession( session );
		final String q = SPARQLQueryUtil.getInstance().getRetrieveLabelQuery( uri );
		if ( isDebugEnabled ) {
			log.debug( "retrieve label query: " + q );
		}
		final String result = postQuery( session, q, READ_VIEW );
		if ( isDebugEnabled ) {
			log.debug( "result " + result );
		}
		final ResultSet resultSet = ResultSetFactory.fromXML( result );
		final List<String> labels = new ArrayList<String>();
		while ( resultSet.hasNext() ) {
			final QuerySolution qs = resultSet.next();
			if ( qs.contains( LABEL_VARIABLE ) ) {
				labels.add( qs.getLiteral( LABEL_VARIABLE ).getString() );
			}
		}
		return labels.isEmpty() ? "" : labels.get( 0 );

	}

	// @Override
	public List<EIInstanceMinimal> listResources(final Session session, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter) throws Exception {
		ProviderUtils.validateSession( session );
		final PostMethod method = new PostMethod( RestCommands.ListResources.getURL() );
		// State restriction
		EIURI state = queryRequest.getWFState();

		if ( state != null && !state.equals( EIURI.NULL_EIURI ) ) {
			method.setParameter( "state", state.toString() );
		} else {
			method.setParameter( "state", "all" );
		}
		final StringBuilder sparql = new StringBuilder();
		// Type restriction
		EIURI classUri = queryRequest.getType();
		if ( classUri != null && !classUri.equals( EIURI.NULL_EIURI ) ) {
			method.setParameter( "type", classUri.toString() );
		} else { // restrict to eagle-i resources
			sparql.append( SPARQLQueryUtil.getInstance().allTypesPattern( true ) );
		}
		// Lab restriction
		EIURI lab = queryRequest.getLab();
		if ( lab != null && !lab.equals( EIURI.NULL_EIURI ) ) {
			sparql.append( SPARQLQueryUtil.getInstance().labRestrictionPattern( lab ) );
		} else {
			sparql.append( SPARQLQueryUtil.getInstance().labRestrictionPattern( EIURI.NULL_EIURI ) );
		}

		commonListResourcesMethodSetup( EIURI.create( session.getUserURI() ), queryRequest, orderBy, strictOwnerFilter, sparql, method );

		final ResultSet resultSet = ResultSetFactory.fromXML( ProviderUtils.getHttpResponse( getHttpClient( session ), method ) );
		List<EIInstanceMinimal> results = EIInstanceMinimalFactory.getInstance().create( resultSet );
		if ( isDebugEnabled ) {
			log.debug( "number of resources retrieved: " + results.size() );
		}
		return results;
	}

	@Override
	public List<EIInstanceMinimal> listReferencingResources(final Session session, final EIURI resourceUri, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter) throws Exception {
		ProviderUtils.validateSession( session );
		final PostMethod method = new PostMethod( RestCommands.ListResources.getURL() );
		method.setParameter( "state", "all" );
		final StringBuilder sparql = new StringBuilder();
		// Restrict to eagle-i resources
		sparql.append( SPARQLQueryUtil.getInstance().allTypesPattern( false ) );
		// Include lab info
		sparql.append( SPARQLQueryUtil.getInstance().labRestrictionPattern( EIURI.NULL_EIURI ) );
		// Restrict to referenced resources
		sparql.append( SPARQLQueryUtil.getInstance().referencedByPattern( resourceUri ) );

		commonListResourcesMethodSetup( EIURI.create( session.getUserURI() ), queryRequest, orderBy, strictOwnerFilter, sparql, method );

		final ResultSet resultSet = ResultSetFactory.fromXML( ProviderUtils.getHttpResponse( getHttpClient( session ), method ) );
		List<EIInstanceMinimal> results = EIInstanceMinimalFactory.getInstance().create( resultSet );
		return results;
	}

	private void commonListResourcesMethodSetup(final EIURI user, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter, final StringBuilder sparql, final PostMethod method)
			throws UnsupportedEncodingException {
		method.setParameter( "detail", "full" );
		method.setParameter( "format", FORMAT_VALUE );

		// TODO see at the UI level if other combinations are needed (e.g. owner=self + unclaimed=true for all resources available to user)
		if ( strictOwnerFilter ) {
			// sparql.append( SPARQLQueryUtil.getInstance().ownerRestrictionPattern( user, true ) );
			method.setParameter( "owner", "self" );
			method.setParameter( "unclaimed", "false" );
		} else {
			method.setParameter( "owner", "all" );
			method.setParameter( "unclaimed", "true" );
		}

		sparql.append( SPARQLQueryUtil.getInstance().modifiedDatePattern() );
		// include additional orderBy variables only if necessary
		if ( orderBy == SortByProperties.status || orderBy == SortByProperties.type ) {
			sparql.append( SPARQLQueryUtil.getInstance().additionalLabelsPattern() );
		}

		// FIXME this is a fix for the erroneously inserted dcterms:created triples
		// should be removed once the data is cleaned up
		sparql.append( " filter(!regex(str(?" ).append( CREATION_DATE_VARIABLE ).append( "), \"http://www.w3.org/2001/XMLSchema#dateTime\"))" );
		// FIXME fix for resources with no labels (needs cleanup)
		sparql.append( " filter(bound(?" ).append( LABEL_VARIABLE ).append( "))" );

		if ( isDebugEnabled ) {
			log.debug( "Sparql pattern sent to workflow/resources is: " + sparql.toString() );

		}

		method.setParameter( "addPattern", encodeToUTF8( sparql.toString() ) );
		method.setParameter( "addResults", "?" + LAB_VARIABLE + " ?" + LAB_NAME_VARIABLE + " ?" + MODIFIED_DATE_VARIABLE );
		StringBuilder modifiers = new StringBuilder();
		// modifiers.append(" ORDER BY ?").append( orderBy.getVariable()).append(" " );
		modifiers.append( " ORDER BY <http://eagle-i.org/ont/repo/1.0/upperCaseStr>(?" ).append( orderBy.getVariable() ).append( ") " );

		// FIXME verify that there are no fence post errors
		if ( queryRequest.isPaginated() && queryRequest.getMaxResults() > 0 && queryRequest.getStartIndex() >= 0 ) {
			modifiers.append( "LIMIT " ).append( String.valueOf( queryRequest.getMaxResults() ) ).append( " OFFSET " ).append( String.valueOf( queryRequest.getStartIndex() ) );
		}
		method.setParameter( "addModifiers", modifiers.toString() );

		if ( isDebugEnabled ) {
			log.debug( "modifiers sent to workflow/resources are: " + modifiers.toString() );

		}
	}

	@Override
	public EIInstance getInstanceWithWorkflowState(final Session session, final EIURI instanceUri) throws Exception {

		return null;
	}

	@Override
	public List<WorkFlowTransition> listWorkFlowTransitions(final Session session, final EIEntity workspaceEntity) throws Exception {

		ProviderUtils.validateSession( session );
		final PostMethod method = new PostMethod( ListTransitions.getURL() );
		method.setParameter( "format", FORMAT_VALUE );
		/*
		 * if not NULL ENTITY or not null then set the URI parameter otherwise don't set which defaults to list from all workspaces
		 */

		if ( workspaceEntity != null && !workspaceEntity.equals( EIEntity.NULL_ENTITY ) ) {
			method.setParameter( "URI", workspaceEntity.getURI().toString() );

		}

		log.info( "executing " + ListTransitions.getURL() + " api call" );
		String responseBody = ProviderUtils.getHttpResponse( getHttpClient( session ), method );
		// if reponseBody equals null throw Exception
		if ( responseBody == null ) {
			throw new EIDataToolsProviderException( NOT_AUTHORIZED_MESSAGE );
		}
		log.info( "parsing response to make list of workflow transitions" );
		final ResultSet resultSet = ResultSetFactory.fromXML( responseBody );
		final List<WorkFlowTransition> transitionList = new ArrayList<WorkFlowTransition>();
		while ( resultSet.hasNext() ) {
			final QuerySolution solution = resultSet.next();
			// Only process solution if the contained transition is allowed
			if ( solution.contains( WORKFLOW_TRANSITION_ALLOWED ) ) {
				final boolean allowed = solution.getLiteral( WORKFLOW_TRANSITION_ALLOWED ).getBoolean();
				if ( allowed ) {
					final EIEntity transitionEntity = getEntityFromSolution( solution, WORKFLOW_TRANSITION_SUBJECT, WORKFLOW_TRANSITION_LABEL );
					final EIEntity fromStateEntity = getEntityFromSolution( solution, WORKFLOW_TRANSITION_FROM, WORKFLOW_TRANSITION_FROM_LABEL );
					final EIEntity toStateEntity = getEntityFromSolution( solution, WORKFLOW_TRANSITION_TO, WORKFLOW_TRANSITION_TO_LABEL );
					final WorkFlowTransition workFlowTransition = new WorkFlowTransition( transitionEntity, fromStateEntity, toStateEntity, allowed );
					transitionList.add( workFlowTransition );
				}
			}
		}
		log.info( transitionList.size() + " transitions were returned" );
		return transitionList;
	}

	private EIEntity getEntityFromSolution(final QuerySolution solution, final String uriVariable, final String labelVariable) {
		if ( solution.contains( uriVariable ) ) {
			final EIURI uri = EIURI.create( solution.getResource( uriVariable ).getURI() );
			String label = "<none>";
			if ( solution.contains( labelVariable ) ) {
				label = solution.getLiteral( labelVariable ).getString();
			}
			return EIEntity.create( uri, label );
		} else {
			return EIEntity.NULL_ENTITY;
		}
	}

	public Map<EIURI, String> getModifiedDates(Session session, List<EIURI> uris) throws Exception {
		Map<EIURI, String> uriToModifiedDate = new HashMap<EIURI, String>();

		for (EIURI uri : uris) {
			String date = getModifiedDate( session, uri );
			if ( isDebugEnabled ) {
				log.debug( date );
			}
			uriToModifiedDate.put( uri, date );
		}

		return uriToModifiedDate;
	}

	private String getModifiedDate(Session session, EIURI uri) throws Exception {
		String query = SPARQLQueryUtil.getInstance().getModifiedDateQuery( uri );

		final String result = postQuery( session, query, READ_VIEW );

		final ResultSet resultSet = ResultSetFactory.fromXML( result );
		String modifiedDate = "";
		while ( resultSet.hasNext() ) {
			final QuerySolution qs = resultSet.next();
			if ( qs.contains( MODIFIED_DATE_VARIABLE ) ) {
				modifiedDate = qs.getLiteral( MODIFIED_DATE_VARIABLE ).getString();
			}
		}
		return modifiedDate;
	}

	@Override
	public List<EIURI> claim(final Session session, final List<EIURI> uris) throws Exception {
		final List<EIURI> successes = new ArrayList<EIURI>();
		ProviderUtils.validateSession( session );
		// String currentModifiedDate = getModifiedDate(session, uri)

		final PostMethod method = new PostMethod( RestCommands.Claim.getURL() );

		for (final EIURI uri : uris) {
			method.setParameter( "uri", uri.toString() );
			try {
				ProviderUtils.getHttpResponse( getHttpClient( session ), method );
				successes.add( uri );
			} catch (final EIDataToolsProviderException e) {
				log.warn( "could not claim " + uri );
				// failures.add(uri);
			}
		}
		return successes;
	}

	@Override
	public List<EIURI> release(final Session session, final List<EIURI> uris) throws Exception {
		log.info( "releasing " + uris.size() + " uris" );
		final List<EIURI> successes = new ArrayList<EIURI>();
		ProviderUtils.validateSession( session );
		final PostMethod method = new PostMethod( RestCommands.Release.getURL() );

		for (final EIURI uri : uris) {
			method.setParameter( "uri", uri.toString() );
			try {
				ProviderUtils.getHttpResponse( getHttpClient( session ), method );
				successes.add( uri );
			} catch (final EIDataToolsProviderException e) {
				log.warn( "could not release " + uri );
				// failures.add(uri);
			}
		}
		return successes;
	}

	@Override
	public List<EIURI> transition(final Session session, final List<EIURI> uris, final EIEntity transitionEntity) throws Exception {
		final List<EIURI> successes = new ArrayList<EIURI>();
		ProviderUtils.validateSession( session );

		for (final EIURI uri : uris) {
			final PostMethod method = new PostMethod( RestCommands.Transition.getURL() );
			method.setParameter( "uri", uri.toString() );
			method.setParameter( "transition", transitionEntity.getURI().toString() );
			try {
				ProviderUtils.getHttpResponse( getHttpClient( session ), method );
				log.info( "transitioned " + uri + " using " + transitionEntity.getLabel() ); // TODO: remove
				successes.add( uri );
			} catch (final EIDataToolsProviderException e) {
				log.warn( "could not transition " + uri );
				// failures.add(uri);
			}
		}

		return successes;
	}

}
