package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.LAB_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_NAME_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.TYPE_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.IS_STUB_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.resultSetVariables;
import static org.eaglei.datatools.model.DataToolsOntConstants.DC_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_DT_PREFIX;
import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_DT_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_REPO_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.FOAF_PERSON;
import static org.eaglei.datatools.model.DataToolsOntConstants.OBI_ORGANIZATION;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDFS_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDF_URI;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.model.EIURI;

/**
 * 
 * @author Daniela Bourges
 * 
 *         queries and query patterns useful to build queries - getQuery methods return a syntactically correct query string - getPattern or getRestriction methods return portions of a query that can be concatenated. To obtain a full query prepend
 *         with an openingClause (e.g. selectClause) and finish with a closingClause
 * 
 *         Keep deprecated queries for reference
 * 
 */

public class SPARQLQueryUtil {

	private static final Log logger = LogFactory.getLog( SPARQLQueryUtil.class );

	private static SPARQLQueryUtil instance;

	private SPARQLQueryUtil() {

	}

	public static SPARQLQueryUtil getInstance() {
		if ( instance == null ) {
			instance = new SPARQLQueryUtil();
		}
		return instance;
	}

	/**
	 * Returns a sparql query that asks the URI of a resource of type rootURI with a label
	 * 
	 * @param rootURI
	 * @param label
	 * @return the sparql query string
	 */

	public String getEIResourcesByLabelQuery(final EIURI rootURI, final String label) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( selectClause( "resource" ) );
		sparql.append( "?resource a <" ).append( rootURI.toString() ).append( "> . " );
		sparql.append( "?resource <http://www.w3.org/2000/01/rdf-schema#label> ?o " );
		// Ignore case
		sparql.append( "FILTER regex(str(?o), \"" + label + "\", \"i\").}" );
		return sparql.toString();

		// The query should be (substitute type and label but keep <> and "" )
		// SELECT ?resource WHERE { ?resource a <type> . ?resource <http://www.w3.org/2000/01/rdf-schema#label> ?o
		// FILTER regex(?o, "label", "i").}
	}

	public String getRetrieveLabelQuery(final EIURI uri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( selectClause( LABEL_VARIABLE ) );
		sparql.append( "<" ).append( uri.toString() ).append( "> <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( LABEL_VARIABLE );
		sparql.append( "}" );
		return sparql.toString();
	}
	

	@Deprecated
	public String getReferencedByQuery(final String user, final EIURI resourceUri, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( namespaces() );
		sparql.append( selectClause( resultSetVariables ) );
		sparql.append( commonPattern() );
		sparql.append( typePattern() );
		sparql.append( referencedByPattern( resourceUri ) );
		sparql.append( typeRestrictionPattern( EIURI.NULL_EIURI, false ) );
		sparql.append( stateRestrictionPattern( EIURI.NULL_EIURI ) );
		sparql.append( labRestrictionPattern( EIURI.NULL_EIURI ) );
		sparql.append( ownerRestrictionPattern( EIURI.create( user ), strictOwnerFilter ) );
		sparql.append( orderByLabelClosingClause() );
		return sparql.toString();
	}

	public String referencedByPattern(final EIURI resourceUri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " ?anyPredicate <" ).append( resourceUri.toString() ).append( "> . " );
		return sparql.toString();
	}

	public String getResourcesForObjectPropertyValuesQuery(final EIURI classUri, final EIURI lab, final EIURI state, final boolean onlyLab) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( selectClause( Arrays.asList( SUBJECT_VARIABLE, LABEL_VARIABLE, TYPE_VARIABLE, TYPE_LABEL_VARIABLE, LAB_VARIABLE, LAB_NAME_VARIABLE ) ) );
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( LABEL_VARIABLE ).append( " . " );
		sparql.append( typePattern() );
		sparql.append( typeRestrictionPattern( classUri, false ) );
		if ( onlyLab ) {
			sparql.append( labRestrictionPattern( lab ) );
		} else {
			sparql.append( labRestrictionPattern( EIURI.NULL_EIURI ) );
		}
		sparql.append( stateRestrictionPattern( state ) );
		sparql.append( orderByLabelClosingClause() );
		return sparql.toString();
	}

	@Deprecated
	public String getFilterQuery(final String user, final EIURI classUri, final EIURI state, final EIURI lab, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( namespaces() );
		sparql.append( selectClause( resultSetVariables ) );
		sparql.append( commonPattern() );
		sparql.append( typePattern() );
		sparql.append( typeRestrictionPattern( classUri, true ) );
		sparql.append( stateRestrictionPattern( state ) );
		sparql.append( labRestrictionPattern( lab ) );
		sparql.append( ownerRestrictionPattern( EIURI.create( user ), strictOwnerFilter ) );
		sparql.append( orderByLabelClosingClause() );
		return sparql.toString();

	}

	@Deprecated
	private String namespaces() {
		final StringBuilder namespaces = new StringBuilder();
		namespaces.append( "PREFIX dcterms: <" ).append( DC_URI ).append( "> " );
		namespaces.append( "PREFIX rdf: <" ).append( RDF_URI ).append( "> " );
		namespaces.append( "PREFIX rdfs: <" ).append( RDFS_URI ).append( "> " );
		namespaces.append( "PREFIX repo: <" ).append( EAGLE_I_REPO_URI ).append( "> " );
		return namespaces.toString();
	}

	public String selectClause(final List<String> resultVariables) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "SELECT DISTINCT " );
		for (final String variable : resultVariables) {
			sparql.append( "?" ).append( variable ).append( " " );
		}
		sparql.append( "WHERE { " );
		return sparql.toString();
	}

	public String selectClause(final String resultVariable) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "SELECT DISTINCT " ).append( "?" ).append( resultVariable ).append( " " ).append( "WHERE { " );
		return sparql.toString();
	}

	@Deprecated
	private String commonPattern() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( LABEL_VARIABLE ).append( " . " );
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " dcterms:created ?" ).append( CREATION_DATE_VARIABLE ).append( " . " );
		return sparql.toString();
	}

	private String typePattern() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "graph ?g{?" ).append( SUBJECT_VARIABLE ).append( " a ?" ).append( TYPE_VARIABLE ).append( "}" );
		sparql.append( "filter(?g != <" ).append( EAGLE_I_REPO_URI ).append( "NG_Inferred>) . " );
		return sparql.toString();
	}

	public String allTypesPattern(final boolean excludeSomeTypes) {
		// Need to use a variable different than ?t,
		// because that gets only the triples that are not inferred types
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " a ?rt . " );
		sparql.append( "?rt <http://eagle-i.org/ont/app/1.0/inClassGroup> <http://eagle-i.org/ont/app/1.0/ClassGroup/resourceRoot> " );
		// FIXME manually exclude Organizations and Persons as resources
		// should be done by annotations in ontology
		if ( excludeSomeTypes ) {
			sparql.append( "filter(?rt != <" ).append( OBI_ORGANIZATION ).append( "> && ?rt != <" ).append( FOAF_PERSON ).append( ">) ." );
		}
		return sparql.toString();
	}

	public String includeOwnerLabelPattern() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "OPTIONAL{?" ).append( SUBJECT_VARIABLE ).append( " <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?" ).append( OWNER_VARIABLE ).append( " . " );
		sparql.append( "?" ).append( OWNER_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( OWNER_NAME_VARIABLE ).append( "} . " );
		return sparql.toString();
	}

	public String getAllPossibleOwnersQuery() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( selectClause( Arrays.asList( OWNER_VARIABLE, OWNER_NAME_VARIABLE ) ) );
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?" ).append( OWNER_VARIABLE ).append( " . " );
		sparql.append( "?" ).append( OWNER_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( OWNER_NAME_VARIABLE ).append( "}" );
		return sparql.toString();
	}

	public String ownerRestrictionPattern(final EIURI user, final boolean strictOwnerFilter) {
		final StringBuilder sparql = new StringBuilder();
		if ( strictOwnerFilter ) {
			sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?" ).append( OWNER_VARIABLE ).append( " . " );
			sparql.append( "FILTER(?" ).append( OWNER_VARIABLE ).append( "= <" );
		} else {
			sparql.append( "OPTIONAL {?" ).append( SUBJECT_VARIABLE ).append( " <http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner> ?" ).append( OWNER_VARIABLE ).append( "} . " );
			sparql.append( "FILTER(!bound(?" ).append( OWNER_VARIABLE ).append( ") || ?" ).append( OWNER_VARIABLE ).append( " = <" );
		}
		sparql.append( user.toString() );
		sparql.append( ">) . " );
		return sparql.toString();
	}

	private String typeRestrictionPattern(final EIURI classUri, final boolean excludeSomeTypes) {
		final StringBuilder sparql = new StringBuilder();
		if ( !isNull( classUri ) ) {
			sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " a <" ).append( classUri.toString() ).append( "> ." );
		} else {
			sparql.append( allTypesPattern( excludeSomeTypes ) );
		}
		return sparql.toString();
	}

	private String stateRestrictionPattern(final EIURI state) {
		final StringBuilder sparql = new StringBuilder();
		if ( !isNull( state ) ) {
			sparql.append( "OPTIONAL {?" ).append( SUBJECT_VARIABLE ).append( " <" ).append( EAGLE_I_REPO_URI ).append( "hasWorkflowState> <" ).append( state.toString() ).append( "> . " );
			sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " <" ).append( EAGLE_I_REPO_URI ).append( "hasWorkflowState> ?" ).append( STATE_VARIABLE );
			sparql.append( "} . " );
		} else {
			sparql.append( "OPTIONAL {?" ).append( SUBJECT_VARIABLE ).append( " <" ).append( EAGLE_I_REPO_URI ).append( "hasWorkflowState> ?" ).append( STATE_VARIABLE );
			sparql.append( "} . " );
		}
		return sparql.toString();
	}

	// FIXME (dbw) for lab != null, do we need to return resources where lab is unbound?
	public String labRestrictionPattern(final EIURI lab) {
		final StringBuilder sparql = new StringBuilder();
		if ( !isNull( lab ) ) {
			sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " ?labProperty ?" ).append( LAB_VARIABLE ).append( " . " );
			sparql.append( "?" ).append( LAB_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( LAB_NAME_VARIABLE ).append( " . " );
			sparql.append( "?labProperty <http://eagle-i.org/ont/app/1.0/inPropertyGroup> <http://eagle-i.org/ont/app/1.0/PropertyGroup/relatedLab> . " );
			sparql.append( "filter(?" ).append( LAB_VARIABLE ).append( " = <" ).append( lab.toString() ).append( ">) . " );
		} else {
			sparql.append( "OPTIONAL {?" ).append( SUBJECT_VARIABLE ).append( " ?labProperty ?" ).append( LAB_VARIABLE ).append( " . " );
			sparql.append( "?labProperty <http://eagle-i.org/ont/app/1.0/inPropertyGroup> <http://eagle-i.org/ont/app/1.0/PropertyGroup/relatedLab> . " );
			sparql.append( "?" ).append( LAB_VARIABLE ).append( " a <" ).append( DataToolsOntConstants.EI_LAB ).append( "> . " );
			sparql.append( "?" ).append( LAB_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( LAB_NAME_VARIABLE ).append( "} . " );
		}
		return sparql.toString();
	}

	public String getResourcesETLedFromFileQuery(final String fileName) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "PREFIX " ).append( EAGLE_I_DT_PREFIX ).append( ": <" ).append( EAGLE_I_DT_URI ).append( "> " );
		sparql.append( selectClause( "resource" ) );
		sparql.append( "?resource  " ).append( EAGLE_I_DT_PREFIX ).append( ":source_file  \"" ).append( fileName ).append( "\".}" );
		return sparql.toString();
	}

	public String orderByLabelClosingClause() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "} order by <" ).append( EAGLE_I_REPO_URI ).append( "upperCaseStr>(?" ).append( LABEL_VARIABLE ).append( ")" );
		return sparql.toString();
	}

	private boolean isNull(final EIURI uri) {
		return uri == null || uri.equals( EIURI.NULL_EIURI ) || "".equals( uri.toString() );
	}

	public String modifiedDatePattern() {
		return " OPTIONAL {?" + SPARQLConstants.SUBJECT_VARIABLE + " <http://purl.org/dc/terms/modified> ?" + MODIFIED_DATE_VARIABLE + "} .";
	}

	public String getModifiedDateQuery(final EIURI uri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( selectClause( MODIFIED_DATE_VARIABLE ) );
		sparql.append( "<" ).append( uri ).append( ">  <http://purl.org/dc/terms/modified> ?" ).append( MODIFIED_DATE_VARIABLE );
		sparql.append( " }" );
		return sparql.toString();
	}

	public String additionalLabelsPattern() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( "OPTIONAL {?" ).append( STATE_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( STATE_LABEL_VARIABLE ).append( "} . " );
		sparql.append( "OPTIONAL {?" ).append( TYPE_VARIABLE ).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( TYPE_LABEL_VARIABLE ).append( "} . " );
		return sparql.toString();
	}

	public String getStubPattern(boolean isOptional) {
		final StringBuilder sparql = new StringBuilder();
		
		sparql.append( "OPTIONAL {?" ).append( SUBJECT_VARIABLE ).append( " <" ).append( DataToolsOntConstants.IS_STUB_URI ).append( "> " );
		sparql.append( "?" + IS_STUB_VARIABLE);
		sparql.append( " } . " );
		
		if (!isOptional) {
			sparql.append( " {?" ).append( SUBJECT_VARIABLE ).append( " <" ).append( DataToolsOntConstants.IS_STUB_URI ).append( "> " );
			sparql.append( "'True'} . " );
		}
		return sparql.toString();
	}
	
	public String getRetrieveStubsQuery(final EIURI mainUri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( selectClause( Arrays.asList(SUBJECT_VARIABLE, LABEL_VARIABLE )));
		sparql.append( "<" ).append( mainUri.toString() ).append( "> ?anyPredicate ?" ).append( SUBJECT_VARIABLE ).append(" . ");
		sparql.append("?").append(SUBJECT_VARIABLE).append(" <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( LABEL_VARIABLE ).append(" . ");
		sparql.append( "?" ).append( SUBJECT_VARIABLE ).append( " <" ).append( DataToolsOntConstants.IS_STUB_URI ).append( "> " ).append( "'True' . " );
		sparql.append( "}" );
		return sparql.toString();
	}
	//select ?r_subject ?r_label where {<mainUri> ?somePredicate ?r_subject. ?r_subject <is_stub_uri> "True" . ?r_subject rdfs:label ?r_label}

}
