/**
 * 
 */
package org.eaglei.datatools.jena;

import static org.eaglei.model.jena.SPARQLConstants.*;
import static org.eaglei.model.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.OBJECT_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.TYPE_VARIABLE;

import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import java.util.Arrays;
import java.util.List;

import org.eaglei.model.EIURI;

/**
 * @author "Sravan Cheriyala" Started Mar 17, 2011
 */
public class BulkCurationQueryBuilder implements QueryBuilder {

	private static BulkCurationQueryBuilder queryBuilder;

	public static QueryBuilder getInstance() {
		if ( queryBuilder == null ) {
			queryBuilder = new BulkCurationQueryBuilder();
			return queryBuilder;
		} else {
			return queryBuilder;
		}

	}

	private BulkCurationQueryBuilder() {

	}

	@Override
	public String diagnoseQuery(EIURI typeURI, EIURI predicateURI, String objectString, RDFobjectType objectType, String graph) {
		final StringBuilder sparql = selectClauseWithPrefix();
		//NOTE Type matching outside of graph command
		if ( !isAllOption( typeURI ) ) {
			sparql.append( subjectTypeClause( typeURI ) );
		} 
		sparql.append( " graph ?g{" );
		sparql.append(typeAndLabelTriplePatterns( ));
		sparql.append( tripleForWhereCondition( predicateURI, objectString, objectType ) );
		sparql.append( "}  filter(?g = <").append(graph).append(">) . ");
		
		//Fetch also ontology labels (for display purposes)
		//In the future we can get from ontModel
		//!! In this query order matters (these triples first stall the repository)
		if(isAllOption( typeURI )) {
			sparql.append(typeLabelTriple());
		}
		if(isAllOption( predicateURI )) {
			sparql.append(predicateLabelTriple());
		}
		if(isAllOptionObject( objectString ) ) {
			sparql.append(objectLabelTriple());
		}
		sparql.append("}");
		return sparql.toString();
	}

	@Override
	public String diagnoseQueryByRegex(EIURI typeURI, EIURI predicateURI, String regex, String graph) {
		final StringBuilder sparql = selectClauseWithPrefix();
		//NOTE Type matching outside of graph command
		if ( !isAllOption( typeURI ) ) {
			sparql.append( subjectTypeClause( typeURI ) );
		} 
		sparql.append( " graph ?g{" );
		sparql.append( typeAndLabelTriplePatterns( ));
		sparql.append( tripleForWhereConditionWhenRegex( predicateURI, regex ) );
		sparql.append( "}  filter(?g = <").append(graph).append(">) . ");
		
		//Fetch also ontology labels (for display purposes)
		//In the future we can get from ontModel
		//!! In this query order matters (these triples first stall the repository)
		if(isAllOption( typeURI)) {
			sparql.append( typeLabelTriple() );
		}
		if(isAllOption( predicateURI )) {
			sparql.append(predicateLabelTriple());
		}
		sparql.append("}");
		return sparql.toString();
	}
	
	@Override
	public String diagnoseQueryByRegexForMetadata(EIURI predicateURI, String regex, String graph) {
		final StringBuilder sparql = selectClauseWithPrefix();
		//Note that for this query we need type and label outside of the graph command
		//because they are not defined in the metadata graph
		sparql.append( " graph ?gt{" );
		sparql.append( " ?").append(SUBJECT_VARIABLE).append( " a ?").append(TYPE_VARIABLE).append(". ");
		sparql.append( "}  filter(?gt != <").append(DatatoolsMetadataConstants.Inferred).append(">) . ");
		sparql.append( " ?").append(SUBJECT_VARIABLE).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?").append(LABEL_VARIABLE).append(". ");
		sparql.append( " graph ?g{" );
		sparql.append( tripleForWhereConditionWhenRegex( predicateURI, regex ) );
		sparql.append( "}  filter(?g = <").append(graph).append(">) . ");
		sparql.append( typeLabelTriple() );
		if(isAllOption( predicateURI )) {
			sparql.append(predicateLabelTriple());
		}
		sparql.append("}");
		return sparql.toString();
	}

	@Override
	public String getQueryToDeleteTriplesByRegexForMetadata(EIURI predicateURI, String regex, String graph) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( constructClause( predicateURI ));
		sparql.append( whereString() );
		sparql.append( graphClause( predicateURI, regex, graph ) );
		sparql.append( "}" );
		return sparql.toString();
	}

	private String predicateLabelTriple() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( " optional{?").append(ANY_PREDICATE_VARIABLE).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?").append(ANY_PREDICATE_LABEL_VARIABLE).append("}. ");
		return sparql.toString();
	}
	
	private String objectLabelTriple() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( " optional{?").append(OBJECT_VARIABLE).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?").append(OBJECT_LABEL_VARIABLE).append("}. ");
		return sparql.toString();
	}
	
	private String typeLabelTriple() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( " optional{?").append(TYPE_VARIABLE).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?" ).append( TYPE_LABEL_VARIABLE ).append("}. ");
		return sparql.toString();
	}
	

	private String typeAndLabelTriplePatterns() {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( " ?").append(SUBJECT_VARIABLE).append( " a ?").append(TYPE_VARIABLE).append(". ");
		sparql.append( " ?").append(SUBJECT_VARIABLE).append( " <http://www.w3.org/2000/01/rdf-schema#label> ?").append(LABEL_VARIABLE).append(". ");
		return sparql.toString();
	}
	
	@Override
	public String getQueryToAddObject(EIURI typeURI, EIURI predicateURI, String oldObjectString, RDFobjectType oldObjectType, String newObjectString, RDFobjectType newObjectType, String graph) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( constructClause( predicateURI, newObjectString, newObjectType ) );
		sparql.append( whereString() );
		if ( !isAllOption( typeURI ) ) {
			sparql.append( subjectTypeClause( typeURI ) );
		}
		sparql.append( graphClause( predicateURI, oldObjectString, graph, oldObjectType ) );
		sparql.append( "}" );
		return sparql.toString();
	}

	private String whereString() {
		return "  where {";
	}

	@Override
	public String getQueryToAddPredicate(EIURI typeURI, EIURI oldPredicateURI, EIURI newPredicateURI, String objectString, RDFobjectType objectType, String graph) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( constructClause( newPredicateURI, objectString, objectType ) );
		sparql.append( whereString() );
		if ( !isAllOption( typeURI ) ) {	
			sparql.append( subjectTypeClause( typeURI ) );
		}
		sparql.append( graphClause( oldPredicateURI, objectString, graph, objectType ) );
		sparql.append( "}" );
		return sparql.toString();
	}

	@Override
	public String getQueryToDeleteTriple(EIURI typeURI, EIURI predicateURI, String objectString, RDFobjectType objectType, String graph) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( constructClause( predicateURI, objectString, objectType ) );
		sparql.append( whereString() );
		if ( !isAllOption( typeURI ) ) {
			sparql.append( subjectTypeClause( typeURI ) );
		}
		sparql.append( graphClause( predicateURI, objectString, graph, objectType ) );
		sparql.append( "}" );
		return sparql.toString();
	}

	@Override
	public String getQueryToAddPredicateAndObject(EIURI typeURI, EIURI oldPredicateURI, EIURI newPredicateURI, String oldObjectString, RDFobjectType oldObjectType, String newObjectString, RDFobjectType newObjectType, String graph) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( constructClause( newPredicateURI, newObjectString, newObjectType ) );
		sparql.append( whereString() );
		if ( !isAllOption( typeURI ) ) {
			sparql.append( subjectTypeClause( typeURI ) );
		}
		sparql.append( graphClause( oldPredicateURI, oldObjectString, graph, oldObjectType ) );
		sparql.append( "}" );
		return sparql.toString();
	}

	@Override
	public String getConstructQueryByRegex(EIURI typeURI, EIURI predicateURI, String regex, String graph) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append( constructClause( predicateURI ));
		sparql.append( whereString() );
		if ( !isAllOption( typeURI ) ) {
			sparql.append( subjectTypeClause( typeURI ) );
		}
		sparql.append( graphClause( predicateURI, regex, graph ) );
		sparql.append( "}" );
		return sparql.toString();
	}

	private String constructClause(EIURI predicate, String object, RDFobjectType objectType) {
		return "construct 	{?" + SUBJECT_VARIABLE + getPredicateParameter(predicate) + getObjectParameter( object, objectType ) + "}";
	}
	private String constructClause(EIURI predicate) {
		return "construct 	{?" + SUBJECT_VARIABLE + getPredicateParameter(predicate) + "?" + OBJECT_VARIABLE + "}";
	}

	private String subjectTypeClause(EIURI typeURI) {
		return "?" + SUBJECT_VARIABLE + " a <" + typeURI + "> .";
	}
	
	private String graphClause(EIURI predicate, String object, String graph, RDFobjectType rdfType) {
		return " graph ?g {" + tripleForWhereCondition( predicate, object, rdfType ) + "}  filter(?g = <" + graph + ">)";
	}
	private String graphClause(EIURI predicate, String regex, String graph) {
		return " graph ?g {" + tripleForWhereConditionWhenRegex( predicate, regex ) + "}  filter(?g = <" + graph + ">)";
	}
	
	private String tripleForWhereCondition(EIURI predicate, String object, RDFobjectType objectType) {
		return " ?" + SUBJECT_VARIABLE + getPredicateParameter( predicate ) + getObjectParameter( object, objectType ) + ". ";
	}

	private String tripleForWhereConditionWhenRegex(EIURI predicate, String regex) {
		return " ?" + SUBJECT_VARIABLE  + getPredicateParameter( predicate ) + " ?" + OBJECT_VARIABLE + " . filter regex(str(?" + OBJECT_VARIABLE + "), \"" + forRegex(regex) + "\") ";
	}

	public static String forRegex(String aRegexFragment) {
		final StringBuilder result = new StringBuilder();
		final StringCharacterIterator iterator = new StringCharacterIterator( aRegexFragment );
		char character = iterator.current();
		while ( character != CharacterIterator.DONE ) {
			 if ( character == '\\' ) {
					result.append( "\\\\" );
				} 
			 else
			 {
				 result.append( character );
			 }
			 character = iterator.next();
		}
		return result.toString();
	}



	//FIXME don' think the prefix is needed; check with Srav (dbw)
	private StringBuilder selectClauseWithPrefix() {
		final StringBuilder sparql = new StringBuilder();
		//sparql.append( "PREFIX " ).append( EAGLE_I_DT_PREFIX ).append( ": <" ).append( EAGLE_I_DT_URI ).append( "> " );
		
		//don't use SparqlUtil version - we don' want select distinct (dbw)
		sparql.append( "SELECT " );
		List<String> resultVariables = Arrays.asList( SUBJECT_VARIABLE, TYPE_VARIABLE, TYPE_LABEL_VARIABLE, LABEL_VARIABLE, ANY_PREDICATE_VARIABLE, ANY_PREDICATE_LABEL_VARIABLE, OBJECT_VARIABLE, OBJECT_LABEL_VARIABLE );
		for (final String variable : resultVariables) {
			sparql.append( "?" ).append( variable ).append( " " );
		}
		sparql.append( "WHERE { " );
		return sparql;
	}

	private String getPredicateParameter(EIURI predicate) {
		if ( isAllOption(predicate)) {
			return " ?" + ANY_PREDICATE_VARIABLE + " ";
		} else {
			return " <" + predicate.toString() + "> ";
		}
	}
	
	private String getObjectParameter(String object, RDFobjectType objectType) {
		if( isAllOptionObject(object)) {
			return " ?" + OBJECT_VARIABLE + " ";
		} else if ( objectType == RDFobjectType.objectIsLiteral ) {
			return " \"" + object + "\" ";
		} else {
			return " <" + object + "> ";
		}
	}
	
	private boolean isAllOption(EIURI parameter) {
		return parameter==null || EIURI.NULL_EIURI.equals( parameter);
	}
	
	private boolean isAllOptionObject(String parameter) {
		return parameter==null;
	}


	public String objectLiteralWrappedinSTRfuntion(final String objectLiteral) {
		return " str(\"" + objectLiteral + "\") ";
	}

	@Override
	public String getSanityCheckQuery(EIURI type, EIURI predicate, String object, RDFobjectType objectType) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("SELECT ?").append(SUBJECT_VARIABLE).append(" where {");
		if(!isAllOption( type )) {
			sparql.append(subjectTypeClause( type ));
		}
		sparql.append("graph ?g{ ?").append(SUBJECT_VARIABLE).append(getPredicateParameter( predicate )).append( getObjectParameter( object, objectType ));
		sparql.append("} filter(?g != <http://eagle-i.org/ont/repo/1.0/NG_Inferred>)}");
		return sparql.toString();
	}


	
	
}
