/**
 * eagle-i Project
 * Harvard University
 * Jan 31, 2010
 */
package org.eaglei.datatools.interim.gsites;

import static org.eaglei.datatools.interim.gsites.GSitesConstants.DROPBOX;
import static org.eaglei.datatools.interim.gsites.GSitesConstants.IDC_PATH;
import static org.eaglei.datatools.interim.gsites.GSitesConstants.PASSWORD;
import static org.eaglei.datatools.interim.gsites.GSitesConstants.RNAVS;
import static org.eaglei.datatools.interim.gsites.GSitesConstants.SITEURL;
import static org.eaglei.datatools.interim.gsites.GSitesConstants.THIS_APP;
import static org.eaglei.datatools.interim.gsites.GSitesConstants.USER;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collection;

import javax.activation.MimetypesFileTypeMap;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.interim.cores.OldFileUtils;

import com.google.gdata.client.ClientLoginAccountType;
import com.google.gdata.client.sites.ContentQuery;
import com.google.gdata.client.sites.SitesService;
import com.google.gdata.data.Link;
import com.google.gdata.data.MediaContent;
import com.google.gdata.data.OutOfLineContent;
import com.google.gdata.data.PlainTextConstruct;
import com.google.gdata.data.media.MediaFileSource;
import com.google.gdata.data.media.MediaSource;
import com.google.gdata.data.sites.AttachmentEntry;
import com.google.gdata.data.sites.BaseContentEntry;
import com.google.gdata.data.sites.ContentFeed;
import com.google.gdata.data.sites.FileCabinetPageEntry;
import com.google.gdata.data.sites.SitesLink;
import com.google.gdata.data.sites.WebPageEntry;
import com.google.gdata.util.AuthenticationException;
import com.google.gdata.util.ServiceException;
/**
 * @author dbw
 *
 */
public class GSitesUtils {
	private static final Log logger = LogFactory.getLog(GSitesUtils.class);
	private static SitesService client;
	private static MimetypesFileTypeMap mediaTypes = new MimetypesFileTypeMap();
	static {
		mediaTypes.addMimeTypes("application/vnd.ms-excel xlsx");
		mediaTypes.addMimeTypes("application/vnd.openxmlformats docx pptx xlsx");
	}
	private static String separator = System.getProperty("file.separator");
    /**
     * Returns an entry's numeric ID.
     */
    private static String getEntryId(BaseContentEntry<?> entry) {
      String selfLink = entry.getSelfLink().getHref();
      return selfLink.substring(selfLink.lastIndexOf("/") + 1);
    }

    /**
     * Returns an entry's numeric ID.
     */
    private static String getEntryId(String selfLink) {
      return selfLink.substring(selfLink.lastIndexOf("/") + 1);
    }
    
	/**
	 * Login to GSite using daemon user
	 */
	public static void login() {
		client = new SitesService(THIS_APP);
		try {
			client.setUserCredentials(USER, PASSWORD, ClientLoginAccountType.HOSTED);
		} catch (AuthenticationException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		// UserToken auth_token = (UserToken) service.getAuthTokenFactory().getAuthToken();
		//TODO Save this token auth_token.getValue();
		//TODO get token instead of logging in each time
	}
	

	/**
	 * Get attachments for all RNavs 
	 * @param basePath path to save files
	 * @throws IOException
	 * @throws ServiceException
	 */
	public static void getAttachments(String basePath, boolean delete) throws IOException, ServiceException {	
		for (String rnav : RNAVS) {
			getAttachmentsForRnav(rnav, basePath, delete);
		}
	}
	
	public static void setAttachments(String basePath) throws IOException, ServiceException {	
		for (String rnav : RNAVS) {
			setAttachmentsForRnav(rnav, basePath);
		}
	}
	
	public static void getDropboxes(String basePath, boolean delete) throws IOException, ServiceException {	
		for (String rnav : RNAVS) {
			getDropboxForRnav(rnav, basePath, delete);
		}
	}
	
	/**
	 * Get attachments for one RNav
	 * @param rnav
	 * @param basePath path to save files
	 * @throws IOException
	 * @throws ServiceException
	 */
	//TODO - use getBaseContent
	
	public static void getAttachmentsForRnav(String rnav, String basePath, boolean delete) throws IOException, ServiceException {
		URL url = new URL(SITEURL);	
		final ContentQuery query = new ContentQuery(url);
		query.setPath(IDC_PATH + rnav);
		//TODO take care of file cabinets
		query.setKind("webpage");
		final ContentFeed contentFeed = client.getFeed(query, ContentFeed.class);
		//There should be only 1
		for (BaseContentEntry<?> p : contentFeed.getEntries()) {
			final String ID = p.getId();
			if(logger.isDebugEnabled()) {
				logger.debug("Inspecting page: " + p.getTitle().getPlainText());
				logger.debug("Page has ID: " + ID);
			}
			WebPageEntry page = new WebPageEntry(p);
			String relPath = page.getPageName().getValue();
			if(!basePath.endsWith(separator))
				getAttachmentsAtPath(getEntryId(ID), basePath + separator + relPath, delete);
			else
				getAttachmentsAtPath(getEntryId(ID), basePath + relPath, delete);
		}
	}
	
	//TODO - use getBaseContent
	
	public static void setAttachmentsForRnav(String rnav, String basePath) throws IOException, ServiceException {
		URL url = new URL(SITEURL);	
		String rnavPath = basePath + separator + rnav;
		File rnavDir = new File(rnavPath);
		OldFileUtils.validateDirectory(rnavDir, false);
		String[] extensions = {"xlsx", "xls"};
		Collection<File> files = FileUtils.listFiles(rnavDir, extensions, true);
		for(File file : files) {
			if(logger.isDebugEnabled())logger.debug("Processing file: " + file.getAbsolutePath());
			//determine URI path 
			final String parentPath = file.getParent();
			final int i = parentPath.indexOf(rnav);
			final String urlFragment;
			if(i>0) 
				urlFragment = parentPath.substring(i);
			else 
				throw new IOException("rnav subdirectory not found in path");
			final String parentUrl = IDC_PATH + urlFragment;
			//get entry id
			final ContentQuery query = new ContentQuery(url);
			query.setPath(parentUrl);
			final ContentFeed contentFeed = client.getFeed(query, ContentFeed.class);
			//There should be only 1
			BaseContentEntry<?> pageEntry;
			for (BaseContentEntry<?> p : contentFeed.getEntries()) {
				final String ID = p.getId();
				pageEntry = (BaseContentEntry<?>)p;
				if(logger.isDebugEnabled()) {
					logger.debug("Inspecting page: " + p.getTitle().getPlainText());
					logger.debug("Page has ID: " + ID);
				}
			//upload
				  AttachmentEntry newAttachment = new AttachmentEntry();
				  newAttachment.setMediaSource(new MediaFileSource(file, mediaTypes.getContentType(file)));
				  String fileName = file.getName();				
				  newAttachment.setTitle(new PlainTextConstruct(fileName));
				  newAttachment.addLink(SitesLink.Rel.PARENT, Link.Type.ATOM, pageEntry.getSelfLink().getHref());
				  client.insert(url, newAttachment);
				  if(logger.isDebugEnabled())logger.debug("Attachment uploaded");
		}
		}
	}
	
	public static void getDropboxForRnav(String rnav, String basePath, boolean delete) throws IOException, ServiceException {
			// "/" is OK as this is URL
			BaseContentEntry<?> p = getBaseContentEntry(rnav + "/" + DROPBOX);
			if(p == null)
				return;
			String rnavPath = basePath + separator + rnav;
			OldFileUtils.validateDirectory(rnavPath, true);
			WebPageEntry page = new WebPageEntry(p);
			String relPath = rnav + separator + page.getPageName().getValue();
			final String ID = p.getId();
			if(!basePath.endsWith(separator))
				getAttachmentsAtPath(getEntryId(ID), basePath + separator + relPath, delete);
			else
				getAttachmentsAtPath(getEntryId(ID), basePath + relPath, delete);	
	}

	private static BaseContentEntry<?> getBaseContentEntry(String urlFragment) throws IOException, ServiceException{
		URL url = new URL(SITEURL);	
		final ContentQuery query = new ContentQuery(url);
		query.setPath(IDC_PATH + urlFragment);
		query.setKind("webpage");
		final ContentFeed contentFeed = client.getFeed(query, ContentFeed.class);
		//There should be only 1 because the path is already set via setPath
		for (BaseContentEntry<?> p : contentFeed.getEntries()) {
			final String ID = p.getId();
			if(logger.isDebugEnabled()) {
				logger.debug("Inspecting page: " + p.getTitle().getPlainText());
				logger.debug("Page has ID: " + ID);
			}
			return p;
		}
		return null;		
	}
	
	/**
	 * Recursive method to traverse the hierarchy
	 * @param node start node
	 * @param relPath relative path to save file
	 * @throws IOException
	 * @throws ServiceException
	 */
	private static void getAttachmentsAtPath(String node, String relPath, boolean delete) throws IOException, ServiceException {
		OldFileUtils.validateDirectory(relPath, true);
		URL url = new URL(SITEURL);	
		final ContentQuery query = new ContentQuery(url);
		query.setParent(node);
		query.setKind("attachment");
		final ContentFeed atContentFeed = client.getFeed(query, ContentFeed.class);	
		//Iterate over attachments
		for(BaseContentEntry<?> att : atContentFeed.getEntries()) {
			String u = ((OutOfLineContent) att.getContent()).getUri();
			String fileName = att.getTitle().getPlainText();
			if(logger.isDebugEnabled()) {
				logger.debug("Downloading file: " + fileName);
				logger.debug("At URL: " + u);
			}
			downloadAttachment(u, relPath, att.getTitle().getPlainText());
			//TODO for now delete only used in dropboxes so everything can be deleted
			//if(delete && (fileName.endsWith("xlsx") || fileName.endsWith("xls"))) {
			if(delete) {
				if(logger.isDebugEnabled()) logger.debug("Deleting attachment");
				att.delete();
			}
		}
		query.setKind("webpage");
		final ContentFeed wpContentFeed = client.getFeed(query, ContentFeed.class);	
		//Iterate over webpages
		for(BaseContentEntry<?> wp : wpContentFeed.getEntries()) {
			WebPageEntry w = new WebPageEntry(wp);
			String ID = getEntryId(wp.getId());
			String name = w.getPageName().getValue();
			if(logger.isDebugEnabled()) {
				logger.debug("Inspecting subpage " + name);
				logger.debug("Subpage ID: " + ID);
			}
			if(!relPath.endsWith(separator))
				getAttachmentsAtPath(ID, relPath + separator + name, delete);
			else
				getAttachmentsAtPath(ID, relPath + name, delete);
		}
		query.setKind("filecabinet");
		final ContentFeed fcContentFeed = client.getFeed(query, ContentFeed.class);	
		//Iterate over filecabinets
		for(BaseContentEntry<?> fc : fcContentFeed.getEntries()) {
			FileCabinetPageEntry f = new FileCabinetPageEntry(fc);
			String ID = getEntryId(fc.getId());
			String name = f.getPageName().getValue();
			if(logger.isDebugEnabled()) {
				logger.debug("Inspecting subpage " + name);
				logger.debug("Subpage ID: " + ID);
			}
			if(!relPath.endsWith(separator))
				getAttachmentsAtPath(ID, relPath + separator + name, delete);
			else
				getAttachmentsAtPath(ID, relPath + name, delete);
		}

	}

	
	/**
	 * Do the actual download
	 * @param uri of the file to download
	 * @param dir the full directory path to save the file
	 * @param fileName - for saving the file
	 * @throws IOException
	 * @throws ServiceException
	 */
	private static void downloadAttachment(String uri, String dir, String fileName) throws IOException, ServiceException {
		OldFileUtils.validateDirectory(dir, true);
		String fullFilePath;
		if(!dir.endsWith(separator))
			fullFilePath = dir + separator + fileName;
		else
			fullFilePath = dir + fileName;
		MediaContent mc = new MediaContent();
		mc.setUri(uri);
		MediaSource ms = client.getMedia(mc);

		InputStream inStream = null;
		FileOutputStream outStream = null;

		try {
			inStream = ms.getInputStream();
			outStream = new FileOutputStream(fullFilePath);

			int c;
			while ((c = inStream.read()) != -1) {
				outStream.write(c);
			}
		} finally {
			if (inStream != null) {
				inStream.close();
			}
			if (outStream != null) {
				outStream.flush();
				outStream.close();
			}
		}
	}

}

