package org.eaglei.datatools.jena;

import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.CreateInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetInstance;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.GetNewInstances;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Logout;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Online;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Query;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.Upload;
import static org.eaglei.datatools.jena.RESTRepositoryProvider.RestCommands.WhoAmI;
import static org.eaglei.datatools.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.datatools.jena.SPARQLConstants.STATE_VARIABLE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.NOT_AUTHORIZED_MESSAGE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.NO_SESSION_MESSAGE;
import static org.eaglei.datatools.provider.RepositoryProviderMessages.UNAVAILABLE_MESSAGE;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.provider.EIDataToolsProviderException;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProviderMessages;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */
public final class RESTRepositoryProvider implements RepositoryProvider {
	
	
	protected static final String	READ_VIEW			= "user";
	//string sent as part of http requests
	private static final String		FORMAT_VALUE		= "application/xml";
	
	//format string used for all Jena calls
	private static final String     RDF_FORMAT 			= "RDF/XML";
	private static final Log		log					= LogFactory.getLog(RESTRepositoryProvider.class);
	private static final boolean	isDebugEnabled		= log.isDebugEnabled();
	private static String			DEFAULT_REPOSITORY	= "";
	private static String			DEFAULT_NAMESPACE	= "";
	
	enum RestCommands {
		GetNewInstances("repository/new"), WhoAmI("repository/whoami"), Logout("repository/logout"), GetInstance("i"), CreateInstance("repository/update"), Query("repository/sparql"), Online(""), Upload("repository/graph"), Workflow("repository/fakeworkflow");
		private RestCommands(final String propKey) {
			key = propKey;
		}
		public String getURL() {
			return DEFAULT_REPOSITORY + key;
		}
		private final String	key;
	}
	
	enum LoginFields {
		USERNAME(0), SESSION(1), USER_URI(2), ERROR_MESSAGE(3), WF_STATE(3);
		
		private final int value;
		private LoginFields(final int value)
		{
			this.value = value;
		}
		
		public int getValue()
		{
			return value;
		}
	}
	
	private ResultSet					results;
	private Map<String, HttpClient>	clientMap	= new HashMap<String, HttpClient>();
		
	/*
	 * Private constructor, must access via static INSTANCE field
	 */
	public RESTRepositoryProvider(DatatoolsConfiguration config) {
		if (config != null) {
			// TODO: at some point we need to get the auth information via
			// Sessions or other method
			DEFAULT_REPOSITORY = config.getDatatoolsRepositoryURL();
			DEFAULT_NAMESPACE = config.getDatatoolsRepositoryNamespace();
		}
	}
	
	private HttpClient makeHttpClient(String username, String password) {
		HttpClient client = new HttpClient();
		client.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
		client.getState().setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(username, password));
		client.getParams().setAuthenticationPreemptive(true);
		
		return client;
	}
	
	private HttpClient getHttpClient(String session) {
		if (clientMap.get(session) == null)
		{
			return null; // or error indicating we need a new one?
		}
		
		return clientMap.get(session);
	}
	
	protected void setWriteParameters(PostMethod method, String id) {
		method.setParameter("uri", id);
		method.setParameter("format", FORMAT_VALUE);
		//As per conversation w Larry, repository/update doesn't need a view/graph except for create
		//and removing it when unneeded avoids mismatch of views
		//will add it only in createInstance
		//method.setParameter("workspace", "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace");
		method.setRequestHeader("charset", "UTF-8");
	}
	
	protected void setReadParameters(PostMethod method) {
		method.setParameter("format", FORMAT_VALUE);
		method.setParameter("view", READ_VIEW);
		method.setParameter("noinferred", "true");
		method.setRequestHeader("charset", "UTF-8");
	}
	
	@Override
	public void createInstance(String session, final EIInstance instance) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if ((instance == null) || (instance.getInstanceURI() == null)) {
			log.warn("Trying to create (duplicate) instance with null instance or null instance URI");
			return;
		}
		final String id = instance.getInstanceURI().toString();
		int status = 0;
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		setWriteParameters(method, id);
		method.setParameter("workspace", "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace");
		method.setParameter("action", "create");

		String instanceAsString = EIInstanceFactory.INSTANCE.serialize(instance, RDF_FORMAT);
		if(isDebugEnabled) log.debug("dump data: " + instanceAsString);
		method.setParameter("insert", instanceAsString);
		if(isDebugEnabled) log.debug("Trying to get create instance at " + id);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("get instance succeded with status: " + status + " response: " + resp);
			} else {
                log.error("create instance failed with status: " + status + " response: " + resp);
                throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("create", status));
            }

        }
        finally
        {
            method.releaseConnection();
        }

	}
	
	public void createInstance(final String session, final String rdfString, String id) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		
		if ((rdfString == null)) {
			log.warn("Trying to create (duplicate) instance with null instance or null instance URI");
			return;
		}
		// final String id = instance.getInstanceURI().toString();
		int status = 0;
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		setWriteParameters(method, id);
		method.setParameter("workspace", "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace");
		method.setParameter("action", "create");
		if(isDebugEnabled) log.debug("dump data: " + rdfString);
		method.setParameter("insert", rdfString);
		if(isDebugEnabled) log.debug("Trying to get create instance at " + id);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("get instance succeded with status: " + status + " response: " + resp);
			} else {
				log.error("create instance failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("create", status));
			}
		} finally {
			method.releaseConnection();
		}
	}
	
	@Override
	public EIInstance deepCopy(String session, final EIURI originalUri) throws Exception {
		EIInstance original = getInstance(session, originalUri);
		if (isDebugEnabled) log.debug("got original instance");
		return duplicateInstance(session, original);
	}
	
	public EIInstance duplicateInstance(String session, final EIInstance original) throws Exception {
		EIInstance duplicate = getEmptyEIInstance(session, original.getInstanceType().getURI());
		
		for (EIEntity property : original.getDatatypeProperties().keySet()) {
			for (String value : original.getDatatypeProperty(property)) {
				duplicate.addDatattypeProperty(property, value);				
			}
		}
		
		for (EIEntity property : original.getObjectProperties().keySet()) {
			for (EIURI value : original.getObjectProperty(property)) {
				duplicate.addObjectProperty(property, value);				
			}
		}
		
		for (EIEntity property : original.getNonOntologyLiteralProperties().keySet()) {
			for (String value : original.getNonOntologyLiteralProperty(property)) {
				duplicate.addNonOntologyLiteralProperty(property, value);				
			}
		}
		
		for (EIEntity property : original.getNonOntologyResourceProperties().keySet()) {
			for (EIURI value : original.getNonOntologyResourceProperty(property)) {
				duplicate.addNonOntologyResourceProperty(property, value);				
			}
		}
		
		if (isDebugEnabled) log.debug("finished making new copy");
		return duplicate;
	}
	
	@Override
	public void deleteInstance(String session, final EIInstance instance) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if ((instance == null) || (instance.getInstanceURI() == null) || (instance.getInstanceURI().toString() == null)) {
			return;
		}
		final String id = instance.getInstanceURI().toString();
		String token = makeResponseRequest(session, instance);
		int status = 0;
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		setWriteParameters(method, id);

		method.setParameter("action", "update");
		method.setParameter("token", token);
		// if(isDebugEnabled) log.debug("dump data: " +
		// EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
		method.setParameter("delete", createDeleteStatement(instance));
		if(isDebugEnabled) log.debug("Trying to get delete instance at " + id);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("delete instance succeded with status: " + status + " response: " + resp);
			} else
            {
                log.error("delete instance failed with status: " + status + " response: " + resp);
                throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("delete", status));
            }
        } finally
        {
            method.releaseConnection();
        }
	}
	
	@Override
	public String updateInstance(String session, final EIInstance instance, String token)  throws Exception
	{
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if ((instance == null) || (instance.getInstanceURI() == null)) {
			log.warn("Trying to update instance with null instance or null instance URI");
			return null;
		}
		if (token == null) {
			return makeResponseRequest(session, instance);
		}
		int status = 0;
		String id;
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		id = instance.getInstanceURI().toString();
		setWriteParameters(method, id);
		if(isDebugEnabled) log.debug("Trying to get update instance at " + id);
		
		method.setParameter("insert", EIInstanceFactory.INSTANCE.serialize(instance, RDF_FORMAT));
		method.setParameter("delete", createDeleteStatement(instance));
		method.setParameter("action", "update");
		method.setParameter("token", token);

		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				// check to see if we asked for token then return it
				// otherwise just return the response
				log.info("update succeded with status: " + status + " response size: " + resp.length());
				return resp;
			} else {
                log.error("update instance failed with status: " + status + " response: " + resp);
                throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("update", status));
            } 
		}finally {
			method.releaseConnection();
		}
	}
	
	
	public String updateInstance(final String session, final String rdfString, final String id, String token)  throws Exception
	{
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if ((rdfString == null)) {
			log.warn("Trying to create (duplicate) instance with null instance or null instance URI");
			return null;
		}
		if (token == null) {
			return makeResponseRequest(session, rdfString, id);
		}
		int status = 0;
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		setWriteParameters(method, id);
	
		method.setParameter("insert", rdfString);
		method.setParameter("delete", createDeleteStatement(id));
		if(isDebugEnabled) log.debug("Trying to get update instance at " + id);
		method.setParameter("action", "update");
		method.setParameter("token", token);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				// check to see if we asked for token then return it
				// otherwise just return the response
				log.info("update succeded with status: " + status + " response size: " + resp.length());
				return resp;
			} else {
				log.error("update instance failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("update", status));
			}
		} finally {
			method.releaseConnection();
		}
	}
	
	// TODO:
	// This is a brute force approach to getting the repository to delete ALL
	// properties before inserting a new one:
	// a better future approach:
	// 1) obtain the CURRENT instance
	// 2) diff this with the "updated" instance
	// 3) only "delete" and "update" those new statements
	private String createDeleteStatement(EIInstance instance) {
		EIURI id = instance.getInstanceURI();
		String s = null;
		String wildcard = "<" + id.toString() + "> <http://eagle-i.org/ont/repo/1.0/MatchAnything> <http://eagle-i.org/ont/repo/1.0/MatchAnything> .";
		log.info("Trying to create a wildcard graph to delete all the statements");
		Model m = ModelFactory.createDefaultModel();
		StringReader st = new StringReader(wildcard);
		m.read(st, null, "N-TRIPLE");
		StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			m.write(sw, RDF_FORMAT);
			s = sw.toString();
			sw.flush();
		} finally {
			try {
				if (sw != null)
					sw.close();
			} catch (IOException e) {
				log.warn("createDelete statement: " + e);
			}
		}
		// if(isDebugEnabled) log.debug("wildcard: " + s);
		return s;
	}
	
	private String createDeleteStatement(String id) {
		// EIURI id = instance.getInstanceURI();
		String s = null;
		String wildcard = "<" + id + "> <http://eagle-i.org/ont/repo/1.0/MatchAnything> <http://eagle-i.org/ont/repo/1.0/MatchAnything> .";
		log.info("Trying to create a wildcard graph to delete all the statements");
		Model m = ModelFactory.createDefaultModel();
		StringReader st = new StringReader(wildcard);
		m.read(st, null, "N3");
		StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			m.write(sw, RDF_FORMAT);
			s = sw.toString();
			sw.flush();
		} finally {
			try {
				if (sw != null)
					sw.close();
			} catch (IOException e) {
				log.warn("createDelete statement: " + e);
			}
		}
		// if(isDebugEnabled) log.debug("wildcard: " + s);
		return s;
	}
	
	protected String makeResponseRequest(String session, final EIInstance instance) throws Exception {
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		String id = instance.getInstanceURI().toString();
		setWriteParameters(method, id);
		
		method.setParameter("insert", EIInstanceFactory.INSTANCE.serialize(instance, RDF_FORMAT));
		method.setParameter("action", "gettoken");
		int status = 0;
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				// check to see if we asked for token then return it
				// otherwise just return the response
				return getResponseToken(resp);
			} else if (status == 409) // TODO: what's the right code here?
			{
				log.error("stale token: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.STALE_TOKEN_MESSAGE + resp);
			} else 
            {
                log.error("get token failed with status: " + status + " response: " + resp);
                throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get token", status));
            }
		} finally {
			method.releaseConnection();
		}
	}
	
	protected String makeResponseRequest(final String session, final String rdfString, final String id) throws Exception {
		final PostMethod method = new PostMethod(CreateInstance.getURL());
		setWriteParameters(method, id);

		method.setParameter("insert", rdfString);
		method.setParameter("action", "gettoken");
		int status = 0;
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				// check to see if we asked for token then return it
				// otherwise just return the response
				return getResponseToken(resp);
			} else {
				log.error("get token failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get token", status));
			}
		} finally {
			method.releaseConnection();
		}
	}
	
	private String getResponseToken(final String resp) {
		results = ResultSetFactory.fromXML(resp);
		final QuerySolution soln = results.nextSolution();
		final Resource r = soln.getResource("token");
		final String tokenid = r.getURI();
		log.info("Retrieved new token " + tokenid);
		return tokenid;
	}
	
	@Override
	public List<EIInstance> getAllResources(final String session, final String rnav) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if(isDebugEnabled) log.debug("getAllResourcesForUser: " + rnav);
		String q = SPARQLQueryUtil.getInstance().getAllResourcesQuery(rnav);
		return EIQuery(session, q);
	}
	
	private synchronized String postQuery(String session, String q) throws Exception {
		return postQuery(session, q, READ_VIEW);
	}
	
	private synchronized String postQuery(String session, String q, String view) throws Exception {
		if (q == null) {
			log.warn("Null query!");
			return null;
		}
		int status = 0;
		final PostMethod method = new PostMethod(Query.getURL());
		if(isDebugEnabled) log.debug("Trying to query: " + Query.getURL());
		setReadParameters(method);
		method.setParameter("view", view);
		method.setParameter("query", q);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				if(isDebugEnabled) log.debug("query sparql endpoint succeded with status: " + status);
				return resp;
			} else
            {
                log.error("query sparql endpoint  failed with status: " + status);
                log.error("Message from repository: " + resp);
                throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("query", status));
            }
		} finally {
			method.releaseConnection();
		}
	}
	
	@Override
	public List<EIInstance> getResourcesOfClass(final String session, final String rnav, final EIURI classUri) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		String q = SPARQLQueryUtil.getInstance().getResourcesOfClassQuery(rnav, classUri);
		if(isDebugEnabled) log.debug("getResourcesOfClass: " + rnav + " URI: " + classUri);
		return EIQuery(session, q);
	}
	
	@Override
	public EIInstance getInstance(String session, final EIURI instanceID) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		if ((instanceID == null) || instanceID.toString() == null) {
			log.warn("Trying to get instance for null EIURI");
			return null;
		}
		int status = 0;
		final PostMethod method = new PostMethod(GetInstance.getURL());
		setReadParameters(method);
		method.setParameter("uri", instanceID.toString());
		HttpClient client = getHttpClient(session);
		if(isDebugEnabled) log.debug("Trying to get instance " + instanceID.toString());
		try {
			status = client.executeMethod(method);
			if (status == HttpStatus.SC_OK) {
				if(isDebugEnabled) log.debug("started with an OK status");
				if(isDebugEnabled) log.debug("reading in input stream as a model");
				String response = getStringFromInputStream(method.getResponseBodyAsStream());
				EIInstance instance = EIInstanceFactory.INSTANCE.create(instanceID, response, RDF_FORMAT);
				if(isDebugEnabled) log.debug("got instance contents: " + instance);
				return instance;
			}  else {
				log.error("get instance " + instanceID + " failed with status: " + status);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get instance", status));
			}
		} finally {
			method.releaseConnection();
		}
	}
	
	@Override
	public List<EIURI> getNewInstanceID(String session, final int count) throws Exception {
		if (session == null) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		int status = 0;
		ResultSet results = null;
		final PostMethod method = new PostMethod(GetNewInstances.getURL());
		final List<EIURI> instances = new ArrayList<EIURI>(count);
		log.info("getting " + count + " instance IDs");
		try {
			method.setParameter("accept", "application/sparql-results+xml");
			method.setParameter("count", Integer.toString(count));
			HttpClient client = getHttpClient(session); 
 			status = client.executeMethod(method);
			// We'll get this as a resource stream and convert into EIURIs
			// if(isDebugEnabled) log.debug("RESPONSE: " + method.getResponseBodyAsString());
			final String response = getStringFromInputStream(method.getResponseBodyAsStream());
			
			if (status == HttpStatus.SC_OK) {
				results = ResultSetFactory.fromXML(response);
			} else { // TODO: check for an unathenticated session
				log.error("Error getting instance IDs; HTTP Status: " + status + " Response: " + response);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get new ID", status));
			}
		} finally {
			method.releaseConnection();
		}
		for (; results.hasNext();) {
			final QuerySolution soln = results.nextSolution();
			final Resource r = soln.getResource("new");
			final String id = r.getURI();
			instances.add(EIURI.create(id));
			log.info("Created new instance id: " + id);
		}
		if(isDebugEnabled) log.debug("Created " + instances.size() + " ids");
		return instances;
	}
	
	protected String[] getUserInformation(final String session, final HttpClient client) throws Exception
	{
		int status = 0;
		ResultSet results = null;
		String error = null;
		final GetMethod method = new GetMethod(WhoAmI.getURL());
		
		try {
			status = client.executeMethod(method);
			if (status == HttpStatus.SC_OK) {
				results = ResultSetFactory.fromXML(getStringFromInputStream(method.getResponseBodyAsStream()));
				if(isDebugEnabled) log.debug("succeeded in logging in");
			} else {
				error = getStringFromInputStream(method.getResponseBodyAsStream());
			}
		} catch (final Exception e) {
			log.error("problem getting user info " + WhoAmI.getURL() + " Message from repo: " + getStringFromInputStream(method.getResponseBodyAsStream()) +"; Exception " + e);
			throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("get user information", 0));
		} finally {
			method.releaseConnection();
		}

		if(status == HttpStatus.SC_NOT_FOUND) {
			log.error("Repo unavailable");
			throw new EIDataToolsProviderException(UNAVAILABLE_MESSAGE);
		}
		// check to see if the status is UNATHENTICATED
		if (status != HttpStatus.SC_UNAUTHORIZED && results != null) {
			// we are authorized so let's return the user name
			final QuerySolution soln = results.nextSolution();
			final Literal username = soln.getLiteral("username");
			final Resource userURI = soln.getResource("uri");
			
			// Repo right now doesn't contain First/Last names so let's just
			// return the username and URI
			
			log.info("Authenticated user: " + username.getString() + " " + userURI.getURI());
						
			String sessionId = (session == null ? UUID.randomUUID().toString() : session);
			if (session == null) {
				clientMap.put(sessionId, client);
			}
			log.info("session ID: " + sessionId + "; new session? " + (session == null ? "yes" : "no"));
			List<String> states = getWFStatesList(sessionId, userURI.getURI());
			
			states.add(LoginFields.USERNAME.getValue(), username.getString());
			states.add(LoginFields.SESSION.getValue(), sessionId);
			states.add(LoginFields.USER_URI.getValue(), userURI.getURI());
			
			log.info("has login fields and states " + states.size());
			return states.toArray(new String[states.size()]);
		} else {
			log.info("not authorized to get user information (login/whoami)");
			return new String[] {null, null, null, error};
		}
	}
	
	@Override
	public String[] login(final String user, final String password) throws Exception {
		HttpClient client = makeHttpClient(user, password);
		client.getParams().setParameter("accept", "application/sparql-results+xml");
		client.getParams().setAuthenticationPreemptive(true);
		
		if(isDebugEnabled) log.debug("Trying to login at " + WhoAmI.getURL() + " with username " + user);
		String[] results = getUserInformation(null, client);
				
		if (results[LoginFields.USERNAME.getValue()] == null)
		{
			log.error("Could not authenticate user: " + user + "; error message is: " + results[3]);
			throw new EIDataToolsProviderException(NOT_AUTHORIZED_MESSAGE);
		}
		
		String sessionId = results[LoginFields.SESSION.getValue()];
		
		return results;
	}
	
	@Override
	public void logout(String session)  throws Exception
	{
		int status = 0;
		final PostMethod method = new PostMethod(Logout.getURL());
		// now let's try to do a whoami if it succeeds then we are logged in
		// otherwise return null
		if(isDebugEnabled) log.debug("Trying to logout at " + Logout.getURL());
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			// if(isDebugEnabled) log.debug("data dump: " + method.getResponseBodyAsString());
		} catch (final Exception e) {
			log.error("problem with logout  " + Logout.getURL() + " " + e);
			throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("logout", 0));
		} finally {
			method.releaseConnection();
		}
		// check to see if we succeeded
		if (status == HttpStatus.SC_OK) {
			if(isDebugEnabled) log.debug("logout succeded");
			clientMap.remove(session);
		} else {
			log.info("Could not logout user: HTTP Status: " + status);
		}
	}
	
	@Override
	public String[] whoami(String session) throws Exception
	{
		if (session == null) {
			log.info("Using null session.  Could not whoami user");
			return new String[]{null, null, null};
		}
		
		HttpClient client = getHttpClient(session); 
		
		if (client == null) {
			log.info("Using stale session.  Could not whoami user");
			return new String[]{null, null, null};
		}
		
		client.getParams().setParameter("accept", "application/sparql-results+xml");
		// now let's try to do a whoami if it succeeds then we are logged in
		// otherwise return null
		if(isDebugEnabled) log.debug("Trying to whoami at " + WhoAmI.getURL());
		
		String [] results = getUserInformation(session, client);
		
		if (results[LoginFields.USERNAME.getValue()] == null)
		{
			log.info("Could not whoami user: http status: " + results[3]);
			return new String[]{null, null, null};
		}
		
		return results;
	}
	
	@Override
	public EIInstance getEmptyEIInstance(String session, EIURI classUri, EIEntity instanceEntity) throws Exception {
		if (session == null) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		return EIInstanceFactory.getInstance().createEmpty(classUri, instanceEntity);
	}
	
	@Override
	public EIInstance getEmptyEIInstance(String session, EIURI classUri) throws Exception {
		if (session == null) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}

		List<EIURI> ids = getNewInstanceID(session, 1);
		EIInstance instance = EIInstanceFactory.getInstance().createEmpty(classUri, EIEntity.create(ids.get(0), ""));
		
		// TODO: debug
		log.info("got new instance of class " + classUri + "; null class ? " + (instance.getClass() == null ? "yes" : "no"));
		
		return instance;
	}
	
	@Override
	public boolean isOnline() {
		int status = 0;
		boolean online = false;
		final GetMethod method = new GetMethod(Online.getURL());
		HttpClient client = new HttpClient(); 
		client.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
		if(isDebugEnabled) log.debug("Trying to see if Repository is available: " + Online.getURL());
		try {
			status = client.executeMethod(method);
			if (status == HttpStatus.SC_OK || status == HttpStatus.SC_UNAUTHORIZED) {
				online = true;
				if(isDebugEnabled) log.debug("Repository is available: " + Online.getURL() + " is available with status: " + status);
			} else {
				if(isDebugEnabled) log.debug("Repository is unavailable: " + Online.getURL() + " is available with status: " + status);
			}
		} catch (final Exception e) {
			log.error("problem checking online status of repository: " + Online.getURL() + " " + e);
		} finally {
			method.releaseConnection();
		}
		return online;
	}
	
	@Override
	// The return format is hardcoded to RDF/XML
	// FIXME should we allow the format to be specified?
	public String query(String session, String sparql) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		
		return postQuery(session, sparql);
	}
	
	private String getStringFromInputStream(InputStream in) throws IOException {
		StringWriter writer = new StringWriter();
		//encoding needs to be explicitly set
		IOUtils.copy(in, writer, "UTF-8");
		return writer.toString();
	}
	
	@Override
	public void uploadInstances(String session, String rdf) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		
		if (rdf == null) {
			log.warn("Trying to upload null rdf");
			return;
		}
		int status = 0;
		final PostMethod method = new PostMethod(Upload.getURL());
		method.setParameter("format", "text/plain");
		method.setParameter("name", "http://eagle-i.org/ont/repo/1.0/NG_Published");
		method.setParameter("type", "published");
		method.setParameter("action", "add");
		method.setParameter("content", rdf);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("upload succeed with status: " + status + " response: " + resp);
			} else {
				log.error("upload sparql endpoint  failed with status: " + status + " response: " + resp);
			}
		} catch (final Exception e) {
			log.error("problem with upload endpoint at URL  " + Upload.getURL() + " because " + e);
		} finally {
			method.releaseConnection();
		}
	}
		
	@Override
	public String claim(String session, String uri, String claimant) throws Exception {
		if (session == null) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		if ((uri == null) || (claimant == null)) {
			// TODO: other error handling
			log.warn("null claim parameter, returning null: " + (uri == null ? "uri" : (claimant == null ? "claimant" : "huh?")));
			return null;
		}
		int status = 0;
		final PostMethod method = new PostMethod(RestCommands.Workflow.getURL());
		method.setParameter("uri", uri);
		method.setParameter("format", FORMAT_VALUE);
		method.setParameter("claim", claimant);
		log.info("Trying to claim " + uri + " for " + claimant);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("claim succeded with status: " + status + " response: " + resp);
				return resp; // Maybe response code only? Do we return it?
			} else {
                log.error("claim failed with status: " + status + " response: " + resp);
                throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("claim", status));
            }
		} finally {
			method.releaseConnection();
		}
	}
	
	@Override
	public String promote(String session, String uri, String newState) throws Exception {
		if (session == null) {
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		
		if (uri == null) {
			log.warn("Trying to promote a null URI");
			return null;
		}
		
		final String newStateUri = newState; // TODO: look up new state URI from state name?
		int status = 0;
		final PostMethod method = new PostMethod(RestCommands.Workflow.getURL());
		method.setParameter("uri", uri);
		method.setParameter("format", FORMAT_VALUE);
		method.setParameter("promote", newStateUri);
		// if(isDebugEnabled) log.debug("dump data: " +
		// EIInstanceFactory.INSTANCE.serialize(instance, "RDF/XML"));
		if(isDebugEnabled) log.debug("Trying to get promote " + uri + " to state " + newState);
		try {
			HttpClient client = getHttpClient(session);
			status = client.executeMethod(method);
			final String resp = getStringFromInputStream(method.getResponseBodyAsStream());
			if (status == HttpStatus.SC_OK) {
				log.info("promote succeded with status: " + status + " response: " + resp);
				return newStateUri;
			} else {
				log.error("promote failed with status: " + status + " response: " + resp);
				throw new EIDataToolsProviderException(RepositoryProviderMessages.getFailedMessage("promote", status));
			}
		} finally {
			method.releaseConnection();
		}
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eaglei.datatools.provider.RepositoryProvider#EIQuery(java.lang.String
	 * , java.lang.String)
	 */
	@Override
	public List<EIInstance> EIQuery(String session, String sparql) throws Exception {
		if (session == null) {
			log.error("Invalid Session - request cannot be completed");
			throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
		}
		
		if(isDebugEnabled) log.debug("Sparql: " + sparql);
		ResultSet resultSet = ResultSetFactory.fromXML(postQuery(session, sparql));
		return EIInstanceFactory.getInstance().create(resultSet);
	}
	
	@Override
	public List<EIInstance> getInstancesForLab(String session, String rnav, String labURI) throws Exception {
		return EIQuery(session, SPARQLQueryUtil.getInstance().getAllResourcesInLabQuery(rnav, EIURI.create(labURI)));
	}
	
	@Override
	public List<EIInstance> getFilterQuery(final String session, final String user, final EIURI classUri, final EIURI state, final EIURI lab) throws Exception {
		return EIQuery(session, SPARQLQueryUtil.getInstance().getFilterQuery(user, classUri, state, lab));
	}
	
	public List<String> getWFStatesList(String session, String user) throws Exception {
		if (user == null)
		{
			throw new EIDataToolsProviderException("cannot get workflow states for null user");
		}
		if (session != null) {
			String q = SPARQLQueryUtil.getInstance().getWFStatesQuery(user);
			if(isDebugEnabled) log.debug("get wf states query: " + q);
			String result = postQuery(session, q, "all");
			ResultSet resultSet = ResultSetFactory.fromXML(result);
			List<String> states = new ArrayList<String>();
			while(resultSet.hasNext()) {
				final QuerySolution qs = resultSet.next();
				//state is the variable name in the query obtained from SPARQLUtil
				if(qs.contains(STATE_VARIABLE))
					states.add(qs.getResource(STATE_VARIABLE).getURI());
			}
			
			return states;
		}
		
		throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
	}
	
	@Override
	public String[] getWFStates(String session, String user) throws Exception {
		List<String> states = getWFStatesList(session, user);
		return states.toArray(new String[states.size()]);
	}
	
	@Override
	public Map<EIEntity, String> retrieveLabels(String session, List<EIEntity> entities) throws Exception
	{
		if(session != null) {
			// TODO: generate a UNION sparql query instead: this may be slow
			
			log.info("retrieving labels");
			
			Map<EIEntity, String> results = new HashMap<EIEntity, String>(entities.size());
			for (EIEntity entity : entities)
			{
				try {
					results.put(entity, retrieveLabel(session, entity.getURI()));
				}
				catch(EIDataToolsProviderException e) {
					int statusLoc = e.getMessage().indexOf("4");
					if (statusLoc < 0) {
						throw(e);
					}
					
					String statusString = e.getMessage().substring(statusLoc);
					if (! statusString.equals("400")) {
						throw(e);
					}
					
					log.warn("could not retrieve label for " + entity.getURI() + "; skipping");
				}
			} 
			
			String q = SPARQLQueryUtil.getInstance().getRetrieveLabelsQuery(entities);
			if(isDebugEnabled) log.debug("retrieve labels query: " + q);
			String labelsResult = postQuery(session, q, "all");
			if(isDebugEnabled) log.debug("result " + labelsResult); 
			
			/*ResultSet resultSet = ResultSetFactory.fromXML(labelsResult);
			
			for (EIEntity entity : entityList)
			{
				results.put(entity, );
			} */
			
			return results; 
		}
		throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
	}
	
	@Override
	public Map<EIURI, String> retrieveUriLabels(String session, List<EIURI> uris) throws Exception
	{
		if(session != null) {
			// TODO: generate a UNION sparql query instead: this may be slow
			
			log.info("retrieving labels");
			
			Map<EIURI, String> results = new HashMap<EIURI, String>(uris.size());
			for (EIURI uri : uris)
			{
				results.put(uri, retrieveLabel(session, uri));
			} 
			
			String q = SPARQLQueryUtil.getInstance().getRetrieveUriLabelsQuery(uris);
			if(isDebugEnabled) log.debug("retrieve labels query: " + q);
			String labelsResult = postQuery(session, q, "all");
			if(isDebugEnabled) log.debug("result " + labelsResult);
			
			return results; 
		}
		throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
	}
	
	@Override
	public String retrieveLabel(String session, EIURI uri) throws Exception
	{
		if(session != null) {
			String q = SPARQLQueryUtil.getInstance().getRetrieveLabelQuery(uri);
			if(isDebugEnabled) log.debug("retrieve label query: " + q);
			String result = postQuery(session, q, "all");
			if(isDebugEnabled) log.debug("result " + result); 
			ResultSet resultSet = ResultSetFactory.fromXML(result);
			List<String> labels = new ArrayList<String>();
			while(resultSet.hasNext()) {
				final QuerySolution qs = resultSet.next();
				//state is the variable name in the query obtained from SPARQLUtil
				if(qs.contains(LABEL_VARIABLE))
					labels.add(qs.getLiteral(LABEL_VARIABLE).getString());
			}
			//FIXME remove this once we're sure it works
			log.info("labels for " + uri.toString() + ": "); // TODO: back to debug
			for (String label : labels) {
				log.info(label);
			}
			return labels.isEmpty()?"":labels.get(0);
		}
		throw new EIDataToolsProviderException(NO_SESSION_MESSAGE);
	}
}
