package org.eaglei.datatools.jena;

import static org.eaglei.datatools.model.DataToolsOntConstants.DC_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.EAGLE_I_REPO_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.FOAF_PERSON;
import static org.eaglei.datatools.model.DataToolsOntConstants.OBI_ORGANIZATION;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDFS_URI;
import static org.eaglei.datatools.model.DataToolsOntConstants.RDF_URI;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;

/**
 * 
 * @author Ricardo De Lima
 * @author Daniela Bourges
 * 
 *         May 12, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */

//FIXME: when we start using SearchRequest this class may go away; if it doesn't we need to refactor to use ARQ queries (dbw)
//I'm deprecating the actual query methods, we should favor for the short term just the strings/ARQ and in the future SearchRequest

//FIXME: change to use SPARQLConstants for variable names

public class SPARQLQueryUtil {

	private static final Log logger = LogFactory
			.getLog(SPARQLQueryUtil.class);
	
	private static SPARQLQueryUtil instance;

	private SPARQLQueryUtil() {

	}

	public static SPARQLQueryUtil getInstance() {
		if (instance == null) {
			instance = new SPARQLQueryUtil();
		}
		return instance;
	}

	// ETL-specific  queries
	/**
	 * Returns a sparql query that asks the URI of a resource
	 * of type rootURI with a label
	 * 
	 * @param rootURI
	 * @param label
	 * @return the sparql query string
	 */

	public String getEIResourcesByLabelQuery(final EIURI rootURI,
			final String label) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdf: <");
		sparql.append(RDF_URI);
		sparql.append("> ");
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("SELECT ?resource WHERE {?resource rdf:type <");
		sparql.append(rootURI.toString());
		sparql.append("> . ");
		sparql.append("?resource rdfs:label ?o ");
		// Ignore case
		sparql.append("FILTER regex(str(?o), \"" + label + "\", \"i\").}");
		return sparql.toString();

		// The query should be (substitute type and label but keep <> and "" )
		// SELECT ?resource WHERE { ?resource a <type> . ?resource rdfs:label ?o
		// FILTER regex(?o, "label", "i").}

	}

	
	// Web-tool queries
	public String getWFStatesQuery(final String user) {

		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX repo: <");
		sparql.append(EAGLE_I_REPO_URI);
		sparql.append(">");
		sparql.append("SELECT DISTINCT ?state WHERE {?state repo:hasWriter ?role . <");
		sparql.append(user);
		sparql.append("> repo:hasRole ?role .}");
		return sparql.toString();
	
		//select distinct ?s where {?s :hasWriter ?role . <user> :hasRole ?role}
	}
	
	public String getRetrieveLabelQuery(final EIURI uri) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("SELECT DISTINCT ?l WHERE {<");
		sparql.append(uri.toString());
		sparql.append("> rdfs:label ?l}");
		return(sparql.toString());
	}
	
	public String getRetrieveLabelsQuery(final List<EIEntity> entities) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("SELECT DISTINCT ?l WHERE {");
		int i = 0;
		for (EIEntity entity : entities){
			sparql.append("{<");
			sparql.append(entity.getURI().toString());
			sparql.append("> rdfs:label ?l}");
			if (i < entities.size() - 1)
			{
				sparql.append(" UNION ");
			}
			++i;
		}
		sparql.append("}");
		return(sparql.toString());
	}
	
	public String getRetrieveUriLabelsQuery(final List<EIURI> uris) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append("PREFIX rdfs: <");
		sparql.append(RDFS_URI);
		sparql.append("> ");
		sparql.append("SELECT DISTINCT ?l WHERE {");
		int i = 0;
		for (EIURI uri : uris){
			sparql.append("{<");
			sparql.append(uri.toString());
			sparql.append("> rdfs:label ?l}");
			if (i < uris.size() - 1)
			{
				sparql.append(" UNION ");
			}
			++i;
		}
		sparql.append("}");
		return(sparql.toString());
	}
	
	/**
	 * Returns a sparql query that gets all the resources with owner = user or no owner
	 * @param user
	 * @return the sparql query string
	 */
	public String getAllResourcesQuery(final String user) {
		final StringBuilder graphPattern = new StringBuilder();
		graphPattern.append("?s repo:hasWorkflowState ?state . ");
		graphPattern.append(allTypesPattern());
		return assembleQuery(user, graphPattern.toString());
	}




	/**
	 * Returns a sparql query that gets all the resources with rdf:type classUri, with owner=user or no owner
	 * @param user
	 * @param classUri
	 * @return the sparql query string
	 */

	public String getResourcesOfClassQuery(final String user, final EIURI classUri) {
		// FIXME add user in query when we have Workflow Owners
		return getFilterQuery(user, classUri, null, null);
	}

	/**
	 * Returns a sparql query that gets all the resources with owner=user or no owner and that are associated to a lab
	 * @param user
	 * @return the sparql query string
	 */
	public String getAllResourcesInLabQuery(final String user, final EIURI labUri) {
		return getFilterQuery(user, null, null, labUri);
	}

	

	
	//FIXME change String parameters to EIURI to avoid confusion
	
	public String getFilterQuery(final String user, final EIURI classUri, final EIURI state, final EIURI lab) {
		return assembleQuery(user, graphPattern(classUri, state, lab));

	}
	
	
	private String assembleQuery(String user, String graphPattern) {
		final StringBuilder sparql = new StringBuilder();
		sparql.append(namespaces());
		sparql.append("SELECT DISTINCT ");
		sparql.append("?s ?t ?l ?date ?own ?state ");
		sparql.append("WHERE {");
		sparql.append("?s rdfs:label ?l . ");
		sparql.append("OPTIONAL{?s dcterms:created ?date} . ");
		sparql.append(typePattern());
		sparql.append(graphPattern);
		sparql.append(ownerPattern(user));
		sparql.append("} order by ?l");
		return sparql.toString();
	}
	
	private String namespaces() {
		final StringBuilder namespaces = new StringBuilder();
		namespaces.append("PREFIX dcterms: <");
		namespaces.append(DC_URI);
		namespaces.append("> ");
		namespaces.append("PREFIX rdf: <");
		namespaces.append(RDF_URI);
		namespaces.append("> ");
		namespaces.append("PREFIX rdfs: <");
		namespaces.append(RDFS_URI);
		namespaces.append("> ");
		namespaces.append("PREFIX repo: <");
		namespaces.append(EAGLE_I_REPO_URI);
		namespaces.append("> ");
		return namespaces.toString();
	}
	
	
	private String typePattern() {
		return("graph ?g{?s rdf:type ?t} filter(?g != repo:NG_Inferred) . ");
	}
	
	private String allTypesPattern() {
		//Need to use a variable different than ?t, 
		//because that gets only the triples that are not inferred types
		StringBuilder allTypes = new StringBuilder();
		allTypes.append("?s rdf:type ?rt . ");
		allTypes.append("?rt <http://eagle-i.org/ont/app/1.0/inClassGroup> <http://eagle-i.org/ont/app/1.0/ClassGroup/resourceRoot> ");
		//FIXME manually exclude Organizations and Persons as resources
		//should be done by annotations in ontology
		allTypes.append("filter(?rt != <");
		allTypes.append(OBI_ORGANIZATION);
		allTypes.append("> && ?rt != <");
		allTypes.append(FOAF_PERSON);
		allTypes.append(">) .");
		return(allTypes.toString());
	}
	
	private String ownerPattern(String user) {
		StringBuilder owner = new StringBuilder();
		owner.append("OPTIONAL {?s repo:hasWorkflowOwner ?own} FILTER(!bound(?own) || ?own = <");
		owner.append(user);
		owner.append(">)");
		return owner.toString();
	}
	
	//The variable parts of the query
	private String graphPattern(final EIURI classUri, final EIURI state, final EIURI lab) {
		final StringBuilder graphPattern = new StringBuilder();
		if(classUri != null) {
			graphPattern.append("?s rdf:type <");
			graphPattern.append(classUri.toString());
			graphPattern.append("> .");
		} else {
			graphPattern.append(allTypesPattern());
		}
		if(state != null) {
			graphPattern.append("?s repo:hasWorkflowState <");
			graphPattern.append(state.toString());
			graphPattern.append("> . ?s repo:hasWorkflowState ?state .");
		} else {
			graphPattern.append("OPTIONAL {?s repo:hasWorkflowState ?state} . ");
		}
		if(lab != null) {
			graphPattern.append("?s ?p1 <" );
			graphPattern.append(lab.toString());	
			graphPattern.append("> . ");
		} 
		return graphPattern.toString();
	}
	
}
