package org.eaglei.datatools.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.client.QueryTokenObject.Mode;
import org.eaglei.datatools.client.event.BrowserBackOrForwardEvent;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.ui.ApplicationStateChangeListener;
import org.eaglei.datatools.client.ui.ResourceProvider;
import org.eaglei.datatools.client.ui.ResourceProviderCache;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;

import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerManager;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;

public class ApplicationState implements ValueChangeHandler<String> {

	private final QueryTokenObject queryTokenObject;
	private static ApplicationState applicationState;
	final private HandlerManager handlerManager = new HandlerManager( this );
	private String sessionID;
	private EIEntity user;
	private final ArrayList<ApplicationStateChangeListener> listeners;
	private final ResourceProviderCache providerCache;
	//Cache of EIClasses used in forms and left panel
	private List<EIClass> embeddedClasses;
	private List<EIClass> resourceTypesForProvider;
	private List<EIClass> resourceTypesForBrowse;
	private Map<EIEntity, String> classDefinitions;
	private boolean isAcending = true;

	private static final GWTLogger log = GWTLogger.getLogger( "ApplicationState" );

	public static ApplicationState getInstance() {
		if ( applicationState == null ) {
			log.info( "making new application state" );
			if ( !Window.Location.getHash().equals( "" ) ) {
				log.info( "making app state with hash" );
				applicationState = new ApplicationState( Window.Location.getHash() );
			} else {
				log.info( "making app state without hash" );
				applicationState = new ApplicationState();
			}
		}
		return applicationState;
	}

	private ApplicationState() {
		queryTokenObject = new QueryTokenObject();
		listeners = new ArrayList<ApplicationStateChangeListener>();
		History.addValueChangeHandler( this );
		providerCache = new ResourceProviderCache();
		update();
	}

	private ApplicationState(final String bookmark) {
		queryTokenObject = new QueryTokenObject( bookmark );
		listeners = new ArrayList<ApplicationStateChangeListener>();
		History.addValueChangeHandler( this );
		providerCache = new ResourceProviderCache();
	}

	public Mode getMode() {
		return queryTokenObject.getMode();
	}

	public void setMode(final Mode newMode) {
		queryTokenObject.setMode( newMode );
		update();
	}

	public EIEntity getInstanceEntity() {
		return queryTokenObject.getInstanceEntity();
	}

	public EIURI getInstanceUri() {
		return getInstanceEntity().getURI();
	}

	protected void setInstanceEntity(final EIEntity newEntity) {
		queryTokenObject.setInstanceEntity( getCorrectEntity( newEntity ) );
	}

	public boolean hasInstance() {
		return getCorrectEntity( getInstanceEntity() ) != EIEntity.NULL_ENTITY;
	}

	public EIEntity getTypeEntity() {
		return queryTokenObject.getTypeEntity();
	}

	public EIURI getTypeUri() {
		return queryTokenObject.getTypeEntity().getURI();
	}

	private void setTypeEntity(final EIEntity typeEntity) {
		queryTokenObject.setTypeEntity( getCorrectEntity( typeEntity ) );
	}

	public EIEntity getResourceProviderEntity() {
		return getCorrectEntity( queryTokenObject.getResourceProviderEntity() );
	}

	public EIURI getResourceProviderUri() {
		return getResourceProviderEntity().getURI();
	}

	public void setResourceProviderEntity(final EIEntity resourceEntity) {
		queryTokenObject.setResourceProviderEntity( getCorrectEntity( resourceEntity ) );
		// deliberately do NOT update here
	}

	public EIEntity getFilterTypeEntity() {
		return queryTokenObject.getFilterTypeEntity();
	}

	public EIURI getFilterTypeUri() {
		return getFilterTypeEntity().getURI();
	}

	public boolean hasFilterTypeEntity() {
		return getFilterTypeEntity() != EIEntity.NULL_ENTITY;
	}

	private void setFilterTypeEntity(final EIEntity filterTypeEntity) {
		queryTokenObject.setFilterTypeEntity( getCorrectEntity( filterTypeEntity ) );
		update();
	}

	public EIEntity getFilterResourceProviderEntity() {
		return queryTokenObject.getFilterResourceProviderEntity();
	}

	public EIURI getFilterResourceProviderUri() {
		return getFilterResourceProviderEntity().getURI();
	}

	public void setFilterResourceProviderEntity(final EIEntity filterResourceProviderEntity) {
		queryTokenObject.setFilterResourceProviderEntity( filterResourceProviderEntity );
		update();
	}

	public boolean isAcending() {
		return isAcending;
	}

	public void setAcending(boolean isAcending) {
		this.isAcending = isAcending;
	}
	
	public EIEntity getFilterWorkflowEntity() {
		return queryTokenObject.getFilterWorkflowEntity();
	}

	public EIURI getFilterWorkflowUri() {
		return getFilterWorkflowEntity().getURI();
	}

	public void setFilterWorkflowEntity(final EIEntity filterWorkflowEntity) {
		queryTokenObject.setFilterWorkflowEntity( filterWorkflowEntity );
		update();
	}

	public boolean isStrictlyFilteredByOwner() {
		return queryTokenObject.isStrictlyFilteredByOwner();
	}

	public void setStrictFilterByOwner(final boolean filterByOwner) {
		queryTokenObject.setStrictFilterByOwner( filterByOwner );
		update();
	}

	public EIEntity getWorkspaceEntity() {
		return queryTokenObject.getWorkspaceEntity();
	}

	public void setWorkspaceEntity(final EIEntity workspaceEntity) {
		queryTokenObject.setWorkspaceEntity( workspaceEntity );
		update();
	}

	public boolean hasType() {
		return getCorrectEntity( getTypeEntity() ) != EIEntity.NULL_ENTITY;
	}

	public boolean hasResourceProvider() {
		return getCorrectEntity( getResourceProviderEntity() ) != EIEntity.NULL_ENTITY;
		// TODO: does this also need to ask about filterprovider?
	}

	public boolean hasResourceProviderType() {
		return ResourceProvider.isResourceProviderType( getTypeEntity() );
	}

	public void onValueChange(final ValueChangeEvent<String> event) {
		handlerManager.fireEvent( new BrowserBackOrForwardEvent( queryTokenObject ) );
		log.info( "history change event value '" + event.getValue() + "'" );
		queryTokenObject.fromString( event.getValue() );
		notifyListeners();
	}

	private void notifyListeners() {
		for (final ApplicationStateChangeListener listener : listeners) {
			listener.onApplicationStateChange();
		}
	}

	public String getSessionID() {
		return sessionID;
	}

	public void setSessionID(final String sessionID) {
		this.sessionID = sessionID;
	}

	public String getUserID() {
		return user.getLabel();
	}

	public EIURI getUserURI() {
		return user.getURI();
	}

	public EIEntity getUser() {
		return user;
	}

	public void setUser(final EIEntity userEntity) {
		if ( userEntity == null ) {
			user = EIEntity.NULL_ENTITY;
		} else {
			user = userEntity;
		}
	}

	public void updateApplicationState(final Mode mode, final EIEntity instanceEntity, final EIEntity typeEntity, final EIEntity resourceProviderEntity) {
		updateApplicationState( mode, instanceEntity, typeEntity, resourceProviderEntity, false );
	}

	public void updateApplicationState(final Mode mode, final EIEntity instanceEntity, final EIEntity typeEntity, final EIEntity resourceProviderEntity, final boolean updateFilterEntities) {
		queryTokenObject.setMode( mode );
		queryTokenObject.setInstanceEntity( getCorrectEntity( instanceEntity ) );
		queryTokenObject.setTypeEntity( getCorrectEntity( typeEntity ) );
		queryTokenObject.setResourceProviderEntity( getCorrectEntity( resourceProviderEntity ) );
		queryTokenObject.setFilterWorkflowEntity( EIEntity.NULL_ENTITY );
		queryTokenObject.setFilterResourceProviderEntity( updateFilterEntities ? getCorrectEntity( resourceProviderEntity ) : EIEntity.NULL_ENTITY );
		queryTokenObject.setFilterTypeEntity( updateFilterEntities ? getCorrectEntity( typeEntity ) : EIEntity.NULL_ENTITY );
		queryTokenObject.setStrictFilterByOwner( false );
		update();
	}

	public void updatePaging(final String sortColumn, final Integer offset, final Integer limit) {
		queryTokenObject.setSortBy( sortColumn );
		queryTokenObject.setOffset( offset );
		queryTokenObject.setLimit( limit );
		update();
	}

	public void updateToDefaultPaging() {
		queryTokenObject.setDefaultPagination();
		update();
	}

	public Integer getOffset() {
		return queryTokenObject.getOffset();
	}

	public Integer getLimit() {
		return queryTokenObject.getLimit();
	}

	public String getSortBy() {
		return queryTokenObject.getSortBy();
	}

	public void updateApplicationFilterState(final EIEntity filterTypeEntity, final EIEntity filterWorkflowEntity, final EIEntity filterResourceProviderEntity, final boolean strictlyFilteredByOwner) {
		queryTokenObject.setMode( Mode.filter );
		queryTokenObject.setFilterTypeEntity( getCorrectEntity( filterTypeEntity ) );
		queryTokenObject.setFilterWorkflowEntity( getCorrectEntity( filterWorkflowEntity ) );
		queryTokenObject.setFilterResourceProviderEntity( getCorrectEntity( filterResourceProviderEntity ) );
		queryTokenObject.setStrictFilterByOwner( strictlyFilteredByOwner );
		update();
	}

	public void refresh() {
		History.fireCurrentHistoryState();
	}

	private void update() {
		History.newItem( queryTokenObject.toString(), false );
        setLocationJSNI();
        recordAnalyticsHit("datatools");
		notifyListeners();
	}

	public boolean hasUser() {
		return getCorrectEntity( user ) != EIEntity.NULL_ENTITY;
	}

	public void addApplicationStateListener(final ApplicationStateChangeListener listener) {
		listeners.add( listener );
	}

	public void removeApplicationStateListener(final ApplicationStateChangeListener listener) {
		listeners.remove( listener );
	}

	private EIEntity getCorrectEntity(final EIEntity entity) {
		return entity == null ? EIEntity.NULL_ENTITY : entity;
	}

	public boolean hasFilters() {
		return getCorrectEntity( getFilterTypeEntity() ) != EIEntity.NULL_ENTITY || getCorrectEntity( getFilterWorkflowEntity() ) != EIEntity.NULL_ENTITY || getCorrectEntity( getFilterResourceProviderEntity() ) != EIEntity.NULL_ENTITY;
	}

	public void clearApplicationState() {
		queryTokenObject.fromString( "" );
	}

	/**
	 * @return
	 */
	public boolean isPaginated() {
		return ( queryTokenObject.isPaginated() );
	}

	public ResourceProviderCache getResourceProviderCache() {
		return providerCache;
	}

	public void updateResourceProviderCache() {
		providerCache.update();
	}

	public void setEmbeddedClassList(List<EIClass> embbeddedClasses) {
		this.embeddedClasses = embbeddedClasses;
	}
	
	public List<EIClass> getEmbeddedClasses() {
		return embeddedClasses;
	}

	public List<EIClass> getResourceTypesForProvider() {
		return resourceTypesForProvider;
	}

	public void setResourceTypesForProvider(List<EIClass> resourceTypesForProvider) {
		this.resourceTypesForProvider = resourceTypesForProvider;
	}

	public List<EIClass> getResourceTypesForBrowse() {
		return resourceTypesForBrowse;
	}

	public void setResourceTypesForBrowse(List<EIClass> resourceTypesForBrowse) {
		this.resourceTypesForBrowse = resourceTypesForBrowse;
	}

	public Map<EIEntity, String> getClassDefinitions() {
		return classDefinitions;
	}
    
    public static native void setLocationJSNI() /*-{
		//Call the setURL javascript funtion in the html page to pass correct URL to feedback form
		$wnd.setCurrentURL();
	}-*/;
    
    public static native void recordAnalyticsHit(String pageName) /*-{
	    $wnd.trackPage(pageName);
	}-*/;
	
}
