package org.eaglei.datatools.client.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.WorkFlowTransition;
import org.eaglei.datatools.client.AbstractEntryPoint;
import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.WorkFlowConstants;
import org.eaglei.datatools.client.QueryTokenObject.Mode;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.BulkWorkflowCallback;
import org.eaglei.datatools.client.ui.listeners.RowDeletionListener;
import org.eaglei.datatools.client.ui.widgets.EIConfirmationPopup;
import org.eaglei.datatools.client.ui.widgets.GridRowWidget;
import org.eaglei.datatools.client.ui.widgets.PaginationWidget;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;

import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.VerticalPanel;

public class EIResourcesGrid extends Composite implements RowDeletionListener {

	private static final String GRID_ROW_COLUMN_HEADER = "gridRowColumnHeader";
	private static final String RESOURCE_COLUMN_NAMES[] = SortByProperties.getColumnNames();
	private static final Map<String, Comparator<EIInstanceMinimal>> resourceSorters = new HashMap<String, Comparator<EIInstanceMinimal>>();
	public static final String DATATOOLS_FILTER = "DatatoolsFilter";
	private List<EIInstanceMinimal> instances = new ArrayList<EIInstanceMinimal>();
	private final Map<EIInstanceMinimal, GridRowWidget> instanceRowMap = new HashMap<EIInstanceMinimal, GridRowWidget>();
	private final Map<String, WorkFlowTransition> actionTransitionMap = new HashMap<String, WorkFlowTransition>();
	private FilterPanel filterPanel;
	private final FlowPanel resources = new FlowPanel();
	private final FlowPanel innerPanel = new FlowPanel();
	private PaginationWidget topPaginationWidget;
	private PaginationWidget bottomPaginationWidget;
	private final FlowPanel nonGridContentPanel = new FlowPanel();
	private final FlowPanel pageHeaderPanel = new FlowPanel();
	private final FlowPanel subPageHeaderPanel = new FlowPanel();
	// private final FlowPanel

	private static final GWTLogger log = GWTLogger.getLogger( "EIResourcesGrid" );

	public EIResourcesGrid() {
		initialize();
		drawFromApplicationState();
	}

	public EIResourcesGrid(final List<EIInstanceMinimal> instances) {
		log.info( "Creating grid with " + instances.size() + " instances" );
		this.instances = instances;
		initialize();

		log.info( "Creating pagination widgets" );
		topPaginationWidget = new PaginationWidget( ApplicationState.getInstance().getOffset(), instances.size(), ApplicationState.getInstance().getLimit() );
		bottomPaginationWidget = new PaginationWidget( ApplicationState.getInstance().getOffset(), instances.size(), ApplicationState.getInstance().getLimit() );

		resources.remove( nonGridContentPanel );
		drawFromApplicationState();
	}

	private void initialize() {
		initWidget( innerPanel );
		if ( resourceSorters.size() == 0 ) {
			initializeSorters();
		}

		filterPanel = new FilterPanel( ApplicationState.getInstance().hasFilterTypeEntity() ? ApplicationState.getInstance().getFilterTypeEntity() : ApplicationState.getInstance().getTypeEntity(), ApplicationState.getInstance()
				.getFilterWorkflowEntity(), ApplicationState.getInstance().getFilterResourceProviderEntity(), ApplicationState.getInstance().isStrictlyFilteredByOwner() );

		resources.getElement().setClassName( "listGrid" );
		final Label pageHeaderLabel = getHeaderLabel();

		for (final WorkFlowTransition transition : ClientRepositoryToolsManager.INSTANCE.getAllowedTransitions()) {
			/*
			 * don't add in map if from state is equal to state, as well as if toState is draft: see workflow design notes https://sites.google.com/a/eagle-i.org/workspace/build-team-workspace/data-repository-workspace/workflow-design-notes
			 */
			if ( isUsefulTransition( transition ) ) {
				// if ( !transition.getFromStateURI().equals( transition.getToStateURI() ) && ( !transition.getToStateURI().equals( WorkFlowConstants.DRAFT_URI ) || ( transition.getName().equalsIgnoreCase( "Return to Draft" ) ) ) ) {
				actionTransitionMap.put( getName( transition ), transition );
			}

		}
		subPageHeaderPanel.setStyleName( "pageHeaderContainer" );
		if ( !ApplicationState.getInstance().hasResourceProviderType() && !instances.isEmpty() ) {
			final ListBox actionList = makeListBoxOfActions();
			final FlowPanel subPageHeaderRight = new FlowPanel();
			subPageHeaderRight.setStyleName( "pageHeaderRight" );
			subPageHeaderRight.add( actionList );
			subPageHeaderPanel.add( subPageHeaderRight );
		}

		pageHeaderPanel.setStyleName( "pageCategory" );
		pageHeaderPanel.add( pageHeaderLabel );

		innerPanel.add( filterPanel );
		innerPanel.add( pageHeaderPanel );
		innerPanel.add( subPageHeaderPanel );
		innerPanel.add( resources );
		nonGridContentPanel.setStyleName( "resourcesHeaderPanel" );
		final Label emptyHeader = new Label( getEmptyListMessage() );
		nonGridContentPanel.add( emptyHeader );
		resources.add( nonGridContentPanel );
	}

	private String getName(WorkFlowTransition transition) {
		if ( transition.getName().contains( "Return to Draft" ) ) {
			EIEntity fromEntity = WorkFlowConstants.WORKFLOW_URI_MAP.get( transition.getFromStateURI() );
			return transition.getName() + " from " + fromEntity.getLabel();
		}

		return transition.getName();
	}

	private boolean isUsefulTransition(final WorkFlowTransition transition) {
		log.debug( "checking if " + transition.getName() + " is useful" );
		if ( transition.getFromStateURI().equals( transition.getToStateURI() ) || transition.getFromStateURI().equals( WorkFlowConstants.NEW_URI ) ) {
			return false;
		}

		return true;
	}

	private String getEmptyListMessage() {
		if ( ApplicationState.getInstance().getMode() == Mode.references ) {
			return "No resources refer to " + ApplicationState.getInstance().getInstanceEntity().getLabel();
		}
		return "To see a list of resources, click the resource type, such as Instrument, or filter by Status or Laboratory above.";
	}

	private void drawFromApplicationState() {
		if ( ApplicationState.getInstance().getMode() == Mode.resources ) {
			return; // Leave the explanation box, and just get out of here

		} else if ( !Mode.isResourcesList( ApplicationState.getInstance().getMode() ) ) {
			resources.clear();
			return;
		}
		// Only set counts if we are not paginating
		// FIXME Obtain the total count even when paginating and display here
		if ( !ApplicationState.getInstance().isPaginated() ) {
			filterPanel.setCount( instances.size() );
		}
		if ( ApplicationState.getInstance().getMode() == Mode.references ) {
			filterPanel.disable();

		}
		if ( instances.size() == 0 ) {
			resources.remove( nonGridContentPanel );
			resources.clear();
			final Label noInstances = new Label( getNoResourcesMessage() );
			resources.add( noInstances );
			return;
		}

		resources.clear();
		subPageHeaderPanel.add( topPaginationWidget );

		setGridHeaders();
		updateGrid( 0, instances );
		resources.add(bottomPaginationWidget); // I don't think this actually goes here
	}

	private String getNoResourcesMessage() {
		if ( ApplicationState.getInstance().getMode() == Mode.references ) {
			return "No resources refer to " + ApplicationState.getInstance().getInstanceEntity().getLabel();
		}
		return "No resources found";
	}

	private void setGridHeaders() {
		final FlowPanel header = new FlowPanel();
		int i = 0;
		for (final SortByProperties orderBy : SortByProperties.values()) {
			final String columnName = orderBy.getColumnName();
			final Anchor label = new Anchor( columnName );
			final FlowPanel headerPanel = new FlowPanel();
			headerPanel.setStyleName( GRID_ROW_COLUMN_HEADER );
			headerPanel.addStyleName( GRID_ROW_COLUMN_HEADER + i++ );
			if ( orderBy.getParameterName() == ApplicationState.getInstance().getSortBy() ) {
				headerPanel.addStyleName( GRID_ROW_COLUMN_HEADER + "Selected" );
			}
			headerPanel.add( label );

			header.add( headerPanel );
			label.addClickHandler( new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					boolean isAsc = ApplicationState.getInstance().isAcending();
					log.debug( "sorting by " + columnName );
					if ( !ApplicationState.getInstance().isPaginated() ) {
						sortGrid( instances, resourceSorters.get( columnName ) );
					} else {
						// Only update if requested column is different from currently selected one
						// TODO hook ascending/descending order here
						if ( !orderBy.getParameterName().equals( ApplicationState.getInstance().getSortBy() ) ) {
							ApplicationState.getInstance().updatePaging( orderBy.getParameterName(), 0, ApplicationState.getInstance().getLimit() );
						} else {
							ApplicationState.getInstance().setAcending( !isAsc );
							ApplicationState.getInstance().updatePaging( orderBy.getParameterName(), 0, ApplicationState.getInstance().getLimit() );
							headerPanel.setStyleName( GRID_ROW_COLUMN_HEADER + "SelectedDesc" );
						}
					}
				}
			} );
		}

		final FlowPanel headerPanel456 = new FlowPanel();
		headerPanel456.setStyleName( GRID_ROW_COLUMN_HEADER );
		headerPanel456.addStyleName( "gridRowColumnHeader456" );
		final FlowPanel headerPanel7 = new FlowPanel();
		headerPanel7.setStyleName( GRID_ROW_COLUMN_HEADER );
		headerPanel7.addStyleName( "gridRowColumnHeader7" );

		if ( !ApplicationState.getInstance().hasResourceProviderType() ) {
			final CheckBox selectAllCheckBox = makeCheckBoxToSelectAll();
			headerPanel7.add( selectAllCheckBox );
		}

		header.add( headerPanel456 );
		header.add( headerPanel7 );
		resources.add( header );
		header.setStyleName( "dtListGridHeader" );
	}

	private CheckBox makeCheckBoxToSelectAll() {
		final CheckBox checkBox = new CheckBox( "" );
		checkBox.addClickHandler( new ClickHandler() {

			@Override
			public void onClick(final ClickEvent event) {
				final CheckBox selectAllCheckBox = (CheckBox)event.getSource();
				for (final EIInstanceMinimal eiInstance : instances) {
					final GridRowWidget row = instanceRowMap.get( eiInstance );
					final CheckBox chkBox = row.getCheckBox();
					if ( chkBox != null ) {
						chkBox.setValue( selectAllCheckBox.getValue() );
					}
				}
			}
		} );
		return checkBox;
	}

	private void removeRow(final EIInstanceMinimal toRemove) {
		if ( instanceRowMap.containsKey( toRemove ) ) {
			resources.remove( instanceRowMap.get( toRemove ) );
			instanceRowMap.remove( toRemove );
			instances.remove( toRemove );
		}
		filterPanel.setCount( instances.size() );
	}

	private void sortGrid(final List<EIInstanceMinimal> eiInstanceList, final Comparator<EIInstanceMinimal> sorter) {
		Collections.sort( instances, sorter );
		for (final EIInstanceMinimal instance : eiInstanceList) {
			resources.remove( instanceRowMap.get( instance ) );
			resources.add( instanceRowMap.get( instance ) );
		}
	}

	private void updateGrid(final int initialRowNumber, final List<EIInstanceMinimal> eiInstanceList) {
		for (final EIInstanceMinimal instance : eiInstanceList) {
			final GridRowWidget gridRowWidget = new GridRowWidget( instance, this );
			instanceRowMap.put( instance, gridRowWidget );
			resources.add( gridRowWidget );
		}
	}

	private Label getHeaderLabel() {
		// TODO: more to it than that? What if there's a lab, or a filter?
		String labelContent = "";
		if ( ApplicationState.getInstance().hasType() ) {
			labelContent = ApplicationState.getInstance().getTypeEntity().getLabel();
		} else if ( ApplicationState.getInstance().getFilterTypeEntity() != EIEntity.NULL_ENTITY ) {
			labelContent = ApplicationState.getInstance().getFilterTypeEntity().getLabel();
		} else if ( ApplicationState.getInstance().getMode() == Mode.references ) {
			labelContent = "Resources that reference " + ApplicationState.getInstance().getInstanceEntity().getLabel();
		} else if ( ApplicationState.getInstance().getMode() == Mode.stubs ) {
			labelContent = "Unfinished Tasks";
		} else if ( !instances.isEmpty() ) {
			labelContent = "All Resource Types";
		}
		final Label headerLabel = new Label( labelContent );
		headerLabel.setStyleName( "pageHeader" );
		return headerLabel;
	}

	private void initializeSorters() {
		resourceSorters.put( RESOURCE_COLUMN_NAMES[0], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(final EIInstanceMinimal arg0, final EIInstanceMinimal arg1) {
				return arg0.getInstanceLabel().compareToIgnoreCase( arg1.getInstanceLabel() );
			}
		} );
		resourceSorters.put( RESOURCE_COLUMN_NAMES[1], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(final EIInstanceMinimal arg0, final EIInstanceMinimal arg1) {
				return arg0.getInstanceType().getLabel().compareToIgnoreCase( arg1.getInstanceType().getLabel() );
			}
		} );
		resourceSorters.put( RESOURCE_COLUMN_NAMES[2], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(final EIInstanceMinimal arg0, final EIInstanceMinimal arg1) {
				return arg0.getCreationDate().compareToIgnoreCase( arg1.getCreationDate() );
			}
		} );
		resourceSorters.put( RESOURCE_COLUMN_NAMES[3], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(final EIInstanceMinimal arg0, final EIInstanceMinimal arg1) {
				return arg0.getWFState().getLabel().compareToIgnoreCase( arg1.getWFState().getLabel() );
			}
		} );
	}

	private ListBox makeListBoxOfActions() {
		final ListBox actionList = new ListBox();
		actionList.addItem( "Actions" );

		final List<String> actionNames = new ArrayList<String>( actionTransitionMap.keySet() );
		Collections.sort( actionNames );

		for (final String actionName : actionNames) {
			actionList.addItem( actionName );
		}

		actionList.addItem( WorkFlowConstants.CLAIM_ALL_ACTION );
		actionList.addItem( WorkFlowConstants.RELEASE_ALL_ACTION );
		actionList.addChangeHandler( new ChangeHandler() {

			@Override
			public void onChange(final ChangeEvent event) {
				final String selectedText = actionList.getItemText( actionList.getSelectedIndex() );
				final List<EIInstanceMinimal> checkedInstances = getCheckedInstances();
				/* Redisplay Action callback after bulk claim and release */
				final ActionRedisplay redisplay = new ActionRedisplay() {

					@Override
					public void drawAfterClaim(final EIInstanceMinimal[] instances, final List<EIInstanceMinimal> successes) {
						for (final EIInstanceMinimal eiInstance : successes) {
							if ( instanceMeetsFilterCriteria( eiInstance ) ) {
								final GridRowWidget row = instanceRowMap.get( eiInstance );
								row.redrawActions();
							} else {
								removeRow( eiInstance );
							}

						}
						AbstractEntryPoint.hideGlasspane();
					}

					@Override
					public void handleFailedTransition(final String error) {
						AbstractEntryPoint.hideGlasspane();
						Window.alert( UIMessages.TRANSITION_FAILURE.replace( "$TRANSITION$", selectedText ).replace( "$RESOURCES$", getNames( checkedInstances ) ) );
					}

					@Override
					public void drawAfterPromote(final EIInstanceMinimal[] instances, final List<EIInstanceMinimal> successes) {
						AbstractEntryPoint.hideGlasspane();
						for (final EIInstanceMinimal eiInstance : successes) {

							if ( instanceMeetsFilterCriteria( eiInstance ) ) {
								final GridRowWidget row = instanceRowMap.get( eiInstance );
								row.getInstance().setWFOwner( eiInstance.getWFOwner() );
								row.getInstance().setWFState( eiInstance.getWFState() );
								row.updateStateData();
							} else {
								removeRow( eiInstance );
							}
						}
					}

					@Override
					public void needsRefresh(final String message) {
						Window.alert( UIMessages.REFRESH_MULTIPLE_MESSAGE );
						AbstractEntryPoint.hideGlasspane();
					}
				};

				if ( checkedInstances.size() == 0 ) {
					Window.alert( UIMessages.SELECT_MESSAGE );
					actionList.setSelectedIndex( 0 );
					return;
				}

				if ( actionTransitionMap.containsKey( selectedText ) ) {
					final WorkFlowTransition transition = actionTransitionMap.get( selectedText );
					bulkOperation( transition, checkedInstances, makeWorkflowCallback( checkedInstances.toArray( new EIInstanceMinimal[checkedInstances.size()] ), redisplay ) );
				} else if ( selectedText.equalsIgnoreCase( WorkFlowConstants.CLAIM_ALL_ACTION ) ) {
					bulkClaimOrRelease( redisplay, OwnershipAction.ClaimAction, checkedInstances, makeOwnershipCallback( new EIInstanceMinimal[checkedInstances.size()], redisplay ) );
				} else if ( selectedText.equalsIgnoreCase( WorkFlowConstants.RELEASE_ALL_ACTION ) ) {
					bulkClaimOrRelease( redisplay, OwnershipAction.ReleaseAction, checkedInstances, makeOwnershipCallback( new EIInstanceMinimal[checkedInstances.size()], redisplay ) );
				}
				actionList.setSelectedIndex( 0 );
			}
		} );
		return actionList;
	}

	private String getNames(final List<EIInstanceMinimal> checkedInstances) {
		final StringBuilder sb = new StringBuilder();
		for (final EIInstanceMinimal checked : checkedInstances) {
			sb.append( checked.getInstanceLabel() + " " );
		}
		return sb.toString();
	}

	protected void makeConfirmationDialog(final List<EIInstanceMinimal> instanceList, final String name, final String errorMessage, final BulkWorkflowCallback callback, final Performable performable) {
		log.debug( "making confirmation dialog for " + name + " and " + instanceList.size() + " instances" );
		final VerticalPanel vpanel = makeVerticalPanelWithHeading( name + " the following resources:" );
		final ScrollPanel scrPanel = getScrollPanelWithDimensionsSet();
		final FlexTable flexTable = getFlextableWithHeadingSet();

		final List<EIInstanceMinimal> validInstances = new ArrayList<EIInstanceMinimal>();
		final List<EIInstanceMinimal> stubInstances = new ArrayList<EIInstanceMinimal>();

		int instanceCount = 0;

		for (final EIInstanceMinimal instance : instanceList) {
			if ( performable.isValidFor( instance ) ) {
				validInstances.add( instance );
				log.debug("valid instance " + instance.getInstanceLabel());
				flexTable.setText( instanceCount + 2, 0, instance.getInstanceLabel() );
				flexTable.setText( instanceCount + 2, 1, instance.getInstanceType().getLabel() );
				flexTable.setText( instanceCount + 2, 2, instance.getWFState().getLabel() );
				instanceCount++;
			} else {
				if ( instance.isStub() ) {
					log.debug( "stub instance" );
					stubInstances.add( instance );
				}
			}
		}

		String stubWarningMessage = ( stubInstances.size() > 0 ) ? makeErrorMessage( UIMessages.BULK_OPERATION_ERROR_FOR_STUBS, "$RESOURCES$", stubInstances ).replace( "$ACTION$", name.toLowerCase() ) : null;
		log.debug("stub warning message: " + stubWarningMessage); // TODO: remove
		
		if ( instanceCount == 0 ) {
			if ( stubInstances.size() > 0 ) {
				Window.alert( stubWarningMessage );
			} else {
				Window.alert( errorMessage );
			}
			return;
		}

		if ( validInstances.size() > 0 ) {
			log.debug("valid instances: " + validInstances.size() + "; stubs? " + stubInstances.size()); // TODO: remove
			final EIConfirmationPopup popup = new EIConfirmationPopup( "100%", "100%", stubWarningMessage );
			popup.addConfirmClickHandler( makeHandlerforBulkOperation( callback, performable, validInstances, popup ) );
			vpanel.add( scrPanel );
			scrPanel.add( flexTable );
			popup.add( vpanel );
			popup.show();
		}
	}

	private String makeErrorMessage(String bulkMessage, String toReplace, List<EIInstanceMinimal> stubInstances) {
		log.debug( "making error message from: " + bulkMessage );
		StringBuilder instanceNames = new StringBuilder( stubInstances.get( 0 ).getInstanceLabel() );
		if ( stubInstances.size() > 1 ) {
			for (EIInstanceMinimal minimal : stubInstances.subList( 1, stubInstances.size() )) {
				instanceNames.append( ", " + minimal.getInstanceLabel() );
			}
		}

		return bulkMessage.replace( toReplace, instanceNames );
	}

	protected boolean instanceMeetsFilterCriteria(final EIInstanceMinimal eiInstance) {
		if ( ApplicationState.getInstance().isStrictlyFilteredByOwner() ) {
			return WorkflowUtils.isNotNull( eiInstance.getWFOwner() );
		}
		return true;
	}

	protected List<EIInstanceMinimal> getCheckedInstances() {
		final List<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
		for (final EIInstanceMinimal eiInstance : instances) {
			final GridRowWidget row = instanceRowMap.get( eiInstance );
			if ( row.getCheckBox() != null && row.getCheckBox().isVisible() && row.getCheckBox().getValue() ) {
				instanceList.add( eiInstance );
			}
		}
		return instanceList;
	}

	protected void bulkClaimOrRelease(final ActionRedisplay redisplay, final OwnershipAction claimOrReleaseAction, final List<EIInstanceMinimal> checkedInstances, final BulkWorkflowCallback bulkWorkflowCallback) {
		makeConfirmationDialog( checkedInstances, claimOrReleaseAction.getName(), UIMessages.BULK_CLAIM_RELEASE_ERROR.replace( "$ACTION$", claimOrReleaseAction.getName().toLowerCase() ), bulkWorkflowCallback, claimOrReleaseAction );
	}

	protected BulkWorkflowCallback makeWorkflowCallback(final EIInstanceMinimal[] instances, final ActionRedisplay redisplay) {
		return new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> failures) {
				redisplay.drawAfterPromote( instances, failures );
			}

			@Override
			public void needsRefresh(final String message) {
				redisplay.needsRefresh( message );
			}

			@Override
			public void onFailure(Throwable caught) {
				redisplay.handleFailedTransition( caught.getMessage() );
				
			}
		};
	}

	protected BulkWorkflowCallback makeOwnershipCallback(final EIInstanceMinimal[] instances, final ActionRedisplay redisplay) {
		return new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> failures) {
				redisplay.drawAfterClaim( instances, failures );
			}

			@Override
			public void needsRefresh(final String message) {
				redisplay.needsRefresh( message );
			}

			@Override
			public void onFailure(Throwable caught) {
				redisplay.handleFailedTransition( caught.getMessage() );				
			}
		};
	}

	protected void bulkOperation(final WorkFlowTransition transition, final List<EIInstanceMinimal> selectedInstances, final BulkWorkflowCallback callback) {
		makeConfirmationDialog( selectedInstances, transition.getName(), UIMessages.BULK_CLAIM_RELEASE_ERROR.replace( "$ACTION$", transition.getName().toLowerCase() ), callback, new Performable() {

			@Override
			public void perform(final EIInstanceMinimal[] instances, final BulkWorkflowCallback callback) {
				ClientRepositoryToolsManager.INSTANCE.transition( instances, transition, callback );
			}

			@Override
			public boolean isValidFor(final EIInstanceMinimal instance) {
				return !instance.isStub() && transition.getFromStateURI().equals( instance.getWFStateUri() ) && ClientRepositoryToolsManager.INSTANCE.canEdit( instance );
			}

		} );
	}

	private VerticalPanel makeVerticalPanelWithHeading(final String headingString) {
		final VerticalPanel vpanel = new VerticalPanel();
		final Label heading = new Label( headingString );
		heading.setStyleName( "dataPanelLabel" );
		vpanel.add( heading );
		return vpanel;
	}

	private ClickHandler makeHandlerforBulkOperation(final BulkWorkflowCallback callback, final Performable performable, final List<EIInstanceMinimal> instanceList, final PopupPanel popup) {
		return new ClickHandler() {

			@Override
			public void onClick(final ClickEvent event) {
				performable.perform( instanceList.toArray( new EIInstanceMinimal[instanceList.size()] ), callback );
				popup.hide();
				AbstractEntryPoint.showGlasspane();
			}
		};
	}

	private FlexTable getFlextableWithHeadingSet() {
		final FlexTable flexTable = new FlexTable();
		flexTable.setStyleName( "dataPanel" );
		flexTable.setHTML( 1, 0, "<b>" + "Resource Name" + "</b>" );
		flexTable.setHTML( 1, 1, "<b>" + "Type" + "</b>" );
		flexTable.setHTML( 1, 2, "<b>" + "Status" + "</b>" );
		return flexTable;
	}

	private ScrollPanel getScrollPanelWithDimensionsSet() {
		final ScrollPanel scrPanel = new ScrollPanel();
		scrPanel.setHeight( "270px" );
		scrPanel.setWidth( "1000px" );
		scrPanel.setHorizontalScrollPosition( -1 );
		return scrPanel;
	}

	@Override
	public void onRowDeletion(final EIInstanceMinimal instance) {
		if ( ResourceProvider.isResourceProviderType( instance.getInstanceType() ) ) {
			ApplicationState.getInstance().updateResourceProviderCache();
		}
		removeRow( instance );
	}
}
