package org.eaglei.datatools.client.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.RootAsyncCallback;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.SessionListener;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.ClientModelManager;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassesInGroupCallback;
import org.eaglei.search.provider.AuthSearchRequest;

public class ResourceProviderCache implements SessionListener {

	public interface ResourceProviderCacheListener {

		void onProvidersLoaded();
	}

	private final Map<String, EIInstanceMinimal> providerMap = new HashMap<String, EIInstanceMinimal>();
	private final Map<EIURI, EIInstanceMinimal> providerURIMap = new HashMap<EIURI, EIInstanceMinimal>();
	private boolean loading = true;
	private boolean loadStarted = false;
	private final List<ResourceProviderCacheListener> listeners;

	private static final GWTLogger log = GWTLogger.getLogger( "ResourceProviderCache" );

	public ResourceProviderCache() {
		log.debug( "making new resource provider cache" );
		listeners = new ArrayList<ResourceProviderCacheListener>();
		ClientRepositoryToolsManager.INSTANCE.addSessionListener( this );
	}

	public void update() {
		if ( loadStarted ) {
			return;
		}

		loading = true;
		loadStarted = true;
		log.debug( "resource provider cache getting providers" );

		final AuthSearchRequest request = new AuthSearchRequest();
		request.setType( ResourceProvider.BASE_RESOURCE_CONTAINER_URI );
		request.setPaginated( false ); // TODO: figure out what *other* filtering we need to do, and enable that
		ClientRepositoryToolsManager.INSTANCE.listResources( request, SortByProperties.label, ApplicationState.getInstance().isAcending(), false, false, new RootAsyncCallback<List<EIInstanceMinimal>>() {

			@Override
			public void onSuccess(List<EIInstanceMinimal> result) {
				providerMap.clear();

				for (final EIInstanceMinimal instance : result) {
					providerMap.put( instance.getInstanceLabel(), instance );
					providerURIMap.put( instance.getInstanceURI(), instance );
				}

				loading = false;
				loadStarted = false;
				notifyListeners();
			}

		} );
	}

	public List<String> getResourceProviderLabels() {
		List<String> labels = new ArrayList<String>( providerMap.keySet() );
		Collections.sort( labels ); // TODO: ignore case!

		return labels;
	}

	public List<EIEntity> getResourceProviderEntities() {
		List<EIEntity> providerEntities = new ArrayList<EIEntity>();

		for (final String label : getResourceProviderLabels()) {
			providerEntities.add( providerMap.get( label ).getEntity() );
		}

		return providerEntities;
	}

	public EIEntity getResourceProvider(String providerLabel) {
		if ( !providerMap.containsKey( providerLabel ) ) {
			log.warn( "trying to get nonexistent entry '" + providerLabel + "'" );
		}
		return providerMap.get( providerLabel ).getEntity();
	}

	public EIEntity getResourceProviderType(EIEntity provider) {
		if ( !providerMap.containsKey( provider.getLabel() ) ) {
			log.warn( "trying to get nonexistent entry '" + provider.getLabel() + "'" );
		}
		return providerMap.get( provider.getLabel() ).getInstanceType();
	}

	public EIEntity getResourceProvider(EIURI providerUri) {
		if ( !providerURIMap.containsKey( providerUri ) ) {
			log.warn( "trying to get nonexistent uri '" + providerUri + "'" );
		}
		return providerURIMap.get( providerUri ).getEntity();
	}
	
	public boolean isLoading() {
		return loading;
	}

	public void addProviderLoadListener(ResourceProviderCacheListener listener) {
		listeners.add( listener );
	}

	private void notifyListeners() {
		for (final ResourceProviderCacheListener listener : listeners) {
			listener.onProvidersLoaded();
		}
	}

	@Override
	public void onLogIn(String username, String userUri) {
		update();
	}

	@Override
	public void onLogOut(boolean isSessionExpired) {
		listeners.clear();
	}
}
