package org.eaglei.lexical;

import java.io.Serializable;

import org.eaglei.model.EIURI;

/**
 * Represents an entity extraction request.
 */
public class EntityMatchRequest implements Serializable {

    public static final long serialVersionUID = 1L;

    /**
     * Required field that holds the text to match.
     */
    private String text;
    /**
     * Optional field that holds the URI of the resolved or selected entity
     * from the eagle-i ontology used to constrain the search term. Either
     * query or entity must be specified.
     */
    private EIURI uri;    
    /**
     * Maximum number of matches to return
     */
    private int maxMatches;
    
    /**
     * Creates a request for a specific text string.
     * @param text Text to match.
     */
    public EntityMatchRequest(String text) {
        assert text != null;
        assert !text.isEmpty();
        this.text = text;
    }
    
    /**
     * Creates a request for a specific text string and type URI constraint
     * @param text Text to match.
     * @param uri URI of class that constrains the matches.
     */
    public EntityMatchRequest(String text, EIURI uri) {
        this(text);
        this.uri = uri;
    }
    
    /**
     * Gets the text to match.
     * 
     * @return The text to match. 
     */
    public String getText() {
        return this.text;
    }

    /**
     * Get the URI for the eagle-i ontology entity used to constrain the match.
     * 
     * @return Entity URI. May be null.
     */
    public EIURI getURI() {
        return this.uri;
    }
    
    /**
     * Gets the max number of matches.
     * 
     * @return Maximum matches to return.
     */
    public int getMaxMatches() {
        return this.maxMatches;
    }

    /**
     * Sets the maximum matches to return.
     * 
     * @param maxMatches Maximum number of matches to return.
     */
    public void setMaxMatches(final int maxMatches) {
        assert maxMatches >= 0;
        this.maxMatches = maxMatches;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((text == null) ? 0 : text.hashCode());
        result = prime * result + ((uri == null) ? 0 : uri.hashCode());
        result = prime * result + this.maxMatches;        
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        EntityMatchRequest other = (EntityMatchRequest) obj;
        if (text == null) {
            if (other.text!= null) {
                return false;
            }
        } else if (!text.equals(other.text)) {
            return false;
        }
        if (uri == null) {
            if (other.uri != null) {
                return false;
            }
        } else if (!uri.equals(other.uri)) {
            return false;
        }
        if (maxMatches != other.maxMatches) {
            return false;
        }        
        return true;
    }  
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.text);
        sb.append(" (");
        if (this.uri != null) {
            sb.append("type: ");
            sb.append(uri.toString());
            sb.append(", ");
        }
        sb.append("max matches: ");
        sb.append(this.maxMatches);
        sb.append(")");
        return sb.toString();
    }
}
