package org.eaglei.model;

import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 1, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * Wrapper around an eagle-i resource instance. A resource instance is identified by an instance URI
 * (contained in an EIEntity)is defined as a set of all triples
 * where subject .
 * This wrapper provides a stripped-down version of this set of triples (only the information
 * relevant to the UI)
 */

public class EIInstance extends EIResource implements Serializable
{

    private static final long serialVersionUID = 1L;
	private final String WORKFLOW_STATE = "http://eagle-i.org/ont/repo/1.0/hasWorkflowState";
	private final String CREATED = "http://purl.org/dc/terms/created";


    // the label and URI of this instance
    // to be initialized with data from e.g. a data collection form: instance ID (a URI
    // obtained from repository) and label (from user)

    EIClass instanceClass;

    Map<EIEntity, Set<EIURI>> objectProperties = new HashMap<EIEntity, Set<EIURI>>();
    Map<EIEntity, Set<String>> datatypeProperties = new HashMap<EIEntity, Set<String>>();
    
    // Hold the IDs and values of properties that were not found in the ontology, but were attached to the data in the repo;
   
    Map<EIEntity, Set<EIURI>> nonOntologyResourceProperties = new HashMap<EIEntity, Set<EIURI>>();
    Map<EIEntity, Set<String>> nonOntologyLiteralProperties = new HashMap<EIEntity, Set<String>>();

    private EIInstance() 
    {
        // for GWT
    }
    
    public EIInstance(final EIClass instanceClass, final EIEntity entity)
    {
        super(entity);
        
        assert instanceClass != null;

        this.instanceClass = instanceClass;
    }

    public void addObjectPropertyToInstance(final EIEntity property, final EIURI resource)
    {
    	Set<EIURI> pValues = objectProperties.get(property);
    	if(pValues == null)
    		objectProperties.put(property, pValues= new HashSet<EIURI>());
    	pValues.add(resource);
    }

    public void addDatattypePropertyToInstance(final EIEntity property, final String literal)
    {
    	Set<String> pValues = datatypeProperties.get(property);
    	if(pValues == null) 
    		datatypeProperties.put(property, pValues=new HashSet<String>())	;	

    	pValues.add(literal);
    }
    
    public void addNonOntologyResourceProperty(final EIEntity property, final EIURI resource)
    {
    	Set<EIURI> pValues = nonOntologyResourceProperties.get(property);
    	if(pValues == null)
    		nonOntologyResourceProperties.put(property, pValues=new HashSet<EIURI>());
    	pValues.add(resource);
    }

    public void addNonOntologyLiteralProperty(final EIEntity property, final String literal)
    {
    	Set<String> pValues = nonOntologyLiteralProperties.get(property);
    	if(pValues == null) 
    		nonOntologyLiteralProperties.put(property, pValues = new HashSet<String>());
    	pValues.add(literal);
    }  
    
    
    public void setObjectProperties(Map<EIEntity, Set<EIURI>> objectProperties)
    {
        this.objectProperties = objectProperties;
    }

    public void setDatatypeProperties(Map<EIEntity, Set<String>> datatypeProperties)
    {
        this.datatypeProperties = datatypeProperties;
    }

    // TODO: add alternative versions of setters for the Object/Datatype properties
    
    
    public Map<EIEntity, Set<EIURI>> getObjectProperties()
    {
        return objectProperties;
    }

    public Map<EIEntity, Set<String>> getDatatypeProperties()
    {
        return datatypeProperties;
    }

    public EIURI getInstanceURI() 
    {
        return getEntity().getURI();
    }
    
    public String getInstanceLabel() 
    {
        return getEntity().getLabel();
    }
    
    public EIClass getInstanceClass() 
    {
        return instanceClass;
    }
    


    @Override
    public String toString() {
        return instanceClass.toString() + " : " + getEntity().toString();
    }

	/**
	 * @return the nonOntologyResourceProperties
	 */
	public Map<EIEntity, Set<EIURI>> getNonOntologyResourceProperties() {
		return nonOntologyResourceProperties;
	}

	/**
	 * @param nonOntologyResourceProperties the nonOntologyResourceProperties to set
	 */
	public void setNonOntologyResourceProperties(
			Map<EIEntity, Set<EIURI>> nonOntologyResourceProperties) {
		this.nonOntologyResourceProperties = nonOntologyResourceProperties;
	}

	/**
	 * @return the nonOntologyLiteralProperties
	 */
	public Map<EIEntity, Set<String>> getNonOntologyLiteralProperties() {
		return nonOntologyLiteralProperties;
	}

	/**
	 * @param nonOntologyLiteralProperties the nonOntologyLiteralProperties to set
	 */
	public void setNonOntologyLiteralProperties(
			Map<EIEntity, Set<String>> nonOntologyLiteralProperties) {
		this.nonOntologyLiteralProperties = nonOntologyLiteralProperties;
	}
    
	public void setEIEntity(EIEntity entity){
		setEntity(entity);
	}
	
	public EIURI getWFState() {
		EIEntity key = EIEntity.create(EIURI.create(WORKFLOW_STATE),"");
		if (nonOntologyResourceProperties.containsKey(key)) {
			Set<EIURI> state = nonOntologyResourceProperties.get(key);
			if (!state.isEmpty()) {
				//there should be only one
				assert(state.size() == 1);
				return state.iterator().next();
			}
			else return null;
		} else return null;
	}
	
	public String getCreationDate() {
		EIEntity key = EIEntity.create(EIURI.create(CREATED),"");
		if (nonOntologyLiteralProperties.containsKey(key)) {
			Set<String> created = nonOntologyLiteralProperties.get(key);
			
			if (!created.isEmpty()) {
				//there should be only one
				assert(created.size() == 1);
				return created.iterator().next();
			}
			else return null;
		} else return null;
	}
	
}
