package org.eaglei.model;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 1, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * Wrapper around an eagle-i resource instance. A resource instance is identified by an instance URI
 * (contained in an EIEntity)is defined as a set of all triples
 * where subject .
 * This wrapper provides a stripped-down version of this set of triples (only the information
 * relevant to the UI)
 * Note that the instance class has to be an EIClass; most EIInstances will have only one; in some cases
 * there may be necessary to include more than one; they'll
 * be contained in the List otherEIClasses.
 * Other non-eagle-i types are kept in nonOntologyObject properties
 */

public class EIInstance extends EIResource implements Serializable, Comparable
{

	private static final long serialVersionUID = 1L;

	//Use our own labels for now
	private static final EIEntity WFStateEntity = EIEntity.create(
			EIURI.create("http://eagle-i.org/ont/repo/1.0/hasWorkflowState"), 
	"Workflow state");
	private static final EIEntity WFOwnerEntity = EIEntity.create(
			EIURI.create("http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner"), 
	"Workflow owner");
	private static final EIEntity CreatedEntity = EIEntity.create(
			EIURI.create("http://purl.org/dc/terms/created"), "Creation date");

	//quick access to the eiClass
	//Most of the time there will only be one
	private EIEntity instanceType;
	private EIClass instanceClass;

	//Some maps are initialized here because GWT complains otherwise
	
	//Hold additional types for the instance
	private List<EIEntity> otherEITypes;

	//Hold ontology properties
	private Map<EIEntity, Set<EIEntity>> objectProperties = new HashMap<EIEntity, Set<EIEntity>>();
	private Map<EIEntity, Set<String>> datatypeProperties = new HashMap<EIEntity, Set<String>>();

	// Hold properties that were not found in the ontology, but were attached to the data in the repo;
	private Map<EIEntity, Set<EIEntity>> nonOntologyResourceProperties = new HashMap<EIEntity, Set<EIEntity>>();
	private Map<EIEntity, Set<String>> nonOntologyLiteralProperties = new HashMap<EIEntity, Set<String>>();

	private Map<EIEntity, EIEntity> readOnlyResourceProperties;
	private Map<EIEntity, String> readOnlyLiteralProperties;

	// Hold the ranges of object properties - to speed up usage of EIInstance at client side
	
	//FIME see if this is still needed
	private Map<EIEntity, List<EIClass>> ranges = new HashMap<EIEntity, List<EIClass>>();

	private EIInstance() 
	{
		// for GWT
	}

	//constructor with type
	private EIInstance(final EIEntity instanceType, final EIEntity instanceEntity)
	{
		super(instanceEntity);
		this.instanceType = instanceType;
		//instance type can be null
	}    

	public static EIInstance create(final EIEntity instanceType, final EIEntity instanceEntity) {
		return new EIInstance(instanceType, instanceEntity);
	}

	public EIClass getInstanceClass() {
		return instanceClass;
	}

	public void setInstanceClass(EIClass instanceClass) {
		this.instanceClass = instanceClass;
	}

	public EIURI getInstanceURI() 
	{
		return getEntity().getURI();
	}

	public String getInstanceLabel() 
	{
		return getEntity().getLabel();
	}

	public EIEntity getInstanceType() 
	{
		return instanceType;
	}

	public void setInstanceType(EIEntity instanceType) {
		this.instanceType = instanceType;
	}

	public EIEntity getWFState() {
		if(readOnlyResourceProperties == null)
			return null;
		if (readOnlyResourceProperties.containsKey(WFStateEntity)) {
			return readOnlyResourceProperties.get(WFStateEntity);
		}
		else return null;
	}

	public void setWFState(EIEntity state) {
		setReadOnlyResourceProperty(WFStateEntity, state);
	}

	public EIEntity getWFOwner() {
		if(readOnlyResourceProperties == null)
			return null;
		if (readOnlyResourceProperties.containsKey(WFOwnerEntity)) {
			return readOnlyResourceProperties.get(WFOwnerEntity);
		} else return null;
	}

	public void setWFOwner(EIEntity ownerUri) {
		setReadOnlyResourceProperty(WFOwnerEntity, ownerUri);
	}

	public String getCreationDate() {
		if(readOnlyLiteralProperties == null)
			return null;
		if (readOnlyLiteralProperties.containsKey(CreatedEntity)) {
			String created = readOnlyLiteralProperties.get(CreatedEntity);
			return created;
		} else return null;
	}

	public void setCreationDate(String date) {
		setReadOnlyLiteralProperty(CreatedEntity, date);
	}

	//Methods for ranges
	@Deprecated
	public void setRangeList(EIEntity entity, List<EIClass> rangeList) {
		ranges.put(entity, rangeList);
	}

	@Deprecated
	public List<EIClass> getRangeList(EIEntity entity) {
		return ranges.get(entity);
	}

	@Deprecated
	public Map<EIEntity, List<EIClass>> getAllRanges() {
		return ranges;
	}

	//Methods for object properties
	//map initialized at declaration


	public Map<EIEntity, Set<EIEntity>> getObjectProperties()
	{
		return objectProperties;    	
	}


	public void addObjectProperty(final EIEntity property, final EIEntity value)
	{
		Set<EIEntity> pValues = objectProperties.get(property);
		if(pValues == null) 
			objectProperties.put(property, pValues = new HashSet<EIEntity>());
		pValues.add(value);	
	}

	public Set<EIEntity> getObjectProperty(final EIEntity property) {
		return objectProperties.get(property);
	}


	public void replaceObjectPropertyValue(final EIEntity property, final EIURI oldValue, final EIEntity newValue) {
		Set<EIEntity> pValues = objectProperties.get(property);
		if(pValues == null)
			objectProperties.put(property, pValues = new HashSet<EIEntity>());
		pValues.remove(EIEntity.create(oldValue, ""));
		if (newValue != null && newValue != EIEntity.NULL_ENTITY)
			pValues.add(newValue);	
	}

	public void replaceObjectPropertyAllValues(final EIEntity property, final Set<EIEntity> values) {
		objectProperties.put(property, values);
	}


	//Datatype property methods
	//map initialized at declaration

	public Map<EIEntity, Set<String>> getDatatypeProperties()
	{
		return datatypeProperties;
	}

	public void addDatattypeProperty(final EIEntity property, final String value)
	{
		Set<String> pValues;	
		pValues = datatypeProperties.get(property);
		if(pValues == null) 
			datatypeProperties.put(property, pValues = new HashSet<String>());
		pValues.add(value);
	}

	public Set<String> getDatatypeProperty(final EIEntity property) {
		return datatypeProperties.get(property);
	}

	public void replaceDatatypePropertyValue(final EIEntity property, final String oldValue, final String newValue) {
		Set<String> pValues = datatypeProperties.get(property);
		if(pValues == null)
			datatypeProperties.put(property, pValues = new HashSet<String>());
		pValues.remove(oldValue);
		if (newValue != null && !newValue.equals(""))
			pValues.add(newValue);	
	}

	public void replaceDatatypePropertyAllValues(final EIEntity property, final Set<String> values) {
		datatypeProperties.put(property, values);
	}

	//Non ontology resource property methods
	//Map initialized at declaration


	public Map<EIEntity, Set<EIEntity>> getNonOntologyResourceProperties() {
		return nonOntologyResourceProperties;
	}



	public void addNonOntologyResourceProperty(final EIEntity property, final EIEntity resource)
	{
		Set<EIEntity> pValues;
		pValues = nonOntologyResourceProperties.get(property);
		if(pValues == null)
			nonOntologyResourceProperties.put(property, pValues=new HashSet<EIEntity>());
		pValues.add(resource);
	}

	public Set<EIEntity> getNonOntologyResourceProperty(final EIEntity property) {
		return nonOntologyResourceProperties.get(property);
	}

	public void replaceNonOntologyResourcePropertyValue(final EIEntity property, final EIURI oldValue, final EIEntity newValue)
	{
		Set<EIEntity> pValues;
		pValues = nonOntologyResourceProperties.get(property);
		if(pValues == null)
			nonOntologyResourceProperties.put(property, pValues = new HashSet<EIEntity>());
		pValues.remove(EIEntity.create(oldValue, ""));
		if (newValue != null && newValue != EIEntity.NULL_ENTITY)
			pValues.add(newValue);
	}

	//FIXME These names are getting quite ridiculous
	public void replaceNonOntologyResourcePropertyAllValues(final EIEntity property, final Set<EIEntity> values) {
		nonOntologyResourceProperties.put(property, values);
	}


	//Non ontology Literal property methods
	//Map initialized at declaration


	public Map<EIEntity, Set<String>> getNonOntologyLiteralProperties() {
		return nonOntologyLiteralProperties;
	}


	public void addNonOntologyLiteralProperty(final EIEntity property, final String literal)
	{
		Set<String> pValues = nonOntologyLiteralProperties.get(property);
		if(pValues == null)
			nonOntologyLiteralProperties.put(property, pValues = new HashSet<String>());
		pValues.add(literal);
	}

	public Set<String> getNonOntologyLiteralProperty(final EIEntity property) {
		return nonOntologyLiteralProperties.get(property);
	}

	public void replaceNonOntologyLiteralProperty(final EIEntity property, final String oldValue, final String newValue) {
		Set<String> pValues = nonOntologyLiteralProperties.get(property);
		if(pValues == null)
			nonOntologyLiteralProperties.put(property, pValues = new HashSet<String>());
		pValues.remove(oldValue);
		if (newValue != null && !newValue.equals(""))
			pValues.add(newValue);	
	}

	public void replaceNonOntologyLiteralPropertyAllValues(final EIEntity property, final Set<String> values) {
		datatypeProperties.put(property, values);
	} 


	//Methods for other EITypes
	//Note that these are rarely used (typically instances in eagle-i have only one asserted type)
	public void addEIType(final EIEntity eiType) {
		if(eiType == null) return;
		if(otherEITypes == null) {
			otherEITypes = new ArrayList<EIEntity>();
		}
		otherEITypes.add(eiType);
	}

	public void setOtherEITypes(List<EIEntity> eiTypes) {
		this.otherEITypes = eiTypes;
	}

	public List<EIEntity> getOtherEITypes() {
		if(otherEITypes == null)
			return Collections.emptyList();
		else
			return otherEITypes;
	}


	//Methods for readOnly properties

	public Map<EIEntity, EIEntity> getReadOnlyResourceProperties() {
		if(readOnlyResourceProperties == null)
			return Collections.emptyMap();
		else
			return readOnlyResourceProperties;
	}

	public void setReadOnlyResourceProperty(EIEntity property, EIEntity value) {
		if(readOnlyResourceProperties == null)
			readOnlyResourceProperties = new HashMap<EIEntity, EIEntity>();
		readOnlyResourceProperties.put(property, value);
	}

	public Map<EIEntity, String> getReadOnlyLiteralProperties() {
		if(readOnlyLiteralProperties == null)
			return Collections.emptyMap();
		else
			return readOnlyLiteralProperties;
	}

	public void setReadOnlyLiteralProperties(
			Map<EIEntity, String> readOnlyLiteralProperties) {
		this.readOnlyLiteralProperties = readOnlyLiteralProperties;
	}

	public void setReadOnlyLiteralProperty(EIEntity property, String value) {
		if(readOnlyLiteralProperties == null)
			readOnlyLiteralProperties = new HashMap<EIEntity, String>();
		readOnlyLiteralProperties.put(property, value);
	}

	@Override
	public int compareTo(Object o) {
		if(!(o instanceof EIInstance))
			return 0;
		return this.getEntity().getLabel().compareToIgnoreCase(((EIInstance)o).getEntity().getLabel());
	}

	@Override
	public String toString() {
		//Display with the first asserted class
		return instanceType.toString() + " : " + getEntity().toString();
	}
}
