package org.eaglei.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 1, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 *       Wrapper around an eagle-i resource instance. A resource instance is identified by an instance URI (contained in an EIEntity)is defined as a set of all triples where subject . This wrapper provides a stripped-down version of this set of
 *       triples (only the information relevant to the UI) Note that the instance class has to be an EIClass; most EIInstances will have only one; in some cases there may be necessary to include more than one; they'll be contained in the List
 *       otherEIClasses. Other non-eagle-i types are kept in nonOntologyObject properties
 */

public class EIInstanceMinimal extends EIResource implements Serializable, Comparable {

	private static final long serialVersionUID = 1L;

	// quick access to the eiClass
	// Most of the time there will only be one
	private EIEntity instanceType;
	// There may be more than one ei types for the instance
	private List<EIEntity> otherEITypes;
	private String created;
	private String modified;
	private EIEntity owner;
	private EIEntity WFState;
	private EIEntity lab;
	
	// Cache whether all required properties have values
	private boolean instanceHasAllRequiredProperties;

	private EIInstanceMinimal() {
		// for GWT
	}

	// constructor with type
	private EIInstanceMinimal(final EIEntity instanceType, final EIEntity instanceEntity) {
		super( instanceEntity );
		this.instanceType = instanceType;
		// instance type can be null
	}

	private EIInstanceMinimal(final EIInstance instance) {
		super( instance.getEntity() );
		instanceType = instance.getInstanceType();
		created = instance.getCreationDate();
		modified = instance.getModificationDate();
		// skip for now - we're not using it
		// this.otherEITypes;
		owner = isNull( instance.getWFOwner() ) ? EIEntity.NULL_ENTITY : instance.getWFOwner();
		WFState = isNull( instance.getWFState() ) ? EIEntity.NULL_ENTITY : instance.getWFState();
	}

	public static EIInstanceMinimal create(final EIEntity instanceType, final EIEntity instanceEntity) {
		return new EIInstanceMinimal( instanceType, instanceEntity );
	}

	public static EIInstanceMinimal create(final EIInstance instance) {
		return new EIInstanceMinimal( instance );
	}

	public void addEIType(final EIEntity eiType) {
		if ( eiType == null ) {
			return;
		}
		if ( otherEITypes == null ) {
			otherEITypes = new ArrayList<EIEntity>();
		}
		otherEITypes.add( eiType );
	}

	public void setOtherEITypes(final List<EIEntity> eiTypes) {
		otherEITypes = eiTypes;
	}

	public List<EIEntity> getOtherEITypes() {
		if ( otherEITypes == null ) {
			return Collections.emptyList();
		} else {
			return otherEITypes;
		}
	}

	public EIURI getInstanceURI() {
		return getEntity().getURI();
	}

	public String getInstanceLabel() {
		return getEntity().getLabel();
	}

	public EIEntity getInstanceType() {
		return instanceType;
	}

	public void setInstanceType(final EIEntity instanceType) {
		this.instanceType = instanceType;
	}

	@Override
	public String toString() {
		// Display with the first asserted class
		return instanceType.toString() + " : " + getEntity().toString();
	}

	public EIURI getWFStateUri() {
		return WFState.getURI();
	}

	public EIEntity getWFState() {
		return WFState;
	}

	public void setWFState(final EIEntity state) {
		WFState = state;
	}

	public EIURI getWFOwnerUri() {
		return owner.getURI();
	}

	public EIEntity getWFOwner() {
		return owner;
	}

	public void setWFOwner(final EIEntity ownerUri) {
		owner = ownerUri;
	}

	public String getCreationDate() {
		return created;
	}

	public void setCreationDate(final String date) {
		created = date;
	}

	public String getModifiedDate() {
		return modified;
	}

	public void setModifiedDate(final String date) {
		modified = date;
	}

	public EIEntity getLab() {
		return lab;
	}

	public void setLab(final EIEntity lab) {
		this.lab = lab;
	}
	
	public void setHasAllRequiredProperties(boolean instanceHasAllRequiredProperties) {
		this.instanceHasAllRequiredProperties = instanceHasAllRequiredProperties;
	}

	public boolean hasAllRequiredProperties() {
		return instanceHasAllRequiredProperties;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(final Object o) {
		if ( !( o instanceof EIInstanceMinimal ) ) {
			return 0;
		}
		return getEntity().getLabel().compareToIgnoreCase( ( (EIInstanceMinimal)o ).getEntity().getLabel() );
	}

	private boolean isNull(final EIEntity entity) {
		return entity == null || entity.equals( EIEntity.NULL_ENTITY );
	}
}
