/**
 * The eagle-i consortium
 * Harvard University
 * Jan 6, 2011
 */
package org.eaglei.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Daniela Bourges-Waldegg
 * Container for eagle-i resource instances.
 * Objects of this class represent a full instance of an eagle-i ontology class, 
 * with embedded instances (hasEmbeddedInstances = true) and optionally, additional 
 * metadata and non-onntology properties (isExtendedInstance = true)
 * The container contains:
 * - an EIBasicInstance to hold the properties of the main instance
 * - a NonOntologyProperties object to hold metadata and anything that was not recognized as conforming to 
 * the eagle-i ontology when reading repository data
 * - a map of embedded instances (EIInstance objects)
 * EIInstance objects are created only with minimal information (uri, type and label), other properties are set by an EIInstanceFactory
 * (when creating from repository data) or by repository client code (e.g. when creating/updating from user input)
 *
 */
public class EIInstance implements Serializable {

	public static EIInstance NULL_INSTANCE = new EIInstance( EIClass.NULL_CLASS, EIEntity.NULL_ENTITY );
	private static final long serialVersionUID = 1L;
	
	private EIBasicInstance mainInstance;
	private NonOntologyProperties extendedInstanceProperties;
	private Map<EIEntity, EIInstance> embeddedInstances;
	private Set<EIEntity> stubs;
	
	private ReferencingResources referencingResources;
	
	private EIInstance() {
		//For GWT
	}
	@Deprecated
	//keeping this for tests for now
	public static EIInstance createEmptyInstance( EIEntity instanceType, EIEntity instanceEntity ) {
		return new EIInstance(instanceType, instanceEntity);
	}

	public static EIInstance createEmptyInstance(  EIClass instanceClass, EIEntity instanceEntity ) {
		return new EIInstance( instanceClass, instanceEntity );
	}
	
	@Deprecated
	//keeping this for tests for now
	private EIInstance(EIEntity instanceType, EIEntity instanceEntity) {
		this(instanceType, instanceEntity, false);
	}

	private EIInstance( EIClass instanceClass, EIEntity instanceEntity ) {
		mainInstance = EIBasicInstance.createEmptyBasicInstance( instanceEntity );
		mainInstance.setInstanceClass(instanceClass);
	}
	
	@Deprecated
	//keeping this for tests for now
	private EIInstance(EIEntity instanceType, EIEntity instanceEntity, boolean isExtended) {
		mainInstance = EIBasicInstance.createEmptyBasicInstance(instanceEntity);
		mainInstance.setInstanceType(instanceType);
	}
	
	public EIEntity getEntity() {
		return mainInstance.getEntity();
	}
	public EIClass getInstanceClass() {
		return mainInstance.getInstanceClass();
	}

	public void setInstanceClass(final EIClass instanceClass) {
		mainInstance.setInstanceClass(instanceClass);
	}

	public EIURI getInstanceURI() {
		return mainInstance.getInstanceURI();
	}

	public void setInstanceLabel(String instanceLabel) {
		mainInstance.setInstanceLabel(instanceLabel);
	}
	
	public String getInstanceLabel() {
		return mainInstance.getInstanceLabel();
	}

	public EIEntity getInstanceType() {
		return mainInstance.getInstanceType();
	}

	public void setInstanceType(final EIEntity instanceType) {
		mainInstance.setInstanceType(instanceType);
	}

	public Map<EIEntity, Set<EIEntity>> getObjectProperties() {
		return mainInstance.getObjectProperties();
	}

	public void addObjectProperty(final EIEntity property, final EIEntity value) {
		mainInstance.addObjectProperty( property, value );
	}

	public Set<EIEntity> getObjectProperty(final EIEntity property) {
		return mainInstance.getObjectProperty( property );
	}

	public void replaceObjectPropertyValue(final EIEntity property, final EIURI oldValue, final EIEntity newValue) {
		mainInstance.replaceObjectPropertyValue( property, oldValue, newValue );
	}

	public void replaceObjectPropertyAllValues(final EIEntity property, final Set<EIEntity> values) {
		mainInstance.replaceObjectPropertyAllValues( property, values );
	}

	public Map<EIEntity, Set<String>> getDatatypeProperties() {
		return mainInstance.getDatatypeProperties();
	}

	public void addDatattypeProperty(final EIEntity property, final String value) {
		mainInstance.addDatattypeProperty( property, value );
	}

	public Set<String> getDatatypeProperty(final EIEntity property) {
		return mainInstance.getDatatypeProperty( property );
	}

	public void replaceDatatypePropertyValue(final EIEntity property, final String oldValue, final String newValue) {
		mainInstance.replaceDatatypePropertyValue( property, oldValue, newValue );
	}

	public void replaceDatatypePropertyAllValues(final EIEntity property, final Set<String> values) {
		mainInstance.replaceDatatypePropertyAllValues( property, values );
	}

	public void addEIType(final EIEntity eiType) {
		mainInstance.addEIType( eiType );
	}

	public void setOtherEITypes(final List<EIEntity> eiTypes) {
		mainInstance.setOtherEITypes( eiTypes );
	}

	public List<EIEntity> getOtherEITypes() {
		return mainInstance.getOtherEITypes();
	}

	//FIXME - is this it? is there more?
	public int compareTo(final Object o) {
		return mainInstance.compareTo( o );
	}

	@Override
	public String toString() {
		return mainInstance.toString();
	}

	public void setHasAllRequiredProperties(boolean instanceHasAllRequiredProperties) {
		mainInstance.setHasAllRequiredProperties( instanceHasAllRequiredProperties );
	}

	public boolean hasAllRequiredProperties() {
		return mainInstance.hasAllRequiredProperties();
	}

	//FIXME return EIEntity.NULL_ENTITY
	public EIEntity getWFState() {
		if(isExtendedInstance()) {
			return extendedInstanceProperties.getWFState();
		}
		else {
			return null;
		}
	}

	public void setWFState(final EIEntity state) {
		extendInstance();
		extendedInstanceProperties.setWFState( state );
	}

	public EIEntity getWFOwner() {
		if(isExtendedInstance()) {
			return extendedInstanceProperties.getWFOwner();
		} else {
			return null; //FIXME
		}
	}

	public void setWFOwner(final EIEntity ownerUri) {
		extendInstance();
		extendedInstanceProperties.setWFOwner( ownerUri );
	}

	public String getCreationDate() {
		if(isExtendedInstance() ) {
			return extendedInstanceProperties.getCreationDate();
		} else {
			return null;
		}
	}

	public void setCreationDate(final String date) {
		extendInstance();
		extendedInstanceProperties.setCreationDate( date );
	}

	public String getModificationDate() {
		if( isExtendedInstance() ) {
			return extendedInstanceProperties.getModificationDate();
		} else {
			return null;
		}
	}

	public void setModificationDate(final String date) {
		extendInstance();
		extendedInstanceProperties.setModificationDate( date );
	}

	public Map<EIEntity, Set<EIEntity>> getNonOntologyResourceProperties() {
		if( isExtendedInstance() ){
			return extendedInstanceProperties.getNonOntologyResourceProperties();
		} else {
			return Collections.emptyMap();
		}
	}

	public void addNonOntologyResourceProperty(final EIEntity property, final EIEntity resource) {
		extendInstance();
		extendedInstanceProperties.addNonOntologyResourceProperty( property, resource );
	}

	public Set<EIEntity> getNonOntologyResourceProperty(final EIEntity property) {
		if( isExtendedInstance() ) {
			return extendedInstanceProperties.getNonOntologyResourceProperty( property );
		} else {
			return Collections.emptySet();
		}
	}

	public void replaceNonOntologyResourcePropertyValue(final EIEntity property, final EIURI oldValue, final EIEntity newValue) {
		extendInstance();
		extendedInstanceProperties.replaceNonOntologyResourcePropertyValue( property, oldValue, newValue );
	}

	public void replaceNonOntologyResourcePropertyAllValues(final EIEntity property, final Set<EIEntity> values) {
		extendInstance();
		extendedInstanceProperties.replaceNonOntologyResourcePropertyAllValues( property, values );
	}

	public Map<EIEntity, Set<String>> getNonOntologyLiteralProperties() {
		if( isExtendedInstance() ) {
			return extendedInstanceProperties.getNonOntologyLiteralProperties();
		} else {
			return Collections.emptyMap();
		}
	}

	public void addNonOntologyLiteralProperty(final EIEntity property, final String literal) {
		extendInstance();
		extendedInstanceProperties.addNonOntologyLiteralProperty( property, literal );
	}

	public Set<String> getNonOntologyLiteralProperty(final EIEntity property) {
		if(isExtendedInstance() ) {
			return extendedInstanceProperties.getNonOntologyLiteralProperty( property );
		} else {
			return Collections.emptySet();
		}
	}

	public void replaceNonOntologyLiteralProperty(final EIEntity property, final String oldValue, final String newValue) {
		extendInstance();
		extendedInstanceProperties.replaceNonOntologyLiteralProperty( property, oldValue, newValue );
	}

	public void replaceNonOntologyLiteralPropertyAllValues(final EIEntity property, final Set<String> values) {
		extendInstance();
		extendedInstanceProperties.replaceNonOntologyLiteralPropertyAllValues( property, values );
	}

	public Map<EIEntity, EIEntity> getReadOnlyResourceProperties() {
		if( isExtendedInstance() ){
			return extendedInstanceProperties.getReadOnlyResourceProperties();
		} else {
			return Collections.emptyMap();
		}
	}

	public void setReadOnlyResourceProperty(final EIEntity property, final EIEntity value) {
		extendInstance();
		extendedInstanceProperties.setReadOnlyResourceProperty( property, value );
	}

	public Map<EIEntity, String> getReadOnlyLiteralProperties() {
		if ( isExtendedInstance() ) {
			return extendedInstanceProperties.getReadOnlyLiteralProperties();
		} else {
			return Collections.emptyMap();
		}
	}

	public void setReadOnlyLiteralProperties(final Map<EIEntity, String> readOnlyLiteralProperties) {
		extendInstance();
		extendedInstanceProperties.setReadOnlyLiteralProperties( readOnlyLiteralProperties );
	}

	public void setReadOnlyLiteralProperty(final EIEntity property, final String value) {
		extendInstance();
		extendedInstanceProperties.setReadOnlyLiteralProperty( property, value );
	}
	
	public void addEmbeddedInstance( EIEntity property, EIInstance instance ) {
		if( instance == null || EIInstance.NULL_INSTANCE.equals(  instance  ) ) {
			return;
		}
		if( embeddedInstances == null) {
			embeddedInstances = new HashMap<EIEntity, EIInstance>();
		}
		
		
		//if the embedded instance is empty, don't add it; use label to determine
		//WRONG we should be able to temporarily hold empty instances
		//if(instance.getInstanceLabel().length() > 0) {
			addObjectProperty( property, instance.getEntity() );
			instance.setEmbedded( true );
			embeddedInstances.put(instance.getEntity(), instance);
		//}
	}
	
	public void removeEmbeddedInstance( EIEntity instanceEntity ) {
		if(hasEmbeddedInstances()) {
			embeddedInstances.remove( instanceEntity );
			//remove the property from the objectPropertyMap
			final EIEntity property = findPropertyForEmbeddedInstance(instanceEntity);
			if( !EIEntity.NULL_ENTITY.equals( ( property ) )) {
				replaceObjectPropertyValue( property, instanceEntity.getURI(), EIEntity.NULL_ENTITY );
			}
		}
	}
	
	private EIEntity findPropertyForEmbeddedInstance(EIEntity instanceEntity) {
		for(Map.Entry<EIEntity, Set<EIEntity>> entry : getObjectProperties().entrySet()) {
			if( entry.getValue().contains( instanceEntity)) {
				return entry.getKey();
			}
		}
		return EIEntity.NULL_ENTITY;
	}
	
	public EIEntity findPropertyForInstance(EIEntity instanceEntity) {
		return mainInstance.findPropertyForInstance( instanceEntity );
	}
	
	public EIInstance getEmbeddedInstance( EIEntity instanceEntity ) {
		if( instanceEntity == null  || instanceEntity == EIEntity.NULL_ENTITY || !hasEmbeddedInstances()) {
			return EIInstance.NULL_INSTANCE;
		}
		return embeddedInstances.containsKey(instanceEntity) ? embeddedInstances.get(instanceEntity) : EIInstance.NULL_INSTANCE;
	}
	
	public Map<EIEntity, EIInstance> getEmbeddedInstances() {
		if ( hasEmbeddedInstances()) {
			return embeddedInstances; 
		} else {
			return Collections.emptyMap();
		}
	}
	
	public List<EIInstance> getEmbeddedInstanceList() {
		if(hasEmbeddedInstances()) {
			return new ArrayList<EIInstance>(embeddedInstances.values());
		} else {
			return Collections.emptyList();
		}
	}

	public List<EIURI> getEmbeddedInstanceUriList() {
		if(hasEmbeddedInstances()) {
			List<EIURI> uris = new ArrayList<EIURI>();
			for( EIEntity entity : embeddedInstances.keySet()) {
				uris.add(entity.getURI());
			}
			return uris;
		} else {
			return Collections.emptyList();
		}
	}

	public boolean hasEmbeddedInstances() {
		return embeddedInstances != null && !embeddedInstances.isEmpty();
	}


	public boolean isExtendedInstance() {
		return extendedInstanceProperties != null;
	}
	
	private void extendInstance() {
		if (extendedInstanceProperties == null) {
			extendedInstanceProperties = new NonOntologyProperties();
		}
	}
	
	public boolean isEmbeddedInstance() {
		return mainInstance.isEmbedded();
	}
	
	public void setEmbedded(boolean isEmbedded) {
		mainInstance.setEmbedded( isEmbedded );
	}
	
	public EIClass getRootSuperType() {
		return mainInstance.getRootSuperType();
	}

	public void setRootSuperType(EIClass rootSuperType) {
		mainInstance.setRootSuperType( rootSuperType );
	}

	public boolean isStub() {
		if( !isExtendedInstance() ) {
			return false;
		}
		final String stubValue = extendedInstanceProperties.getStubPropertyValue();
		if(stubValue == null) {
			return false;
		}
		return stubValue.equalsIgnoreCase( "True" );
	}
	
	/**
	 * Returns true if the property points to an embedded instance.
	 * Note that a property with multiple values will contain either no embedded instances, 
	 * or all its values will be embedded instances
	 * (i.e. we assume embedded classes are not part of a multi range range specification)
	 * @param propertyEntity
	 * @return
	 */
	public boolean isPropertyValueAnEmbeddedInstance( EIEntity propertyEntity) {
		if(propertyEntity == null || EIEntity.NULL_ENTITY.equals( propertyEntity ) || !hasEmbeddedInstances() ) {
			return false;
		}
		//ASSUMPTION: embedded instances cannot have embedded instances
		if(isEmbeddedInstance() ){
			return false;
		}
		final Set<EIEntity> propertyValues  = getObjectProperty( propertyEntity );
		if(propertyValues==null || propertyValues.isEmpty())
			return false;

		//get first property value and use it to check
		//ASSUMPTION: this assumes all values in set are embedded or not (embedded classes are not part of a multirange property)
		//TODO verify if assumption holds
		final EIEntity propertyValue = propertyValues.iterator().next();
		if ( propertyValue == null || EIEntity.NULL_ENTITY.equals(propertyValue)) {
			return false;
		}
			
		return embeddedInstances.containsKey( propertyValue );
	}

	//TODO do a bit more validation (e.g a stub doesn't have stubs)
	public void setStubs(Set<EIEntity> stubs) {
		this.stubs = stubs;
	}
	
	
	public Set<EIEntity> getStubs() {
		if(stubs != null) {
			return this.stubs;
		} else {
			return Collections.emptySet();
		}
	}
	
	public boolean isPropertyValueAStub(EIEntity propertyEntity, EIEntity propertyValue){
		if(propertyEntity == null || EIEntity.NULL_ENTITY.equals( propertyEntity ) 
				|| propertyValue == null || EIEntity.NULL_ENTITY.equals( propertyValue )
				|| !hasStubs() ) {
			return false;
		}
		if(getObjectProperty( propertyEntity ) == null || !getObjectProperty( propertyEntity ).contains( propertyValue ) ) {
			return false;
		}
		return stubs.contains( propertyValue );
	}
	
	public boolean hasStubs() {
		return stubs!= null && !stubs.isEmpty();
	}
	
	public boolean hasReferencingResources() {
		return referencingResources != null;
	}
	

	public void addMaterializedInverseProperty(EIEntity property, EIInstanceMinimal instance) {
		if(referencingResources == null) {
			referencingResources = new ReferencingResources();
		}
		referencingResources.addMaterializedInverseProperty( property, instance );
		//we only want the inverse property in one place
		mainInstance.replaceObjectPropertyValue( property, instance.getInstanceURI(), EIEntity.NULL_ENTITY );
	}
	
	public Map<EIEntity, List<EIInstanceMinimal>> getMaterializedInverseProperties() {
		if(referencingResources == null) {
			return Collections.emptyMap();
		}
		return referencingResources.getMaterializedInverseProperties();
	}
	
	public void setReferencedByList(List<EIInstanceMinimal> resources) {
		if(referencingResources == null) {
			referencingResources = new ReferencingResources();
		}
		referencingResources.setReferencedByList( resources );
	}
	
	public List<EIInstanceMinimal> getReferencedByList() {
		if(referencingResources == null) {
			return Collections.emptyList();
		}
		return referencingResources.getReferencedByList();
	}
}