/**
 * The eagle-i consortium
 * Harvard University
 * Jan 7, 2011
 */
package org.eaglei.model;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Daniela Bourges-Waldegg
 *
 * Objects of this class represent a bare-bones instance of an eagle-i ontology class:
 * - an instance uri, a type and a label
 * - object properties and their values (URIs)
 * - datatype properties and their values
 * These objects do not contain repository metadata or any other property outside of the EI ontology.
 * EIBasicInstance objects are always contained in an EIInstance object, either as the main instance or as embedded instances.
 * They are created only with minimal information (uri and label), other properties are set via their container EIInstance
 *
 */
public class EIBasicInstance extends EIResource implements Serializable {
	private static final long serialVersionUID = 1L;
	private EIEntity instanceType;
	private EIClass instanceClass;
	private List<EIEntity> otherEITypes;
	private EIClass rootSuperType;
	private Map<EIEntity, Set<EIEntity>> objectProperties = new HashMap<EIEntity, Set<EIEntity>>();
	private Map<EIEntity, Set<String>> datatypeProperties = new HashMap<EIEntity, Set<String>>();
	private boolean instanceHasAllRequiredProperties;
	private boolean isEmbedded;
	//Minimal basic instance does not require a type/class
	public static EIBasicInstance createEmptyBasicInstance( EIEntity instanceEntity ) {
		return new EIBasicInstance( instanceEntity );
	}
	
	private EIBasicInstance() {
		//for GWT
	}
	
	private EIBasicInstance(EIEntity instanceEntity) {
		super( instanceEntity );
	}
	
	public EIClass getInstanceClass() {
		return instanceClass;
	}

	public void setInstanceClass(final EIClass instanceClass) {
		this.instanceClass = instanceClass;
		instanceType = instanceClass.getEntity();
	}

	public EIURI getInstanceURI() {
		return getEntity().getURI();
	}

	public String getInstanceLabel() {
		return getEntity().getLabel();
	}
	
	public void setInstanceLabel(String instanceLabel) {
		 setLabel(instanceLabel);
	}

	public EIEntity getInstanceType() {
		return instanceType;
	}

	public void setInstanceType(final EIEntity instanceType) {
		this.instanceType = instanceType;
	}

	public Map<EIEntity, Set<EIEntity>> getObjectProperties() {
		return objectProperties;
	}

	public void addObjectProperty(final EIEntity property, final EIEntity value) {
		Set<EIEntity> pValues = objectProperties.get( property );
		if ( pValues == null ) {
			pValues = new HashSet<EIEntity>();
			objectProperties.put( property, pValues );
		}
		pValues.add( value );
	}

	public Set<EIEntity> getObjectProperty(final EIEntity property) {
		return objectProperties.get( property );
	}

	public void replaceObjectPropertyValue(final EIEntity property, final EIURI oldValue, final EIEntity newValue) {
		Set<EIEntity> pValues = objectProperties.get( property );
		if ( pValues == null ) {
			pValues = new HashSet<EIEntity>();
			objectProperties.put( property, pValues );
		}
		pValues.remove( EIEntity.create( oldValue, "" ) );
		if ( newValue != null && newValue != EIEntity.NULL_ENTITY ) {
			pValues.add( newValue );
		}
	}

	public void replaceObjectPropertyAllValues(final EIEntity property, final Set<EIEntity> values) {
		objectProperties.put( property, values );
	}

	public Map<EIEntity, Set<String>> getDatatypeProperties() {
		return datatypeProperties;
	}

	public void addDatattypeProperty(final EIEntity property, final String value) {
		Set<String> pValues;
		pValues = datatypeProperties.get( property );
		if ( pValues == null ) {
			pValues = new HashSet<String>();
			datatypeProperties.put( property, pValues );
		}
		pValues.add( value );
	}

	public Set<String> getDatatypeProperty(final EIEntity property) {
		return datatypeProperties.get( property );
	}

	public void replaceDatatypePropertyValue(final EIEntity property, final String oldValue, final String newValue) {
		Set<String> pValues = datatypeProperties.get( property );
		if ( pValues == null ) {
			pValues = new HashSet<String>();
			datatypeProperties.put( property, pValues );
		}
		pValues.remove( oldValue );
		if ( newValue != null && !newValue.equals( "" ) ) {
			pValues.add( newValue );
		}
	}

	public void replaceDatatypePropertyAllValues(final EIEntity property, final Set<String> values) {
		datatypeProperties.put( property, values );
	}

	public void addEIType(final EIEntity eiType) {
		if ( eiType == null ) {
			return;
		}
		if ( otherEITypes == null ) {
			otherEITypes = new ArrayList<EIEntity>();
		}
		otherEITypes.add( eiType );
	}

	public void setOtherEITypes(final List<EIEntity> eiTypes) {
		otherEITypes = eiTypes;
	}

	public List<EIEntity> getOtherEITypes() {
		if ( otherEITypes == null ) {
			return Collections.emptyList();
		} else {
			return otherEITypes;
		}
	}

	public int compareTo(final Object o) {
		if ( !( o instanceof EIBasicInstance ) ) {
			return 0;
		}
		return getEntity().getLabel().compareToIgnoreCase( ( (EIBasicInstance)o ).getEntity().getLabel() );
	}

	@Override
	public String toString() {
		// Display with the first asserted class
		return instanceType.toString() + " : " + getEntity().toString();
	}

	public void setHasAllRequiredProperties(boolean instanceHasAllRequiredProperties) {
		this.instanceHasAllRequiredProperties = instanceHasAllRequiredProperties;
	}

	public boolean hasAllRequiredProperties() {
		return instanceHasAllRequiredProperties;
	}

	public boolean isEmbedded() {
		return isEmbedded;
	}

	public void setEmbedded(boolean isEmbedded) {
		this.isEmbedded = isEmbedded;
	}

	public EIClass getRootSuperType() {
		return rootSuperType;
	}

	public void setRootSuperType(EIClass rootSuperType) {
		this.rootSuperType = rootSuperType;
	}
	
	public EIEntity findPropertyForInstance(EIEntity instanceEntity) {
		if (objectProperties == null) {
			return EIEntity.NULL_ENTITY;
		}
		
		for ( EIEntity property : objectProperties.keySet() ) {
			if ( objectProperties.get(property).contains( instanceEntity ) ) {
				return property;
			}
		}
		
		return EIEntity.NULL_ENTITY;
	}
 
}
