package org.eaglei.network.actions;

import static org.spin.tools.Util.guardNotNull;

import java.io.IOException;

import org.apache.log4j.Logger;
import org.eaglei.search.common.SerializationException;
import org.eaglei.search.common.Serializer;
import org.eaglei.search.provider.SearchProvider;
import org.spin.node.QueryContext;
import org.spin.node.actions.QueryException;

/**
 * 
 * @author Clint Gilbert
 * 
 * Jul 7, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 * @param <In>
 */
public abstract class SearchProviderQueryAction<In, Out> extends WithSerializerQueryAction<In> implements HasSearchProvider
{
    private static final Logger log = Logger.getLogger(SearchProviderQueryAction.class);
    
    private static final boolean INFO = log.isInfoEnabled();
    private static final boolean DEBUG = log.isDebugEnabled();
    
    private final Serializer<Out> outputSerializer;
    
    private final SearchProviderOp<In, Out> operation;
    
    SearchProviderQueryAction(final Serializer<In> inputSerializer, final Serializer<Out> outputSerializer, final SearchProviderOp<In, Out> operation)
    {
        super(inputSerializer);
        
        guardNotNull(outputSerializer);
        guardNotNull(operation);
        
        this.outputSerializer = outputSerializer;
        this.operation = operation;
    }
    
    @Override
    public SearchProvider getSearchProvider()
    {
        return operation.getSearchProvider();
    }

    @Override
    public String perform(final QueryContext context, final In request) throws QueryException
    {
        guardNotNull(request);

        if(DEBUG)
        {
            try
            {
                log.debug("Raw input JSON: '" + getInputSerializer().serialize(request) + "'");
            }
            catch(SerializationException e)
            {
                log.error("Error serializing SearchRequest to JSON for debug output", e);
            }
        }

        final Out result;
        
        try
        {
            //NB: Lazily invoke init on searchProvider to work around circular init dependency issue
            //See http://jira.eagle-i.net:8080/browse/SEARCH-158 -Clint
            Shared.initializeProviderIfNecessary();
            
            result = operation.perform(request);
        }
        catch(final IOException e)
        {
            throw new QueryException("Error performing SearchProvider operation: ", e);
        }

        if(INFO)
        {
            log.info("Performed SearchProvider operation");
        }

        // TODO: Error handling for Gson serialization?
        // TODO: Is calling toJson() on a shared Gson instance thread-safe?
        // Googling seems to indicate that it is, but I wouldn't mind some
        // reassurance. -Clint
        final String jsonResults;

        try
        {
            jsonResults = outputSerializer.serialize(result);
        }
        catch(SerializationException e)
        {
            throw new QueryException("Error serializing results: ", e);
        }

        if(DEBUG)
        {
            log.debug("Raw JSON results '" + jsonResults + "'");
        }

        return jsonResults;
    }
}
