package org.eaglei.network.actions;

import static java.util.Collections.unmodifiableList;

import java.util.Arrays;
import java.util.List;

import org.eaglei.network.driver.Query;
import org.spin.node.actions.QueryAction;
import org.spin.tools.Util;
import org.spin.tools.config.QueryTypeConfig;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;

/**
 * 
 * @author Clint Gilbert
 * 
 * Nov 22, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 */
public enum EagleIQueryActions
{
    RDF(Query.RDF, RDFQueryAction.class), 
    Count(Query.Count, CountQueryAction.class);
    
    private final Query query;
    
    private final Class<? extends QueryAction<?>> queryActionClass;

    private EagleIQueryActions(final Query query, final Class<? extends QueryAction<?>> queryActionClass)
    {
        this.query = query;
        this.queryActionClass = queryActionClass;
    }

    public Query query()
    {
        return query;
    }

    public Class<? extends QueryAction<?>> queryActionClass()
    {
        return queryActionClass;
    }
    
    public QueryAction<?> obtainQueryAction(final ApplicationContext context) throws BeansException
    {
        return context.getBean(queryActionClass());
    }
    
    @Override
    public String toString()
    {
        return name() + " query=" + query + " queryActionClass=" + queryActionClass;
    }

    public QueryTypeConfig toQueryTypeConfig()
    {
        return new QueryTypeConfig(query.queryType, queryActionClass.getName());
    }
    
    public static final List<String> queryTypes = makeQueryTypeList();

    private static List<String> makeQueryTypeList()
    {
        final List<String> result = Util.makeArrayList(values().length);
        
        for(final EagleIQueryActions queryActionDef : values())
        {
            result.add(queryActionDef.query.queryType);
        }
        
        return unmodifiableList(result);
    }
    
    public static final List<Query> queries = makeQueryList();

    private static List<Query> makeQueryList()
    {
        final List<Query> result = Util.makeArrayList(values().length);
        
        for(final EagleIQueryActions queryActionDef : values())
        {
            result.add(queryActionDef.query);
        }
        
        return unmodifiableList(result);
    }
    
    public static EagleIQueryActions fromQueryType(final String queryType)
    {
        return fromQuery(Query.fromQueryType(queryType));
    }
    
    public static EagleIQueryActions fromQuery(final Query query)
    {
        for(final EagleIQueryActions queryActionDef : values())
        {
            if(queryActionDef.query == query)
            {
                return queryActionDef;
            }
        }
        
        throw new IllegalArgumentException("Unknown query '" + query + "', allowed values are " + Arrays.asList(values()));
    }
}
