#!/bin/bash
#
# Copyright (c) 2015, President and Fellows of Harvard College
# Portions Copyright (c) 2015, VectorC, LLC
# Portions Copyright (c) 2015, Wonder Lake Software, LLC
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. The name of the author may not be used to endorse or promote products
# derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#


## Changing to use getopts to allow for more complex combination of arguments
## to allow for the optional argument of what tier to munge instance uris to
## as well as the optional argument of a migration directory (SKC, 6/25/2013)

# Fail on subprocess failures or undefined variables
set -o errexit
set -o nounset

usernameSet=false
passwordSet=false
remoteHostSet=false
migrationFileDir=""

printUsage() {
	echo '--------------'
	echo 'usage: $0 [-u username] [-p password] [-r repo-URL-root] [-d migration-file-directory (OPTIONAL)]'
    echo 'e.g. $0  -u bigbird -p passwird -r https://repo.mysite.edu:8443'
	echo '**** It is a good idea to make a backup before executing this script ****'
	echo 'eagle-i-datatools-datamanagement.jar *must* be present in this directory for the script to succeed.'
	echo 'Download the version you need and rename it to eagle-i-datatools-datamanagement.jar'
	echo 'e.g. wget -O eagle-i-datatools-datamanagement.jar http://repo.eagle-i.net/nexus/content/repositories/releases/org/eagle-i/eagle-i-datatools-datamanagement/1.7-MS1.02/eagle-i-datatools-datamanagement-1.7-MS1.02.jar'
	echo '--------------'
} 

setExpectations() {
    if [ $expectingValue == true ]; then
        error=true
        echo "Was expecing a value for for preceding modifier; got $1"
    else
        expectingValue=true
    fi
}

# Need to verify that the parameters are correct and be able to explain why they're not.
# should be $0 flag1 value1 flag2 value2 flag3 value3 [flag4 value4]
error=false
expectingValue=false
expectingUser=false
expectingPassword=false
expectingRemote=false
expectingMigrationDir=false
while [ $# -gt 0 ]; do
    #echo "$1; expectValue = $expectingValue"
    case "$1" in
        -u)
            expectingUser=true
            setExpectations
            ;;
        -p)
            expectingPassword=true
            setExpectations
            ;;
        -r)
            expectingRemote=true
            setExpectations
            ;;
        -d)
            expectingMigrationDir=true
            setExpectations
            ;;
        -*)
            error=true
            echo "Unknown option, $1"
            expectingValue= true;
            ;;
        *)
            if [ $expectingUser == true ]; then
                export username="$1"
                expectingUser=false
                usernameSet=true
            elif [ $expectingPassword == true ]; then
                export password="$1"
                expectingPassword=false
                passwordSet=true
            elif [ $expectingRemote == true ]; then
                export remoteHost="$1"
                expectingRemote=false
                remoteHostSet=true
            elif [ $expectingMigrationDir == true ]; then
                export migrationFileDir="$1"
                expectingMigrationDir=false
                migrationFileDirSet=true
            fi
            if [ $expectingValue == false ]; then
                error=true
                echo "Was expecting modifier-flag, not value $1"
            fi
            expectingValue=false
            ;;
    esac
    shift
done # darn!  that was tedious....

if ! $usernameSet
then
    echo "-u username must be specified!" >&2
    error=true
fi
if ! $passwordSet
then
    echo "-p password must be specified!" >&2
    error=true
fi
if ! $remoteHostSet
then
    echo "-r remoteHostName must be specified!" >&2
    error=true
fi

if ! $usernameSet || ! $passwordSet || ! $remoteHostSet
then
	echo 'Arguments must include non-void values for -u -p -r' >&2
    error=true
fi

if [ $error == true ]; then
    printUsage
exit 1
fi

creds=$username:$password
executableDir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

cp="$executableDir:$executableDir/eagle-i-datatools-datamanagement.jar"

if [ ! -d logs ] ; then
    mkdir logs
fi

echo "Java info: "
which java
java -version 2>&1
echo ""
echo "using classpath : $cp"
echo ""


standard () {
if [ -z "$migrationFileDir" ]; then
    migrationFileDir=migration-files
fi

basecsv=${migrationFileDir}/migration_file_
propcsv=${basecsv}property.csv
classcsv=${basecsv}class.csv
instancecsv=${basecsv}instance.csv

# Check return codes from java; fail if > 0
java -cp $cp org.eaglei.datatools.datamanagement.command.ChangePredicate -c $creds -f ${propcsv} -r ${remoteHost}
subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ${propcsv} - exiting"
    exit $subProcExitCode
else
    echo "ChangePredicate on ${propcsv} complete."
fi


java -cp $cp org.eaglei.datatools.datamanagement.command.ChangeObject -c $creds -f ${classcsv} -r ${remoteHost}
subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ${classcsv} - exiting"
    exit $subProcExitCode
else
    echo "ChangeObject on ${classcsv} complete."
fi

java -cp $cp org.eaglei.datatools.datamanagement.command.ChangeObject -c $creds -f ${instancecsv} -r ${remoteHost}
subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ${instancecsv} - exiting"
    exit $subProcExitCode
else
    echo "ChangeObject on ${instancecsv} complete."
fi


echo "Done applying standard data migration procedures. "
}

globalData() {
if [ -z "$migrationFileDir" ]; then
    migrationFileDir=migration-files
fi

basecsv=${migrationFileDir}/migration_file_
globalcsv=${basecsv}global.csv
genericcsv=${basecsv}generic.csv

echo "======= global data migration ======="
java -cp $cp org.eaglei.datatools.datamanagement.command.ChangeAndDelete -c $creds -f ${globalcsv} -r ${remoteHost}
	
subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ${globalcsv} - exiting"
    exit $subProcExitCode
else
    echo "ChangeAndDelete on ${globalcsv} complete."
fi
echo "Done translating global data migration procedures. "
}

############
# SPECIALS #
############
special_1_8MS1 () {
echo "======= 1.8MS1.01 special data migration ======="
echo "For description of these changes, see: https://docs.google.com/a/eagle-i.org/document/d/1n5UrvJ9Z7aNpS5ea2Gvg7SzhlBWY0Xsl9TGtokU5ceY"
# (1) move away from software commercial and open source subtypes
software=http://purl.obolibrary.org/obo/ERO_0000071
commercialSoftware=http://purl.obolibrary.org/obo/ERO_0000075
openSoftware=http://purl.obolibrary.org/obo/ERO_0000074
commercialLicense=http://www.ebi.ac.uk/swo/license/SWO_1000002
openLicense=http://www.ebi.ac.uk/swo/license/SWO_1000008
hasLicense=http://purl.obolibrary.org/obo/ERO_0001824

echo "(1) Migrating obsolete software types..."
java -cp $cp org.eaglei.datatools.datamanagement.command.MoveExistingTypeAndSetNewType -c $creds -r ${remoteHost} -t ${commercialSoftware} -p ${hasLicense} -o ${commercialLicense} -nt ${software}
subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java MoveExistingTypeAndSetNewType ${commercialSoftware} - exiting"
    exit $subProcExitCode
else
    echo "Migration of Commercial Software complete."
fi

java -cp $cp org.eaglei.datatools.datamanagement.command.MoveExistingTypeAndSetNewType -c $creds -r ${remoteHost} -t ${openSoftware} -p ${hasLicense} -o ${openLicense} -nt ${software}

subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java MoveExistingTypeAndSetNewType ${openSoftware} - exiting"
    exit $subProcExitCode
else
    echo "Migration of Open Source Software complete."
fi

# (2) New embedded instances for algorithms
hasDataInput=http://purl.obolibrary.org/obo/ERO_0000076
hasDataOutput=http://purl.obolibrary.org/obo/ERO_0000077

echo "(2) Creating embedded instances for software input and output..."
java -cp $cp org.eaglei.datatools.datamanagement.command.ConvertObjectToEmbeddedInstance -c $creds -r ${remoteHost} -p ${hasDataInput}

subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ConvertObjectToEmbeddedInstance ${hasDataInput} - exiting"
    exit $subProcExitCode
else
    echo "Creation of embedded instances for data input complete."
fi
java -cp $cp org.eaglei.datatools.datamanagement.command.ConvertObjectToEmbeddedInstance -c $creds -r ${remoteHost} -p ${hasDataOutput}

subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ConvertObjectToEmbeddedInstance ${hasDataOutput} - exiting"
    exit $subProcExitCode
else
    echo "Creation of embedded instances for data output complete."
fi

# (3) New embeded instances for Organisms and change obsolete predicate
oldHasPhenotype=http://purl.obolibrary.org/obo/ERO_0000578
newHasPhenotype=http://purl.obolibrary.org/obo/ERO_0001836
phenotype=http://purl.obolibrary.org/obo/ERO_0001828

echo "(3) Creating embedded instances for phenotypes and migrating obsolete hasPhenotype predicate..."
java -cp $cp org.eaglei.datatools.datamanagement.command.ConvertObjectToEmbeddedInstance -c $creds -r ${remoteHost} -p ${oldHasPhenotype} -asLiteral -et ${phenotype}

subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ConvertObjectToEmbeddedInstance ${oldHasPhenotype} - exiting"
    exit $subProcExitCode
else
    echo "Creation of embedded instances for phenotype complete."
fi

java -cp $cp org.eaglei.datatools.datamanagement.command.ChangePredicate -c $creds -r ${remoteHost} -op ${oldHasPhenotype} -np ${newHasPhenotype}

subProcExitCode=$?
if [ $subProcExitCode -gt 0 ] ; then
    echo "non-zero exit code $subProcExitCode from java ChangePredicate - exiting"
    exit $subProcExitCode
else
    echo "Migration of obsolete hasPhenotype predicate complete."
fi


echo "Done applying 1.8 MS1.01 special data migration. "

}

special_3_1(){
if [[ $remoteHost == *shared* ]]
then
    echo "======= 3.1 special data migration for shared node ======="
    #Types to migrate
    accessService=http://purl.obolibrary.org/obo/ERO_0000391
    resourceCollection=http://purl.obolibrary.org/obo/ERO_0002190
    #Properties to migrate
    serviceProvidedBy=http://purl.obolibrary.org/obo/ERO_0000390
    developedBy=http://purl.obolibrary.org/obo/ERO_0001719
    relatedResource=http://purl.obolibrary.org/obo/ERO_0000029
    isMemberOf=http://purl.obolibrary.org/obo/ERO_0002191

    echo "(1) Swapping object and subject for collection membership"
    java -cp $cp org.eaglei.datatools.datamanagement.command.SwapObjectAndSubject -c $creds -r ${remoteHost} -op ${relatedResource} -tp ${isMemberOf} -t ${accessService}    
    echo "(2) Changing type from access service to resource collection"
    java -cp $cp org.eaglei.datatools.datamanagement.command.ChangeObject -c $creds -r ${remoteHost} -oo ${accessService} -no ${resourceCollection}
    echo "(3) Changing relationship collection -> organization"
    java -cp $cp org.eaglei.datatools.datamanagement.command.ChangePredicate -c $creds -r ${remoteHost} -op ${serviceProvidedBy} -np ${developedBy}
else
    echo -e "***** There are no special procedures for this release."
fi
}

## Xxx When there is a special procedure in the current release,
## add a call to it here and change 'false' to 'true' in the next line:

if true; then
  echo -e "\n***** *S*P*E*C*I*A*L*  data migration procedures\n"
  special_3_1
else
    echo -e "\n***** There are no special procedures for this release."
fi

# Call the standard migration at the end
echo -e "\n***** Standard data migration procedures\n"
standard
globalData
echo ""
echo ""
echo "See:"
echo "Migration report (for curators): logs/data-migration-report.log"
echo "Full Log (for sysadmins): logs/datamanagement.log"

