#!/bin/bash
#
# Copyright (c) 2015, President and Fellows of Harvard College
# Portions Copyright (c) 2015, VectorC, LLC
# Portions Copyright (c) 2015, Wonder Lake Software, LLC
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. The name of the author may not be used to endorse or promote products
# derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#


# Move "everything" from one repo to another
#
#   * * * * *    W A R N I N G    * * * * *
#
# THIS IS VERY DANGEROUS AND EVEN WHEN IT WORKS AS INTENDED IT IS A BAD IDEA.
#
# It makes a quasi-functional clone of a repo's data
# This script grabs ALL of the resource and user data from one repo,
# EDITS it to replace all of the locally-prefixed URIs with the prefix for
# the destination site so they appear to be "local" there.  Since it is
# replacing all of the data on the destination repo there is no possibility
# of colliding with URIs already present there; in that sense this is
# fairly "safe".  However, it still has the following problems:
#
# 1. Since the resources are such accurate copies they might be confused
#    with the prototypes.  This would be bad.
#
# 2. If users are in a position to access both the copy and the original
#    they might not be able to tell the difference and would make updates
#    to the wrong site.
#
# 3. Claims and other admin metadata are preserved.  Caveat emptor.
#
# 4. The same user accounts, INCLUDING PASSWORDS, will work on destination.
#
# Copies made with this script MUST be treated as TEMPORARY AND DISPOSABLE
# for these reasons.
#
# See Also: make-snapshot.sh for the script that creates a "snapshot"
# backup of user and resource data in the local filesystem, which can be
# read via the --from-snapshot option instead of a live copy from the source.
#
# $Id: move-everything.sh 19943 2016-03-16 18:48:46Z scheng $
# Author: Larry Stone

#------- Constants

format=application/x-trig

# useful namespace prefixs
repo="http://eagle-i.org/ont/repo/1.0/"
owl="http://www.w3.org/2002/07/owl#"
rdfs="http://www.w3.org/2000/01/rdf-schema#"

# NOTE: These must match the paths used by make-snapshot.sh
snapshotResourceFile=resources.trig
snapshotUserFile=users.trig

#-------- boilerplate
# keep files in /tmp subdir, cleanup on exit
tmpDir="/tmp/repo-moveevery-$$"
mkdir $tmpDir

cleanup () {
  ## XXX insert echo here to retain the tmp work dir
  rm -rf $tmpDir
}
trap cleanup EXIT

# dump resources
# Usage: $0  login url  file
doGetResources () {
    local login=${1}
    local repoURL=$2
    local toFile=$3

    local status=""
    if status=`curl -k -s -S -G -u "${login}" -o "${toFile}" -d all=true \
      --write-out '%{http_code}' -d "format=${format}" "${repoURL}/repository/graph"`; then
        if [ "$status" != "200" ]; then
            echo "HTTP Request to get graph dump failed, status=${status}"
            if [ -f "$toFile" ]; then
                echo "Response document:"
                cat "$toFile"
            fi
            return 1
        fi
    else
        echo "curl failed to send HTTP request to get graph dump."
        return 1
    fi
    return 0
}

# dump users
# Usage: $0  login url  file
doGetUsers () {
    local login=${1}
    local repoURL=$2
    local toFile=$3

    local status=""
    if status=`curl -k -s -S -G -u "${login}" -o "${toFile}" -d type=user -d "exclude=${excludeUsers}" \
        --write-out '%{http_code}' -d "format=${format}" "${repoURL}/repository/export"` ; then
        if [ "$status" != "200" ]; then
            echo "HTTP Request to get user dump failed, status=${status}"
            if [ -f "$toFile" ]; then
                echo "Response document:"
                cat "$toFile"
            fi
            return 1
        fi
    else
        echo "curl failed to send HTTP request to get user dump."
        return 1
    fi
    return 0
}


# upload resources to destination
# Usage: $0  login url  file
doPutResources () {
    local login=${1}
    local repoURL=$2
    local fromFile=$3

    local status=""
    if status=`curl -k -s -S -u "${login}" -F all=true -F action=replace \
      -F "format=${format}" -F "content=@$fromFile" \
      --write-out '%{http_code}' "${repoURL}/repository/graph"`; then
        if [ "$status" != "201" ]; then
            echo "HTTP Request to PUT graph dump failed, status=${status}"
            return 1
        fi
    else
        echo "curl failed to send HTTP request to PUT graph dump."
        return 1
    fi
    return 0
}

# upload users to destination
# Usage: $0  login url  file
doPutUsers () {
    local login=${1}
    local repoURL=$2
    local fromFile=$3

    local status=""
    if status=`curl -k -s -S -u "${login}" -F type=user -F duplicate=replace \
      -F transform=no -F "format=${format}" -F "content=@$fromFile" \
      --write-out '%{http_code}' "${repoURL}/repository/import"`; then
        if [ "$status" != "200" ]; then
            echo "HTTP Request to /import users failed, status=${status}"
            return 1
        fi
    else
        echo "curl failed to send HTTP request to /import users."
        return 1
    fi
    return 0
}

# Usage: $0  login  url
# returns the URI prefix of the repo
getPrefix () {
    local login=${1}
    local repoURL=$2

    local tmp=${tmpDir}/prefix.txt
    local status=""
    rm -f ${tmp}

    if status=`curl -k -s -S -u "${login}" -o "${tmp}" -d format=text/plain \
                --write-out '%{http_code}' "${repoURL}/repository/new"`; then
        if [ "$status" != "200" ]; then
            echo "HTTP Request /new to get URI prefix failed, status=${status}"
            if [ -f "$tmp" ]; then
                echo "Response document:"
                cat "$tmp"
            fi
            return 1
        fi
    else
        echo "curl failed to send HTTP request to get URI prefix."
        return 1
    fi
    tail -n +2 "${tmp}" | egrep -o '^(.+/)'
}

# Usage: $0 login repoURL username filepath
saveAdminUser () {
    local login=${1}
    local repoURL=$2
    local username=$3
    local out=$4
    local status=""
    rm -f "$out"

    # export the admin user's state
    if status=`curl -k -s -S -G -u "${login}" -o "${out}" -d type=user \
                -d "include=$username" -d format=${format} \
                --write-out '%{http_code}' "${repoURL}/repository/export"`; then
        if [ "$status" != "200" ]; then
            echo "HTTP Request /export to get admin user failed, status=${status}"
            if [ -f "$out" ]; then
                echo "Response document:"
                cat "$out"
            fi
            return 1
        fi
    else
        echo "curl failed to send HTTP request to get /export, status=$?"
        return 1
    fi
}


# Usage: $0 login repoURL username filepath
# fix admin login user if necessary - check if it has URI, if not, invoke /whoami to create
restoreAdminUser () {
    local login=${1}
    local repoURL=$2
    local username=$3
    local in=$4
    local status=""
    local out=${tmpDir}/restore.out

    if status=`curl -k -s -S -u "${login}" -F type=user -F duplicate=replace \
      -F "include=$username" -F transform=no -F "format=${format}" -F "content=@$in" \
      --write-out '%{http_code}' "${repoURL}/repository/import"`; then
        if [ "$status" != "200" ]; then
            echo "HTTP Request to import admin user failed, status=${status}"
            return 1
        fi
    else
        echo "curl failed to send HTTP request to import admin user."
        return 1
    fi
    return 0
}

# move to-repo from a file-based resource file (trig format)
#------------------  Main ----------------------------
usageOptions="[--version] [ -f | --force ] [ --nousers ] [ --exclude-users userlist ]"
usageFrom="[ from-username  from-password  from-repo-URL | --from-snapshot directory --from-prefix from-prefix ]"
usage0="Usage: $0  {options}  {source}  {destination-user} {dest-password} {dest-repo-url}"
usage1="  ..where {source} is EITHER the user/password/URL args to access a repository or --from-snapshot and --from-prefix"
usage="${usage0}\n  ${usage1}\n  i.e.\n  $0 $usageOptions $usageFrom to-username  to-password  to-repo-URL"

excludeUsers=""
moveUsers=true
fromSnapshot=""
fromPrefix=""

while echo "$1" | grep -q "^-"; do
    case "$1" in
        --version)
            echo "$0 from release 4.5.1 SCM revision 20108"
            exit 0 ;;
        -h|--help)
            echo -e "$usage"
            exit 0 ;;
        -f|--force)
            forceOption=true ;;
        --exclude-users)
            shift;
            excludeUsers="$1" ;;
        --nousers)
            moveUsers=false ;;
        --from-snapshot)
            shift;
            fromSnapshot="$1" ;;
        --from-prefix)
            shift;
            fromPrefix="$1" ;;
        -*)
              echo "Unknown option $1"
              echo -e $usage
              exit 1;;
    esac
    shift
done

# sanity check file args - from-snapshot must include from-prefix
if [ \( -n "${fromSnapshot}" -a -z "${fromPrefix}" \) -o \
     \( -z "${fromSnapshot}" -a -n "${fromPrefix}" \) ]; then
    echo "ERROR, The --from-snapshot and --from-prefix must both be present."
    echo "Enter $0 --help for usage."
    exit 1
fi

# sanity check arg list
if [ \( -n "${fromSnapshot}" -a $# -ne 3 \) -o \
     \( -z "${fromSnapshot}" -a $# -ne 6 \) ]; then
            echo "ERROR, wrong number of arguments."
            echo -e $usage
            exit 1
fi

# sort out positional args
if [ -z "${fromSnapshot}" ]; then
        fromLogin="${1}:$2"
        fromURL=$3
        toUsername=$4
        toLogin="${4}:$5"
        toURL=$6
else
        toUsername=$1
        toLogin="${1}:$2"
        toURL=$3
fi

if [ "${forceOption:false}" != true ]; then
    echo "*************************************"
    echo "*****  DANGER! DANGER! DANGER!  *****   This is a VERY DANGEROUS procedure, and"
    echo "*****  DANGER! DANGER! DANGER!  *****   even when it works as intended it is a"
    echo "*****  DANGER! DANGER! DANGER!  *****   VERY BAD IDEA.  Reply YES below ONLY "
    echo "*****  DANGER! DANGER! DANGER!  *****   if you KNOW what will happen and are "
    echo "*****  DANGER! DANGER! DANGER!  *****   STILL SURE you want to continue."
    echo "*****  DANGER! DANGER! DANGER!  ***** "
    echo "*****  DANGER! DANGER! DANGER!  *****   This will ***DESTROY*** ALL contents"
    echo "*****  DANGER! DANGER! DANGER!  *****   of the repository on the repository at:"
    echo "*****  DANGER! DANGER! DANGER!  *****      $toURL "
    echo "*************************************"
    echo ""
    echo -n "Are you STILL sure you want to continue? "
    read answer
    if echo "$answer" | egrep -iq '^[[:space:]]*y(es)?'; then
        echo "OK, continuing..."
    else
        echo "Aborting."
        exit 0
    fi
fi

### Move "Everything"

# setup for source as URL..
if [ -z "${fromSnapshot}" ]; then
    resources=${tmpDir}/resources.trig
    users=${tmpDir}/users.trig

    ## Get URL prefixes
    if fromPrefix=`getPrefix "$fromLogin" "$fromURL"` ; then
        true
    else
        echo "Failed getting prefix for source."
        echo $fromPrefix
        exit 1
    fi

    echo -n "  fetching RDF contents from ${fromURL}..."
    if doGetResources "$fromLogin" "$fromURL" "$resources" ; then
        echo "done."
    else
        echo "FAILED fetching source contents."
        exit 1
    fi

    if $moveUsers ; then
        echo -n "  fetching users from ${fromURL}..."
        if doGetUsers "$fromLogin" "$fromURL" "$users" ; then
            echo "done."
        else
            echo "FAILED fetching source users."
            exit 1
        fi
    fi

# indicate local files for resources
# fromPrefix is already set by arg
else
    resources=${fromSnapshot}/${snapshotResourceFile}
    users=${fromSnapshot}/${snapshotUserFile}

    # sanity check files
    if [ ! -r "$resources" ]; then
        echo "Cannot access source file for resources: [Resource {targetPath: etc, filtering: true, FileSet {directory: /home/bamboo/bamboo-home/xml-data/build-dir/EAGLEI-RELEASE10-JOB1/target/checkout/repository/scripts/src/main/etc, PatternSet [includes: {**}, excludes: {**/*.bak}]}}]: Not found."
        exit 1
    fi
    if $moveUsers && [ ! -r "$users" ]; then
        echo "Cannot access source file for users: ${users}: Not found."
        exit 1
    fi
fi

if toPrefix=`getPrefix "$toLogin" "$toURL"` ; then
    true
else
    echo "Failed getting prefix for destination."
    echo $toPrefix
    exit 1
fi

# Save current state of the dest admin user for later restoration if needed:
adminUser=${tmpDir}/adminUser.trig
if saveAdminUser "$toLogin" "$toURL" "$toUsername" "$adminUser" ; then
    echo "  Saved state of user $toUsername"
else
    echo "Failed to save state of user $toUsername"
    exit 1
fi

## Transform resource URLs
newResources=${tmpDir}/newResources.trig
newUsers=${tmpDir}/newUsers.trig
prog="\$f=quotemeta('${fromPrefix}'); while (<>) {s@<\${f}@<${toPrefix}@g;print}"
if perl -CSD -e "$prog" < $resources > $newResources ; then
    true
else
    echo "Transformation of resource data failed, see error messages above."
    exit 1
fi

## Transform User URLs
if $moveUsers ; then
    if perl -CSD -e "$prog" < $users > $newUsers ; then
        true
    else
        echo "Transformation of user data failed, see error messages above."
        exit 1
    fi
fi

echo -n "  uploading RDF contents to ${toURL}...."
if doPutResources "$toLogin" "$toURL" "$newResources" ; then
    echo "done."
else
    echo "FAILED uploading resource contents."
    exit 1
fi

if $moveUsers ; then
    echo -n "  uploading users to ${toURL}..."
    if doPutUsers "$toLogin" "$toURL" "$newUsers" ; then
        echo "done."
    else
        echo "FAILED uploading users"
        exit 1
    fi

    if restoreAdminUser "$toLogin" "$toURL" "$toUsername" "$adminUser"; then
        echo "  Restored state of user $toUsername"
    else
        echo "Failed to restore state of user $toUsername"
        exit 1
    fi
fi
