#!/bin/bash
#
# Copyright (c) 2015, President and Fellows of Harvard College
# Portions Copyright (c) 2015, VectorC, LLC
# Portions Copyright (c) 2015, Wonder Lake Software, LLC
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. The name of the author may not be used to endorse or promote products
# derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#


# This script is to be run AFTER replicating the repository.
# This script will munge objects with a global namespace from the production namespace
# to the corresponding tier global namespace

# Fail on undefined variables and/or subcommand failures
set -o nounset
set -o errexit

usernameSet=false
passwordSet=false
remoteHostSet=false
tierSet=false

printUsage() {
	echo '--------------'
	echo 'usage: $0 [-u username] [-p password] [-r repo-URL-root] [-d migration-file-directory (OPTIONAL)] [-t tier-to-munge-URLs], e.g. $0  -u bigbird -p passwird -t qa - r https://repo.mysite.edu:8443'
	echo '**** It is a good idea to make a backup before executing this script ****'
	echo 'eagle-i-datatools-datamanagement.jar *must* be present in this directory for the script to succeed.'
	echo 'Download the version you need and rename it to eagle-i-datatools-datamanagement.jar'
	echo 'e.g. wget -O eagle-i-datatools-datamanagement.jar http://repo.eagle-i.net/nexus/content/repositories/releases/org/eagle-i/eagle-i-datatools-datamanagement/1.7-MS1.02/eagle-i-datatools-datamanagement-1.7-MS1.02.jar'
	echo '--------------'
} 
            
while getopts ":u:p:r:t:" opt; do
    case $opt in
    	u  ) export username=$OPTARG
    		 usernameSet=true
    		# echo $username
    		 ;;
    	p  ) export password=$OPTARG
    		 passwordSet=true
    		# echo $password
    		 ;;
    	r  ) export remoteHost=$OPTARG
    		 remoteHostSet=true
    		# echo $remoteHost
    		 ;;    		 
    	t  ) export mungeToTier=$(echo $OPTARG | awk '{print tolower($0)}')
    		 tierSet=true
#    		 echo $mungeToTier
    		 ;;
        \? ) printUsage
             exit 1
             ;;
		:  ) printUsage
			 echo "Option -$OPTARG requires an argument." >&2
      		 exit 1
    esac
done

if ! $usernameSet || ! $passwordSet || ! $tierSet || ! $remoteHostSet
then
	echo 'Required arguments missing!'
	printUsage
	exit 1
fi

if [ "$mungeToTier" != "dev" ] && [ "$mungeToTier" != "qa" ] && [ "$mungeToTier" != "stage" ]
then
	echo 'Invalid tier specified, must be one of [dev/qa/stage]'
	exit 1
fi

creds=$username:$password
executableDir="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"

cp="$executableDir:$executableDir/eagle-i-datatools-datamanagement.jar"

if [ ! -d logs ] ; then
    mkdir logs
fi

echo "Java info: "
which java
java -version 2>&1
echo ""
echo "using classpath : $cp"
echo "Munging objects from production to tier: "$mungeToTier

# Group everything up through 'global.' into group 1
# Group everything from 'eagle-i.net' to the end into group 2
# DOESN'T WORK :'( .. doing it the uglier way
# oldreg="(?i)(.*global\.)(eagle\-i\.net.*)"
oldrex="http://global.eagle-i.net/(.*?)"
# newrex="$1"${tier}".$2"
newrex="http://global."${tier}".eagle-i.net/$1"

#SKC this isn't working!!
java -cp $cp org.eaglei.datatools.datamanagement.command.ChangeObjectByRegex -c $creds -r ${remoteHost} -oldregex ${oldrex} -newregex ${newrex} -asResource -s
echo -e "\n***** Global munging objects done\n"

echo ""
echo "See:"
echo "Migration report (for curators): logs/data-migration-report.log"
echo "Full Log (for sysadmins): logs/datamanagement.log"

echo ""