#!/bin/bash
#
# Copyright (c) 2015, President and Fellows of Harvard College
# Portions Copyright (c) 2015, VectorC, LLC
# Portions Copyright (c) 2015, Wonder Lake Software, LLC
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. The name of the author may not be used to endorse or promote products
# derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#


# Repository Upgrade Script
#
# Run this after installing a new release of the repository webapp
# *ONLY* WHEN PRESERVING THE SESAME RDF DATABASE CONTENTS OF AN OLDER RELEASE.
# It will automatically upgrade the repository contents.
#
# Any data migration operations include a test for whether there is anything
# needing migration, which means you can run this script again and it will
# simply skip the migration steps which have already successfully completed.
#
# See "usage" variable below for synopsis of arguments.
#
# Note: All diagnostic output is prefaced with either INFO, WARNING, or ERROR
#  so a script running this can simply grep for WARNING|ERROR to see
#  if there are any messages worth forwarding to a human.
#
# $Id: upgrade.sh 19120 2015-04-30 05:13:28Z scheng $
# Author: Larry Stone

#------- Constants
usage="Usage: $0  username  password  repo-URL-root, e.g. $0  bigbird passwird https://repo.mysite.edu:8443"

# useful namespace prefixs
repo="http://eagle-i.org/ont/repo/1.0/"
owl="http://www.w3.org/2002/07/owl#"
rdfs="http://www.w3.org/2000/01/rdf-schema#"

# delete wildcard
wildcard="${repo}MatchAnything"

# named graphs
NG_Repo="${repo}"
NG_Internal=${repo}NG_Internal
NG_Metadata=${repo}NG_Metadata
NG_Query=${repo}NG_Query

# data model ontology graph URI:
dataModel=http://purl.obolibrary.org/obo/ero.owl

inClassGroup="http://eagle-i.org/ont/app/1.0/inClassGroup"
embeddedClass="http://eagle-i.org/ont/app/1.0/ClassGroup_EmbeddedResourceType"


#-------- boilerplate
# keep files in /tmp subdir, cleanup on exit
tmpDir="/tmp/repo-upgrade-$$"
mkdir $tmpDir

cleanup () {
  ## XXX insert echo here to retain the tmp work dir
  echo rm -rf $tmpDir
}
trap cleanup EXIT

Debug=false

# ------------------ Function definitions ---------------------------------

# return a new URI via stdout
# usage: $0 [ optional-suffix ]
newURI () {
    local suffix="$1"
    local status
    local uriFile="${tmpDir}/uri"
    rm -f $uriFile
    if status=`curl -k -s -u $login -o "$uriFile" -d format=text/plain --write-out "%{http_code}" ${repoURL}/repository/new`; then
        if [ "$status" != "200" ]; then
            echo "newURI FAILED to get new URI, HTTP status=$status" 1>&2
            test -f $uriFile && cat $uriFile 1>&2
            return 1
        fi
    else
        echo "newURI FAILED sending HTTP request (curl) to get new URI: shell status=$?" 1>&2
        return 1
    fi
    local result=`tail -1 < "$uriFile" | tr -d ' \r\n\t'`
    if [ -n "$suffix" ]; then
        echo "${result}-$suffix"
    else
        echo "$result"
    fi
}


# run /update with insert, delete..
# args: LOGIN, action, subject, inserts, removes, workspace [expectStatus]
# NOTE: since this gets called inside a `` quote, so careful with output!
doUpdate () {
    local login=$1
    local action="$2"
    local subject="$3"
    local inserts="$4"
    local removes="$5"
    local wkspace="$6"
    local expectStatus="$7"
    local bypassSanity="$8"
    local status
    local insArg="insert"
    local delArg="delete"

    if [ -z "$expectStatus" ]; then
        if [ "$action" = "update" ]; then
            expectStatus="200"
        else
            expectStatus="201"
        fi
    fi

    local insaneArg="x_preserveSanity="
    if [ -n "$bypassSanity" ]; then
        insaneArg="bypassSanity="
    fi

    # disable functions by renaming http args
    if [ -z "$inserts" ]; then
        insArg="x_insert"
    fi
    if [ -z "$removes" ]; then
        delArg="x_delete"
    fi
    local out="${tmpDir}/out"
    rm -rf "$out"
    local token=""
    local tokenArg="foo=bar"
    if [ "$action" = "update" ]; then
        if token=`getToken "$login" "$subject"`;then
            tokenArg="token=$token"
        else
            echo "Failed to get token!"
            return 1
        fi
    fi

    local wkspaceArg="xworkspace="
    if [ -n "${wkspace}" ]; then
        wkspaceArg="workspace=${wkspace}"
    fi

    if status=`curl -k -o $out -s -u $login -F "action=${action}" \
     --form-string "${tokenArg}" --form-string 'format=text/rdf+n3' \
     --form-string "${wkspaceArg}" \
     --form-string "${insaneArg}" \
     --form-string "uri=$subject" \
     --form-string "${insArg}=${inserts}" --form-string "${delArg}=${removes}" \
     --write-out '%{http_code}' ${repoURL}/repository/update`; then
        if [ "$status" = "${expectStatus}" ]; then
            $Debug && echo "DEBUG: doUpdate OK." 1>&2
            if [ "$expectStatus" -gt 201 -a -f "$out" ]; then
                echo "OK - got expected status = $status"
                fgrep -A 1 '<b>Description</b>' $out
            fi
        else
            echo "doUpdate FAILED to update Instance=${subject}, HTTP status=$status" 1>&2
            test -f $out && cat $out 1>&2
            return 1
        fi
    else
        echo "DoUpdate  FAILED sending HTTP request (curl) to mung Instance=${subject}: shell status=$?" 1>&2
        return 1
    fi
    return 0
}

# Usage: $0 login uri
# get edit token, returns it through stdout
getToken () {
  local login="$1"
  local uri="$2"

  local tFile="${tmpDir}/token"
  rm -f $tFile
  local status
  if status=`curl -k -s -u $login -o "$tFile" -F action=gettoken -F 'format=text/plain' -F "uri=$uri" --write-out "%{http_code}" ${repoURL}/repository/update`; then
      if [ "$status" != "200" ]; then
        echo "getToken: Failed to get Token: HTTP status=$status " 1>&2
        test -f $tFile && cat $tFile 1>&2
        return 1
      fi
  else
      echo "getToken: Failed sending HTTP request to get token, status=$?" 1>&2
      return 1
  fi
  awk 'FNR ==2 { print $1 }'  "$tFile"
}

# Usage: doImport FILE TYPE DUPE TRANSFORM
doImport () {
    local afile=$1
    local atype=$2
    local dupe=$3
    local xform=$4

    local status=`curl -k -s -S -q -u "$login" --write-out "%{http_code}"  \
     -F type=${atype} -F "content=@${afile};type=application/x-trig" \
     -F duplicate=$dupe -F transform=$xform ${repoURL}/repository/import`
    cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
      echo "FAILED ${atype} import: HTTP status=$status cmd=$cmd " 1>&2
      return 1
    fi
}


# REturns the useful data from the /model query
# line contains 3 quoted strings: loaded, jar, and EI-model versions
getDataModelVersions () {
    local result="${tmpDir}/model"
    rm -f $result

    local status=`curl -k -s -S -G -u "$login" -o "$result" \
      -d "format=text/plain" --write-out "%{http_code}" ${repoURL}/repository/model`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
        echo "/model: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 11
    fi
    test -f "$result" && tail -n +2 "$result"
    return 0
}

# Usage:  N  line   -- where N is 1-3, line is returned by /model
# Also cleans up residual "'s at start or end.
selectModelVersion () {
    local n=$1
    echo "$2" | awk -F '"[ \t]+"' "{print \$${n}}" | sed -e 's/^"//' -e 's/"$//'
}

# Usage:  $0  versionA  versionB
# compare two Data Model Ontology version strings, success if the first
# is "newer" (later) than the second.
# Each version string has the following internal format:
#    <maven-version>   <SVN-revision>  <date>  <time>
# The maven version is REQUIRED, the otehrs may be elided.
# So, this only compares maven versions.  Note that 0.6.1.3 is later
# than 0.6.1 in this format.
modelVersionNewer () {
    local a=`echo "$1" | awk '{print $1}'`
    local b=`echo "$2" | awk '{print $1}'`

    # if destination isn't a real version, and proposed one is, say it's newer
    # since destination may be an empty graph or something like that.
    if echo "$a"|grep -q '^[0-9]'; then
        if echo "$b"|grep -q '^[0-9]'; then
            mavenVersionLessThan $b $a
        else
            return 0
        fi
    else
        echo "WARNING: Cannot compare anything to target version \"${a}\"" 1>&2
        return 1
    fi
}

# compare two version strings of the form  0.4.9 and 0.4.3
# NOTE: This ASSUMES it is given conforming NUMERIC Maven version strings
#  so be sure there is no "SNAPSHOT" or other junk in there.
mavenVersionLessThan () {
    echo "DEBUG: comparing \"${1}\" to \"${2}\" "
    set -o xtrace
    local a=(`echo "$1" | tr '.' ' '`)
    local b=(`echo "$2" | tr '.' ' '`)
    for i in 0 1 2 3 4; do
        echo "DEBUG: a[${i}]=\"${a[$i]}\", b[${i}]=\"${b[$i]}\""
        if [ -z "${b[$i]}" -a -n "${a[$i]}" ]; then
            set +o xtrace
            return 1
        elif [ -z "${a[$i]}" -a -n "${b[$i]}" ]; then
            set +o xtrace
            return 0
        # equal is not less than
        elif [ -z "${a[$i]}" -a -z "${b[$i]}" ]; then
            set +o xtrace
            return 1
        elif [ "${a[$i]}" -gt "${b[$i]}" ]; then
            set +o xtrace
            return 1
        elif [ "${a[$i]}" -lt "${b[$i]}" ]; then
            set +o xtrace
            return 0
        fi
    done
    # they are equal
    set +o xtrace
    return 1
}

# Usage: $0  jar|model
loadDataModelOntology () {
    local source=$1
    local result="${tmpDir}/model2"
    rm -f $result

    echo "INFO: Loading Data Model Ontology from $source"
    local status=""
    if status=`curl -k -s -S -u "$login" -o "$result" \
          -d action=load -d "source=$source" \
          --write-out '%{http_code}' "${repoURL}/repository/model"` ; then
        if [ "$status" = 200 -o "$status" = 201 ]; then
            echo "INFO: Done loading Data Model Ontology, HTTP status=$status"
        else
            echo "ERROR: Failed to load Data Model Ontology, HTTP status=$status"
            test -f $result && cat $result
            commandStatus=1
        fi
    else
        echo "ERROR: Failed sendign HTTP request to load data model ontology, see output above."
        commandStatus=1
    fi
}

# Usage: $0  uri
# Returns the useful data from the /internal service
# line contains URI and 2 quoted strings: loaded, and available (jar) versions
getInternalVersions () {
    local result="${tmpDir}/internal"
    rm -f $result

    if doInternal "$result" "${1}" ; then
        test -f "$result" && tail -n +2 "$result"
    else
        return 1
    fi
    return 0
}

# returns all internal graphs
# get first column of all retunred rows
getInternalGraphs () {
    local result="${tmpDir}/internal"
    rm -f $result

    if doInternal "$result" && test -f "$result" ; then
        awk "-F\t" 'NR > 1 {print $1}' < "$result" | tr '\n' ' '
    else
        return 1
    fi
}


# Usage: $0  filepath  [ uri ]
# writes output of /internal into given file paht.
doInternal () {
    local result=$1
    local uriArg="x=y"
    if [ $# -gt 1 ]; then
        uriArg="name=$2"
    fi
    local status=`curl -k -s -S -G -u "$login" -o "$result" -d "format=text/plain" \
      -d $uriArg --write-out "%{http_code}" ${repoURL}/repository/internal`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
        echo "/internal: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 1
    fi
    return 0
}

#usage: $0  uri
# forceibly loads an Internal graph from its source resource file
loadInternalGraph () {
    local result="${tmpDir}/internal"
    rm -f $result

    local status=`curl -k -s -S -u "$login" -o "$result" -d action=load \
      --data-urlencode name=$1 --write-out "%{http_code}" ${repoURL}/repository/internal`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "201" ]; then
        echo "/internal: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 1
    fi
    return 0
}

# compare internal version markers
# Usage: $0 "$available" "$loaded"
#  Typeical format of version is "$Id: upgrade.sh 19120 2015-04-30 05:13:28Z scheng $"
#  possibly including double-quotes
internalAvailableVersionIsNewer () {
    if vs=`getInternalVersions "$uri"`; then
        local loaded=`echo "$vs"|awk "-F\t" '{print $2}'`
        local available=`echo "$vs"|awk "-F\t" '{print $3}'`
        local lv=`getInternalRevision "$loaded"`
        local av=`getInternalRevision "$available"`
         
        # if second term is empty, first is always newer
        if [ -n "$av" -a -z "$lv" ]; then
            $Debug && echo "DEBUG: intVersion: No loaded version and avail=$av for $uri" 1>&2
            return 0
        elif [ -n "$lv" -a -n "$av" ]; then
            $Debug && echo "DEBUG: intVersion: Comparing loaded=$lv to avail=$av for $uri" 1>&2
            test "$lv" -lt "$av"
        else
            $Debug && echo "DEBUG: intVersion: Failed getting versions: loaded=$lv to avail=$av for $uri" 1>&2
            return 1
        fi
    else
        echo "WARNING: Failed to get version info for internal graph $uri" 1>&2
        return 1
    fi
}

# compare internal version markers
# Usage: $0 version-string
#  Typeical format of version is "$Id: upgrade.sh 19120 2015-04-30 05:13:28Z scheng $"
#  possibly including double-quotes
getInternalRevision () {
    if echo "$1" | egrep -q '^"?\$Id: '; then
        echo "$1" | awk '{print $3}'
    else
        return 1
    fi
}

# usage:  $0  view  query
#  returns text/plain results (ntriples for construct, TSV for select)
doTupleQuery () {
    doQueryInternal "$1" text/plain "$2"
}

# usage:  $0  view  query
#  returns true | false
doBooleanQuery () {
    doQueryInternal "$1" text/boolean "$2"
}

# SPARQL query prefixes
nl=$'\n'
queryPrefix="PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
PREFIX owl: <http://www.w3.org/2002/07/owl#>
PREFIX dcterms: <http://purl.org/dc/terms/>
PREFIX : <http://eagle-i.org/ont/repo/1.0/>"

# usage: $0  view  response-format  query
#  guts of SPARQL query request, result on stdout
doQueryInternal () {
    local view=$1
    local format=$2
    local query="${queryPrefix}${nl}$3"

    local expectStatus="200"
    local result="${tmpDir}/sparql"
    rm -f $result
    local status=""

    if status=`curl -k -s -S -u "$login" -o "$result" -F "format=$format"  \
      -F "query=$query" -F "view=$view" \
      --write-out "%{http_code}" ${repoURL}/repository/sparql` ; then
        if [ "$status" != "$expectStatus" ]; then
            echo "doQueryInternal: SPARQL Protocol request failed: HTTP status=$status cmd=$cmd " 1>&2
            test -f $result && cat $result 1>&2
            echo "" 1>&2
            echo "Query=" 1>&2
            echo "$query"|cat -n 1>&2
            return 1
        fi
    else
        echo "Failed sending HTTP /sparql request, curl status=$?" 1>&2
    fi

    # skip first line if select??
    if [ -f "$result" ]; then
        if echo "$query"|grep -qi "^SELECT"; then
            tail -n +2 "$result"
        else
            cat "$result"
        fi
    fi
    return 0
}

# usage:  graph-name  action  format-mimeType contents
#  uploads contents to named graph, action should be add or replace
doGraph () {
    local name=$1
    local action=$2
    local format=$3
    local content=$4
    local expectStatus="200"
    local status=""
    local result="${tmpDir}/init"
    rm -f $result
    local contentOpt="--form-string"

    # if the content arg is @<filename> for a real file path, use -F
    # otherwise the @ might be part of @PREFIX..
    if echo "$content"|grep -q '^@' && test -f `echo "$content"|sed -e 's/^@//'`; then
        contentOpt="-F"
    fi

    if status=`curl -k -s -S -u "$login" -o "$result" -F "format=$format"  \
      -F "name=$name" -F "action=$action" "$contentOpt" "content=$content" \
      --write-out "%{http_code}" ${repoURL}/repository/graph` ; then
        if [ "$status" != "$expectStatus" ]; then
            echo "doGraph: SPARQL Protocol request failed: HTTP status=$status cmd=$cmd " 1>&2
            test -f $result && cat $result 1>&2
            return 1
        fi
    else
        echo "doGraph: Failed sending HTTP /graph request, curl status=$?" 1>&2
    fi
}

# Usage: $0  construct-query  uri ...
# queries the initializer for given internal graph with CONSTRUCT query
queryInitializer () {
    local query="${queryPrefix}${nl}$1"
    local nameArgs=""
    shift
    while [ $# -gt 0 ]; do
        nameArgs="${nameArgs} -d name=${1}"
        shift
    done

    local result="${tmpDir}/init"
    rm -f $result

    if status=`curl -k -s -S -u "$login" -o "$result" \
      $nameArgs -d "action=query" -d "query=$query" \
      --write-out "%{http_code}" ${repoURL}/repository/internal` ; then
        if [ "$status" != "200" ]; then
            echo "queryInitializer: /internal request failed: HTTP status=$status cmd=$cmd " 1>&2
            test -f $result && cat $result 1>&2
            echo "" 1>&2
            return 1
        fi
        if [ -s "$result" ]; then
            cat "$result"
        else
            $Debug && echo "[DEBUG: QueryInitializer got empty results.]" 1>&2
        fi
        return 0
    else
        echo "Failed sending HTTP request /internal, curl status=$?" 1>&2
        return 1
    fi
}

# Force the repo server to decache all its in-memory caches of RDF data
doDecache () {
    local expectStatus="200"

    local status=`curl -k -s -S -u "$login" -d "action=decache" --write-out "%{http_code}" ${repoURL}/repository/internal`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "$expectStatus" ]; then
        echo "doDecache: Internal decache request failed: HTTP status=$status cmd=$cmd " 1>&2
        return 1
    fi
}


#---------------------------------------------------------------------------
# Compare loaded version of data model ontology against the version
# available in the webapp's jar -- load the one from the jar if newer,
# or if none appears to be loaded.
upgradeDataModelOntology () {
    # this will have versions 1 = loaded, 2 = jar, 3 = eiModel
    local versions=""
    if versions=`getDataModelVersions`; then
        local loaded=`selectModelVersion 1 "$versions"`
        local jar=`selectModelVersion 2 "$versions"`
        echo "INFO:  Loaded version = ${loaded}"
        if echo "$loaded"|grep -q "not loaded"||modelVersionNewer "$jar" "$loaded"; then
            echo "INFO: Loading new Data Model Ontology from internal jar, version: $jar"
            if loadDataModelOntology jar; then
                true
            else
                echo "ERROR: Failed loading data model ontology."
                commandStatus=1
            fi
        else
            echo "INFO: Data Model Ontology is already up to date, version: $loaded"
        fi
    else
        echo "ERROR: Failed getting data model ontology versions."
        commandStatus=1
    fi
}

# ---------------- Functions for testing ---------------------
# usage: $0  expect  fromversion  toversion
# test rig used with Ontology Version test
mvTest () {
    local expect=$1
    shift
    modelVersionNewer "$1" "$2"
    cmd=$?
    echo "  modelVersionNewer=$cmd for \"${1}\" NEWER THAN \"${2}\""
    if [ $cmd -ne $expect ]; then
        echo "FAIL: Last test failed."
        exit 1
    fi
}

# test the ontology version comparison
testVersionComparison () {
    echo "Testing Version Compare:"
    # these return 1
    mvTest 1 "0.6.4" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4 5727 2010-11-22 12:52:35" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4 5729 2010-11-22 12:52:35" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4 5727 2010-11-23 12:52:35" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.1" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.1.2" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4.1" "0.6.4.1 5727 2010-11-22 12:52:35"
    mvTest 1 "GIGO"                           "(not loaded or not in ontology)"
    # these will return 0
    mvTest 0 "0.6.4.2" "0.6.4.1 5727 2010-11-22 12:52:35"
    mvTest 0 "0.6.4.1" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.6.5" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.7" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.7.4" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "1" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "1.0.0" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.6.4 5727 2010-11-22 12:52:35" "(not loaded or not in ontology)"
    mvTest 0 "0.1" "(not loaded or not in ontology)"
    mvTest 0 "0" "(not loaded or not in ontology)"
    echo "Test Passed!"
    echo ""
}


# ------------ Special procedures and upgrades -----------------

# Dispatcher for special Property Groups access control upgrade (since 1.5MS1)
#
# Note that this *always* needs to be applied as it is not part of the initial dataset loaded int NG_Internal
# This checks for and adds, if necesary, the grants for the access
# control on properties.  These are required for REPO-157 implementaiton.
#
special_PropertyGroups () {
    local result=0
    local checkGrants="ASK WHERE { GRAPH <http://eagle-i.org/ont/repo/1.0/NG_Internal> {
      <http://eagle-i.org/ont/app/1.0/PropertyGroup_AdminData>
       <http://eagle-i.org/ont/repo/1.0/hasReadAccess>
        <http://eagle-i.org/ont/repo/1.0/Role_RNAV> ,
        <http://eagle-i.org/ont/repo/1.0/Role_Curator> ,
        <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> ,
        <http://eagle-i.org/ont/repo/1.0/Role_AdminReadOnly> ,
        <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
      <http://eagle-i.org/ont/app/1.0/PropertyGroup_ContactInformation>
       <http://eagle-i.org/ont/repo/1.0/hasReadAccess>
        <http://eagle-i.org/ont/repo/1.0/Role_RNAV> ,
        <http://eagle-i.org/ont/repo/1.0/Role_Curator> ,
        <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> ,
        <http://eagle-i.org/ont/repo/1.0/Role_AdminReadOnly> ,
        <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
      }}"

    if answer=`doBooleanQuery all "$checkGrants"`; then
        if [ "$answer" = true ]; then
            echo "INFO: Skipping step 1, property group grants are already in order."
        elif [ "$answer" = false ]; then
            echo "====>>> Step 1 - Adding property access grants."
            if cat <<EOF | doImport - grant abort no; then echo "INFO: imported property access grants."; else echo "ERROR: FAIL: @$LINENO Failed importing property grants"; commandStatus=1; result=1; fi
<http://eagle-i.org/ont/repo/1.0/NG_Internal> {
  <http://eagle-i.org/ont/app/1.0/PropertyGroup_AdminData> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> , <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> , <http://eagle-i.org/ont/repo/1.0/Role_AdminReadOnly> , <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
  <http://eagle-i.org/ont/app/1.0/PropertyGroup_ContactInformation> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> , <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> , <http://eagle-i.org/ont/repo/1.0/Role_AdminReadOnly> , <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
}
EOF
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL PROPERTY GROUPS step 1"
           commandStatus=1
           result=1
    fi
    return $result
}




# Check if roles are up to date
# Delete labels and descriptions of old roles
# Upload new labels and descriptions for old roles and new roles
special_1_6_MS6_update_roles () {
    echo "====>>> Step 1. Updating roles."
    ## New values - include updated label/descriptions, new roles and their grants
    local addme="${tmpDir}/addme-roles.txt"
    cat >"$addme"<<EOF
<http://eagle-i.org/ont/repo/1.0/Role_RNAV> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/Role> .
<http://eagle-i.org/ont/repo/1.0/Role_RNAV> <http://www.w3.org/2000/01/rdf-schema#label> "Level 2"^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_RNAV> <http://www.w3.org/2000/01/rdf-schema#comment> "eagle-i level 2 access: user can create Draft resources, send them to Curation, and return them from In Curation to Draft."^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_Curator> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/Role> .
<http://eagle-i.org/ont/repo/1.0/Role_Curator> <http://www.w3.org/2000/01/rdf-schema#label> "Level 4"^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_Curator> <http://www.w3.org/2000/01/rdf-schema#comment> "eagle-i level 4 access: curator with all privileges."^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/Role> .
<http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> <http://www.w3.org/2000/01/rdf-schema#label> "Level 1"^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> <http://www.w3.org/2000/01/rdf-schema#comment> "eagle-i level 1 access: user can create Draft resources and send them to Curation."^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/Role> .
<http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> <http://www.w3.org/2000/01/rdf-schema#label> "Level 3"^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> <http://www.w3.org/2000/01/rdf-schema#comment> "eagle-i level 3 access: curator."^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/Role> .
<http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> <http://www.w3.org/2000/01/rdf-schema#label> "Automated Agent"^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> <http://www.w3.org/2000/01/rdf-schema#comment> "eagle-i automated agent for data curation and data migration."^^<http://www.w3.org/2001/XMLSchema#string> .
EOF
    ### ASK
    local askQuery="ask  where { graph :NG_Internal {"`cat $addme`"}}"
    if answer=`doBooleanQuery all "$askQuery"`; then
	if [ "$answer" = false ]; then
        ### DELETE
	    local deleteme="${tmpDir}/deleteme-roles.txt"
	    local query='construct {?s rdfs:label ?label . ?s rdfs:comment ?desc} where {?s a :Role . ?s rdfs:label ?label . ?s rdfs:comment ?desc . filter (?s = :Role_RNAV || ?s = :Role_Curator || ?s = :Role_DraftEditor)}'
	    if doQueryInternal "all" text/plain "$query" > "$deleteme"; then
		if doGraph "$NG_Internal" delete "text/plain" "@$deleteme"; then
		    echo "INFO: OK - deleted obsolete role labels and descriptions"
		else
		    echo "ERROR: FAIL: Failed deleting obsolete labels and descriptions"
		    return 1
		fi
	    else
		echo "ERROR: FAIL: Failed querying roles"
		return 1
	    fi
        ### ADD
	    if doGraph "$NG_Internal" add "text/plain" "@$addme"; then
		echo "INFO: OK - added new role definitions"
	    else
		echo "ERROR: FAIL: Failed adding new role definitions"
		return 1
	    fi
	elif [ "$answer" = true ]; then
            echo "INFO: Skipping Step 1, roles are up to date"
	else
            echo "ERROR: Step 1 got unknown result = \"$answer\""
	    return 1
        fi
    else
	echo "ERROR: FAILED evaluating ask query in special_1_6_MS6_update_roles"
	return 1
    fi
    return 0
}

# Check if transitions are up to date
# Delete old transitions
# Upload new transitions
special_1_6_MS6_update_transitions () {
    echo "====>>> Step 2. Updating workflow transitions."
    local addme="${tmpDir}/addme-wft.txt"
    cat >"$addme"<<EOF
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://www.w3.org/2000/01/rdf-schema#label> "Default Create" .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://www.w3.org/2000/01/rdf-schema#comment> "Create new resources." .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_New> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> .
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/order> "100"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://www.w3.org/2000/01/rdf-schema#label> "Send to Curation" .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://www.w3.org/2000/01/rdf-schema#comment> "Transition a resource from Draft to In Curation; does not incur a graph change." .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_1> <http://eagle-i.org/ont/repo/1.0/order> "200"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://www.w3.org/2000/01/rdf-schema#label> "Publish" .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://www.w3.org/2000/01/rdf-schema#comment> "Transition a resource from In Curation to Published; resource is moved to the default published graph." .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_2> <http://eagle-i.org/ont/repo/1.0/order> "300"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://www.w3.org/2000/01/rdf-schema#label> "Return to Curation" .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://www.w3.org/2000/01/rdf-schema#comment> "Send a Published resource back to In Curation (ONLY for DEFAULT published graph and draft workspace)" .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace> .
<http://eagle-i.org/ont/repo/1.0/WFT_3> <http://eagle-i.org/ont/repo/1.0/order> "400"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://www.w3.org/2000/01/rdf-schema#label> "Return to Draft" .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://www.w3.org/2000/01/rdf-schema#comment> "Send a resource In Curation back to Draft; does not incur a graph change." .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> .
<http://eagle-i.org/ont/repo/1.0/WFT_4> <http://eagle-i.org/ont/repo/1.0/order> "500"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://www.w3.org/2000/01/rdf-schema#label> "Withdraw" .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://www.w3.org/2000/01/rdf-schema#comment> "Withdraw a Published resource; resource is moved from the default published graph to the default withdrawn graph." .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_5> <http://eagle-i.org/ont/repo/1.0/order> "600"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://www.w3.org/2000/01/rdf-schema#label> "Withdraw" .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://www.w3.org/2000/01/rdf-schema#comment> "Withdraw a resource In Curation; resource is moved to the default withdrawn graph." .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_6> <http://eagle-i.org/ont/repo/1.0/order> "700"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://www.w3.org/2000/01/rdf-schema#label> "Return to Draft" .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://www.w3.org/2000/01/rdf-schema#comment> "Send a Withdrawn resource back to draft (ONLY for DEFAULT withdrawn graph and draft workspace)" .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace> .
<http://eagle-i.org/ont/repo/1.0/WFT_7> <http://eagle-i.org/ont/repo/1.0/order> "800"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://www.w3.org/2000/01/rdf-schema#label> "No-op Draft" .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://www.w3.org/2000/01/rdf-schema#comment> "Can be used by applications to fine tune display of resources in Draft." .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_1> <http://eagle-i.org/ont/repo/1.0/order> "910"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://www.w3.org/2000/01/rdf-schema#label> "No-op Curation" .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://www.w3.org/2000/01/rdf-schema#comment> "Can be used by applications to fine tune display of resources In Curation." .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_2> <http://eagle-i.org/ont/repo/1.0/order> "920"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://www.w3.org/2000/01/rdf-schema#label> "No-op Published" .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://www.w3.org/2000/01/rdf-schema#comment> "Can be used by applications to fine tune display of Published resources." .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Published> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_3> <http://eagle-i.org/ont/repo/1.0/order> "930"^^<http://www.w3.org/2001/XMLSchema#integer> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://www.w3.org/2000/01/rdf-schema#label> "No-op Withdrawn" .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://www.w3.org/2000/01/rdf-schema#comment> "Can be used by applications to fine tune display of Withdrawn resources." .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> .
<http://eagle-i.org/ont/repo/1.0/WFT_0_4> <http://eagle-i.org/ont/repo/1.0/order> "940"^^<http://www.w3.org/2001/XMLSchema#integer> .
EOF
    ### ASK
    local askQuery="ask where { graph :NG_Internal {"`cat $addme`"}}"
    if answer=`doBooleanQuery all "$askQuery"`; then
        if [ "$answer" = false ]; then
        ### DELETE
	    local deleteme="${tmpDir}/deleteme-wft.txt"
	    local query='construct {?s ?p ?o} where {?s a :WorkflowTransition . ?s ?p ?o}'
	    if doQueryInternal "all" text/plain "$query" > "$deleteme"; then
		if doGraph "$NG_Internal" delete "text/plain" "@$deleteme"; then
		    echo "INFO: OK - deleted obsolete workflow transitions"
		else
		    echo "ERROR: FAIL: Failed deleting obsolete workflow transitions"
		    return 1
		fi
	    else
		echo "ERROR: FAIL: Failed querying transitions"
	    fi
        ### ADD
	    if doGraph "$NG_Internal" add "text/plain" "@$addme"; then
		echo "INFO: OK - added new workflow transitions"
	    else
		echo "ERROR: FAIL: Failed adding new workflow transitions"
		return 1
	    fi
	elif [ "$answer" = true ]; then
            echo "INFO: Skipping Step 2, workflow transitions are up to date"
	else
            echo "ERROR: Step 2 got unknown result = \"$answer\""
	    return 1
        fi
    else
	echo "ERROR: FAILED evaluating ask query in special_1_6_MS6_update_transitions"
	return 1
    fi
    return 0
}

# Check if global proxy graph is already defined
# Add if not
special_1_6_MS6_globals () {
    echo "====>>> Step 3. Adding Global Proxy graph."
    ## Definition of global proxy graph
    local addme="${tmpDir}/addme-globals.txt"
    cat >"$addme"<<EOF
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://www.w3.org/1999/02/22-rdf-syntax-ns#type> <http://eagle-i.org/ont/repo/1.0/NamedGraph> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://www.w3.org/2000/01/rdf-schema#label> "Global proxy workspace"^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://www.w3.org/2000/01/rdf-schema#comment> "Workspace for caching minimal information about global instances."^^<http://www.w3.org/2001/XMLSchema#string> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/hasAddAccess> <http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/hasRemoveAccess> <http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Authenticated> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/ngType> <http://eagle-i.org/ont/repo/1.0/NGType_Workspace> .
EOF
    ### ASK
    local askQuery="ask  where { graph :NG_Internal {:NG_GlobalProxy ?p ?o}}"
    if answer=`doBooleanQuery all "$askQuery"`; then
	if [ "$answer" = false ]; then
        ### ADD
	    if doGraph "$NG_Internal" add "text/plain" "@$addme"; then
		echo "INFO: OK - added Global Proxy graph"
	    else
		echo "ERROR: FAIL: Failed adding Global Proxy graph"
		return 1
	    fi
	elif [ "$answer" = true ]; then
            echo "INFO: Skipping Step 3, Global Proxy graph already exists"
	else
            echo "ERROR: Step 3 got unknown result = \"$answer\""
	    return 1
        fi
    else
	echo "ERROR: FAILED evaluating ask query in special_1_6_MS6_globals"
	return 1
    fi
    return 0
}

special_1_7_MS1_graph_permissions () {
    echo "====>>> Updating read permissions for graphs"
    local addme="${tmpDir}/addme-gp.txt"
    cat >"$addme"<<EOF
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> .
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_AdminReadOnly> .
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> .
<http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_CuratorWithRestrictions> .
EOF

    local deleteme="${tmpDir}/deleteme-gp.txt"
    cat >"$deleteme"<<EOF
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Authenticated> .
EOF
    if doGraph "$NG_Internal" delete "text/plain" "@$deleteme"; then
	echo "INFO: OK - deleted incorrect read permission for :NG_Withdrawn"
    else
	echo "ERROR: FAIL: Failed deleting incorrect read permission for :NG_Withdrawn"
	return 1
    fi
        ### ADD
    if doGraph "$NG_Internal" add "text/plain" "@$addme"; then
	echo "INFO: OK - added graph read permissions"
    else
	echo "ERROR: FAIL: Failed adding graph read permissions"
	return 1
    fi
    
    return 0
}

special_2_0_MS1_globals () {
    echo "====>>> Updating type and permissions of Global Proxy graph"
    local addme="${tmpDir}/addme-globals-2.txt"
    cat >"$addme"<<EOF
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/ngType> <http://eagle-i.org/ont/repo/1.0/NGType_Metadata> .
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Anonymous> .
EOF

    local deleteme="${tmpDir}/deleteme-globals-2.txt"
    cat >"$deleteme"<<EOF
<http://eagle-i.org/ont/repo/1.0/NG_GlobalProxy> <http://eagle-i.org/ont/repo/1.0/ngType> <http://eagle-i.org/ont/repo/1.0/NGType_Workspace> .
EOF
    if doGraph "$NG_Internal" delete "text/plain" "@$deleteme"; then
	echo "INFO: OK - deleted incorrect type for :NG_Globals"
    else
	echo "ERROR: FAIL: Failed deleting incorrect type for :NG_Globals"
	return 1
    fi
        ### ADD
    if doGraph "$NG_Internal" add "text/plain" "@$addme"; then
	echo "INFO: OK - added global proxy graph read permissions"
    else
	echo "ERROR: FAIL: Failed adding global proxy graph read permissions"
	return 1
    fi
    
    return 0
}

special_2_0_MS3_permissions () {
    echo "====>>> Updating read permissions for graph and workflow"
    local addme="${tmpDir}/addme-gp.txt"
    cat >"$addme"<<EOF
<http://eagle-i.org/ont/repo/1.0/WFT_0> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_AutoCurator> .
<http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Anonymous> .
EOF

  ### ADD
    if doGraph "$NG_Internal" add "text/plain" "@$addme"; then
	echo "INFO: OK - added graph read permissions"
    else
	echo "ERROR: FAIL: Failed adding graph read permissions"
	return 1
    fi
    
    return 0
}

# Dispatcher for special upgrades

special_all () {
    if special_1_6_MS6_update_roles ; then
	echo "INFO: Done with step 1"
    else
	return 1
    fi
    if special_1_6_MS6_update_transitions ; then
	echo "INFO: Done with step 2"
    else
	return 1
    fi
    if special_1_6_MS6_globals ; then
	echo "INFO: Done with step 3"
    else
	return 1
    fi
    if special_1_7_MS1_graph_permissions ; then
	echo "INFO: Done with step 3"
    else
	return 1
    fi
    if special_2_0_MS1_globals ; then
	echo "INFO: Done with step 4"
    else
	return 1
    fi    
    if special_2_0_MS3_permissions ; then
	echo "INFO: Done with step 5"
    else
	return 1
    fi     
		return 0
}

# ------------------ Main Thread ---------------------------------

# Sanity-check args
if [ "$1" = '--version' ]; then
    echo "$0 from release 4.5.1 SCM revision 20108"
    exit 0

# Secret test mode
elif [ "$1" = '--test' ]; then
    testVersionComparison
    exit 0

elif [ $# -lt 3 ]; then
    echo "ERROR: $usage"
    exit 1
fi

adminUser=$1
login="${1}:$2"
repoURL=$3

if echo "$repoURL"|egrep -qv '^https?:'; then
    echo "ERROR: Repository URL must be a valid http(s) URL: \"$repoURL\""
    exit 1
fi

commandStatus=0

# ------------ Upgrade Procedure Steps:

echo ""
echo "***** Checking and Upgrading Repository's Internal Graphs"
echo ""
if internalGraphs=`getInternalGraphs` ; then
    for uri in $internalGraphs ; do
        if internalAvailableVersionIsNewer "$uri" ; then
            if loadInternalGraph "$uri" ; then
                echo "INFO: Upgraded internal graph: $uri"
            else
                echo "WARNING: FAILED while trying to upgrade internal graph: $uri"
                commandStatus=1
            fi
        else
            echo "INFO: Skipping, internal graph is already up to date: $uri"
        fi
    done
else
    echo "WARNING: FAILED to get list of internal graphs, continuing.."
    commandStatus=1
fi

echo ""
echo "***** Checking and Upgrading Data Model Ontology"
echo ""
upgradeDataModelOntology

echo ""
echo "***** Checking and Upgrading Property Group Access Control Grants"
echo ""
  if special_PropertyGroups ; then
      echo ""
      echo "OK - Property Group Access Control grants all succeeded."
  else
      echo ""
      echo "FAIL - there were failures in uploading Property Group Access Control grants, YOU MUST RESTORE THE BACKUP."
      commandStatus=1
  fi

## XXX When there is an internal data migration procedure in the current release (e.g. roles, permissions,...),
## add a call to it here and change 'false' to 'true' in the next line:

## Leave a few releases enabled, in case of upgrades from several versions behind
if true; then
  echo -e "\n***** *S*P*E*C*I*A*L*  UPGRADES for 1.6MS6, 1.7MS1 and 2.0MS1\n"
  if special_all ; then
      echo ""
      echo "OK - specials all succeeded."
  else
      echo ""
      echo "FAIL - there were failures in specials, YOU MUST RESTORE THE BACKUP."
      commandStatus=1
  fi

else
    echo -e "\n***** There are no special upgrades for this release."
fi

exit $commandStatus
