package org.eaglei.repository;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import java.util.ArrayList;
import java.util.List;
import java.security.Principal;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.query.BindingSet;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandler;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BooleanQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;

import org.eaglei.repository.servlet.WithRepositoryConnection;

/**
 * Access control for the repository.
 *
 * The Access class is two things:
 *  1. Enumerated type describing the type of operation the user is allowed to do.
 *     These correspond to REPO constants (and maybe should move there?)
 *  2. A collection of static utility methods to answer access control questions
 *
 * Started April, 2010
 *
 * @author Larry Stone
 * @version $Id: $
 */
public enum Access
{
    /** Types of access to be granted */
    READ   (REPO.HAS_READ_ACCESS),
    ADD    (REPO.HAS_ADD_ACCESS),
    REMOVE (REPO.HAS_REMOVE_ACCESS),
    ADMIN  (REPO.HAS_ADMIN_ACCESS);

    // the related property URI for this access type
    private URI uri = null;

    // Session Attr to cache User object for authenticated principal
    private static final String S_USER = "org.eaglei.repository.Access.User";

    // Session Attr to save original client IP for later checks
    private static final String S_REMOTE_IP = "org.eaglei.repository.Access.REMOTE_IP";

    // SPARQL query to find grants on an instance (need to bind ?instance)
    private static final String getGrantsQuery =
        "SELECT DISTINCT * WHERE { \n"+
        "GRAPH ?graph { ?instance ?access ?agent } . \n"+
        "?access <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+"> \n"+
        "OPTIONAL { ?access <"+RDFS.LABEL+"> ?accessLabel }\n"+
        "OPTIONAL { ?agent <"+RDFS.LABEL+"> ?agentLabel }\n"+
        "OPTIONAL { ?agent <"+RDF.TYPE+"> ?agentType FILTER (?agentType = <"+REPO.ROLE+"> || ?agentType = <"+REPO.PERSON+">) \n"+
        "  OPTIONAL { ?agentType <"+RDFS.LABEL+"> ?agentTypeLabel }}}";

    private static Logger log = LogManager.getLogger(Access.class);

    /** Superuser role name, i.e. name of role that gets set by container authentication system */
    public static final String SUPERUSER_ROLE_NAME = "superuser";

    /** Constructor */
    Access(String uri)
    {
        this.uri = new URIImpl(uri);
    }

    /** Constructor */
    Access(URI uri)
    {
        this.uri = uri;
    }

    /**
     * Get the URI referenced by this access type.
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }

    /**
     * Predicate, does current user have the indicated
     * permission on this resource?  Answer is computed by asking:
     *
     *  1. Do we have the Superuser role?  If so, always grant permission.
     *  1. Is there a direct statement <resrc> :hasXAccess <user> ?
     *  2. Indirect through role,   <resrc> :hasXAccess <role>, and
     *     <role> either is or is superclass of a role <user> has.
     *  3. Resource grants access to :Role_Authenticated which is implicit
     *     in every authenticated user
     *
     * NOTE that if there is NO authenticated user, it checks for
     * :Role_Anonymous access since that's what getPrincipalURI() returns.
     *
     * ALSO must pass the type check, that the resource URI is
     * of the expected type, to prevent a permission on the wrong kind
     * of resource from being leveraged (unlikely as this might be).
     */

    // XXX maybe optimize later by storing "prepared" BooleanQuery,
    //  maybe in app context. it depends on the sesame repo obj.
    private static final String hasPermissionQuery = makeAccessQuery("resource", "ASK", null);

    // dataset of internal graphs plus NG_Users to get user's rdf:type
    private static DatasetImpl internalAccessGraphs = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        internalAccessGraphs.addDefaultGraph(REPO.NG_USERS);
    }

    /**
     * <p>hasPermission - general permission test.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param subject the object being tested for access
     * @param pred the type of access
     * @return a boolean, true if access was granted.
     */
    public static boolean hasPermission(HttpServletRequest request, Resource subject, Access pred)
    {
        if (isSuperuser(request)) {
            log.debug("Superuser elides check: hasPermission("+subject+", "+pred+") => true");
            return true;
        }

        try {
            URI pu = getPrincipalURI(request);
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // Parameterized prepared query - should be able to
            // re-use prepared query object so long as Repository doesn't change.
            // replacement because there's no equiv. of a Prepared Statement..
            // This is like an SQL prepared statement, setBinding plugs
            // values into the variables and seems to work..

            BooleanQuery q = rc.prepareBooleanQuery(QueryLanguage.SPARQL, hasPermissionQuery);
            q.setIncludeInferred(true);
            q.setDataset(internalAccessGraphs);
            q.clearBindings(); // needed if re-using query
            q.setBinding("user", pu);
            q.setBinding("access", pred.uri);
            q.setBinding("resource", subject);

            boolean result = q.evaluate();
            log.debug("hasPermission("+subject+", "+pred+", "+pu+") => "+result);
            return result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in access check: ",e);
        }
    }

    /**
     * Does current authenticated user have permission to modify the
     * User object associated with this username?
     * True if it matches the current logged-in user, or we are superuser.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal (i.e. RDBMS username, value of :hasPrincipal), a {@link java.lang.String} object.
     * @return a boolean, true if permission is gratned.
     */
    public static boolean hasPermissionOnUser(HttpServletRequest request, String username)
    {
        Principal p = request.getUserPrincipal();
        return isSuperuser(request) ||
               (username != null && p != null && username.equals(p.getName()));
    }

    /**
     * Filters results of query by what the current user has indicated permission
     * on.. Resource (URI) expected to be in  variable named "?{name}"  so
     * this SPARQL pattern group fragment (in "{ }") can be combined with the
     * rest of the query.
     *
     * See hasPermission() for algorithm to figure permission.  The only
     * difference is that this does NOT work for superuser.  (It could
     * be added if there is a need.)
     *
     * Results are returned by calling tuple query handler.
     * DO NOT call this if you are superuser! do test that outside of call.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param principal URI of user or role being checked for permission
     * @param name name of the variable in query, a {@link java.lang.String} object.
     * @param patternGroup query fragment
     * @param pred type of access being tested, a {@link org.eaglei.repository.Access} object.
     * @param handler a {@link org.openrdf.query.TupleQueryResultHandler} object.
     */
    public static void filterByPermission(HttpServletRequest request, URI principal, String name, String patternGroup, Access pred, TupleQueryResultHandler handler)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (principal == null)
                principal = getPrincipalURI(request);

            // Parameterized prepared query - should be able to
            // re-use prepared query object so long as Repository doesn't change.
            // replacement because there's no equiv. of a Prepared Statement..
            // This is like an SQL prepared statement, setBinding plugs
            // values into the variables and seems to work..

            // XXX maybe optimize later by storing "prepared" BooleanQuery,
            //  maybe in app context. it depends on the sesame repo obj.
            String qs = makeAccessQuery(name, "SELECT ?"+name+" WHERE", patternGroup);

            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setIncludeInferred(true);
            q.setDataset(internalAccessGraphs);
            q.clearBindings(); // needed if re-using query
            q.setBinding("user", principal);
            q.setBinding("access", pred.uri);
            q.evaluate(handler);

        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in access check: ",e);
        }
    }

    /**
     * Find (and create if necessary) the URI of the :Person object
     * for the current authenticated user, if any.
     * Returns the anonymous user when not authenticated.
     * This has to be synchronized because it is effectively performing
     * a test-and-set operation on the RDF database; if the user doesn't
     * exist it creates a new one, so another request at the same time would
     * create another :Person for the same user.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return the :Person object of current authenticated user, as a {@link org.openrdf.model.URI} object.
     */
    public static URI getPrincipalURI(HttpServletRequest request)
    {
        User u = getPrincipalUser(request);
        if (u == null) {
            log.debug("getPrincipalURI: Returning :Anonymous for unauthenticated user.");
            return REPO.ROLE_ANONYMOUS;
        }
        return u.getURI();
    }

    /**
     * invalidate the cache if the changed user is the same URI as cahced.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param u user to decache, a {@link org.eaglei.repository.User} object.
     */
    public static void decacheUser(HttpServletRequest request, User u)
    {
        HttpSession session = getValidatedSession(request);
        User cu = (User)session.getAttribute(S_USER);
        if (cu != null && cu.equals(u))
            session.removeAttribute(S_USER);
    }

    private static synchronized User getPrincipalUser(HttpServletRequest request)
    {
        // 1. if we're not authenticated, return anonymous.
        Principal p = request.getUserPrincipal();
        if (p == null)
            return null;

        // 2. check for cached answer first:
        //    make sure session isn't invalidated by (a) Remote IP addr check,
        //    and (b) generation count staleness
        // XXX consider setting session timeout interval.. can do it in web.xml
        String pname = p.getName();
        HttpSession session = getValidatedSession(request);
        if (DataRepository.getInstance().isSessionStale(session)) {
            session.removeAttribute(S_USER);
            log.debug("Invalidating cached user URI in stale session, principal="+pname);
        } else {
            User result = (User)session.getAttribute(S_USER);
            if (result != null) {
                log.debug("Returning cached user for principal="+pname+" from session context: "+result);
                return result;
            }
        }

        try {
            // 3. is there a :Person for this Principal or do we have to
            //    create a new one?
            User u = User.findUsername(request, pname);
            if (u != null) {

                // check if session has superuser but User obj hasn't, and update:
                if ((!u.hasRoleP(REPO.ROLE_SUPERUSER)) &&
                      request.isUserInRole(SUPERUSER_ROLE_NAME)) {
                    u.addRole(request, Role.find(request, REPO.ROLE_SUPERUSER), true);
                    // NOTE: we should NOT commit here in case there have
                    // already been some changes that may not want to commit..
                    // Trust that sooner or later transaction will commit,
                    // or else we'll get to try this again next request.
                    // NO: u.update(request);
                }
                session.setAttribute(S_USER, u);
                return u;
            }
             
            // 4. Unknown user, so we have to build a :Person for this principal
            // NOTE: the rdf:type statement (and any other publishable props)
            // must be publically visible, but relation to principal must NOT.
            // KLUDGE: final arg is to skip access checks sicne this is internal
            // AND may be part of an access check anyway so it could loop.
            u = User.create(request, pname, true);

            // propagate the container auth's "superuser" role into RDF,
            // so we only have to check RDF in the future:
            if (request.isUserInRole(SUPERUSER_ROLE_NAME))
                u.addRole(request, Role.find(request, REPO.ROLE_SUPERUSER), true);
            u.update(request);
            session.setAttribute(S_USER, u);
            log.info("Login created new User for principal="+pname+", as User="+u.toString());
            return u;
        } catch (ServletException e) {
            throw new InternalServerErrorException(e.getMessage(), e);
        }
    }

    // true if current auth'd user has superuser role either in RDF
    // assertion or in the container's authz.
    /**
     * <p>isSuperuser</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return a boolean, true if current authenticated user has Superuser (Admin) privilege.
     */
    public static boolean isSuperuser(HttpServletRequest request)
    {
        // 1. if we're not authenticated, can't be superuser.
        User u = getPrincipalUser(request);
        if (u == null)
            return false;
         
        // 2. check if User has the Superuser role in the RDF db
        return u.hasRoleP(REPO.ROLE_SUPERUSER);
    }

    /**
     * Utility to get a session and check that it came from the client
     * IP that originally created it, to prevent session hijacking
     * through cookie stealing (note..  this probably wouldn't help much
     * for attacker and victim behind the same proxy or NAT, but there's
     * only so much you can do.)
     *
     */
    private static HttpSession getValidatedSession(HttpServletRequest request)
    {
        HttpSession result = request.getSession(false);
        String remoteIP = request.getRemoteAddr();

        // need to create a new session, save a record of the creator's IP
        if (result == null) {
            result = request.getSession(true);
            result.setAttribute(S_REMOTE_IP, remoteIP);

        // confirm that accessor has same IP as creator.
        } else {
            String testIP = (String)result.getAttribute(S_REMOTE_IP);
            if (testIP == null) {
                log.debug("Initializing session's record of remote address, IP="+remoteIP);
                result.setAttribute(S_REMOTE_IP, remoteIP);
            } else if (!testIP.equals(remoteIP)) {
                // XXX call logout to wipe session now?
                log.error("POSSIBLE SESSION HIJACKING: session created by IP="+testIP+", but is being accessed by IP="+remoteIP);
                throw new BadRequestException("Authentication denied, this session may only be accessed from the address that created it. Please login again.");
            }
        }
        return result;
    }

    /**
     * Destroy current session and credentials (if possible)
     * most web browsers cache the HTTP Basic creds so user needs to
     * trash those explicitly right after running this.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     */
    public static void logout(HttpServletRequest request)
    {
        HttpSession s = request.getSession(false);
        if (s == null)
            log.debug("Logout finds no session to destroy!");
        else {
            s.invalidate();
            log.debug("Logout is destroying session ID="+s.getId());
        }
    }

    // cons up a sparql triple pattern to test access.
    // XXX note the parentheses don't balance
    private static String makeAccessQuery(String resourceName, String prologue, String patternGroup)
    {
        StringBuilder result = new StringBuilder();
        result.append(prologue).append(" { ");
        if (patternGroup != null)
            result.append(patternGroup);

        // user is in a role that has access
        result.append("{ { ?user <").append(REPO.HAS_ROLE)
              .append("> ?r . ?r a ?ar . ?").append(resourceName)
              .append(" ?access ?ar }\n");

        // user is a :Person and :Role_Authenticated has access
        result.append(" UNION { ?user a <").append(REPO.PERSON).append("> . ?")
              .append(resourceName).append(" ?access ?ar . <")
              .append(REPO.ROLE_AUTHENTICATED).append("> a ?ar }\n");

        // direct grant to user
        result.append(" UNION { ?").append(resourceName).append(" ?access ?user } } }");
        return result.toString();
    }

    /**
     * Remove specified grant of access from an instance.
     * Returns true if grant was there, false if not.
     * WARNING: You will need to commit() these changes to the repo connection!
     * XXX does NOT check if access URI is really an access URI..
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to remove access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if there was a grant to be removed.
     */
    public static boolean removeGrant(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        // sanity check: do not allow null, which is wildcard delete:
        if (instance == null || access == null || agent == null)
            throw new BadRequestException("removeGrant called with an illegal null URI.");
        if (hasPermission(request, instance, Access.ADMIN)) {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (rc.hasStatement(instance, access, agent, false, REPO.NG_INTERNAL)) {
                    rc.remove(instance, access, agent, REPO.NG_INTERNAL);
                    return true;
                }
                else
                    return false;
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in remove ACL: ",e);
            }
        } else
              throw new ForbiddenException("You are not allowed to change access controls on "+instance);
    }

    /**
     * Add the specified grant to the instance.  Requires ADMIN access.
     * WARNING: You will need to commit() these changes to the repo connection!
     * XXX FIXME: does NOT check if access URI is really an access URI..
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param instance subject from which to add access grant a {@link org.openrdf.model.URI} object.
     * @param agent principal to whom the access was granted, a {@link org.openrdf.model.URI} object.
     * @param access type of access, a {@link org.openrdf.model.URI} object.
     */
    public static void addGrant(HttpServletRequest request, URI instance, URI agent, URI access)
    {
        if (hasPermission(request, instance, Access.ADMIN)) {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (!rc.hasStatement(instance, access, agent, false, REPO.NG_INTERNAL)) {
                    rc.add(instance, access, agent, REPO.NG_INTERNAL);
                }
            } catch (OpenRDFException e) {
                log.error(e);
                throw new InternalServerErrorException("Failed in add ACL: ",e);
            }
        } else
              throw new ForbiddenException("You are not allowed to change access controls on "+instance);
    }

    /**
     * Get list of access grants on this instance
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri subject on which to find grants, a {@link org.openrdf.model.URI} object.
     * @return all grants in a {@link java.lang.Iterable} object, possibly empty.
     */
    public static Iterable<AccessGrant> getGrants(HttpServletRequest request, URI uri)
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            log.debug("SPARQL query to get grants = "+getGrantsQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, getGrantsQuery);
            q.setBinding("instance", uri);
            q.setDataset(internalAccessGraphs);
            grantHandler h = new grantHandler(rc, uri);
            q.evaluate(h);
            return h.result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query: ",e);
        }
    }

    private static final String allRolesQuery =
      "SELECT DISTINCT * WHERE { ?uri <"+RDFS.SUBCLASSOF+"> <"+REPO.ROLE+">\n"+
      " OPTIONAL { ?uri <"+RDFS.LABEL+"> ?label } \n"+
      " FILTER( ?uri != <"+REPO.ROLE+"> && ?uri != <"+REPO.ROLE_SUPERUSER+"> ) }";

    /**
     *  Get all roles described in the RDF - note this can be extended
     *  by each repository instance by adding statements.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     */
    public static Iterable<AccessGrant.Term> getAllRoles(HttpServletRequest request)
    {
        log.debug("getAllRolees query = "+allRolesQuery);
        return getAllTermsInternal(request, allRolesQuery);
    }

    private static final String allAccessesQuery =
      "SELECT DISTINCT * WHERE { "+
      "?uri <"+RDFS.SUBPROPERTYOF+"> <"+REPO.HAS_ANY_ACCESS+"> \n"+
      " OPTIONAL { ?uri <"+RDFS.LABEL+"> ?label }}";

    /**
     *  Get all access types described in the RDF - note this CAN be extended
     *  by each repository instance by adding statements but it is NOT
     *  likely since the practical implementation of access is hardcoded.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all access types in a {@link java.lang.Iterable} object.
     */
    public static Iterable<AccessGrant.Term> getAllAccesses(HttpServletRequest request)
    {
        log.debug("getAllAccesses query = "+allAccessesQuery);
        return getAllTermsInternal(request, allAccessesQuery);
    }

    // utility to fetch terms from RDF
    private static Iterable<AccessGrant.Term> getAllTermsInternal(HttpServletRequest request, String query)
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, query);
            q.setDataset(SPARQL.InternalGraphs);
            termHandler h = new termHandler();
            q.evaluate(h);
            return h.result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in query: ",e);
        }

    }

    // Term query result handler: stash tuple query results in List<AccessGrant.Term>
    // Looks for bindings of "uri" and "label"
    private static class termHandler extends TupleQueryResultHandlerBase
    {
        private List<AccessGrant.Term> result = new ArrayList<AccessGrant.Term>();

        // columns: namedGraphURI, namedGraphLabel, typeURI, typeLabel, anon
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value theURI = bs.getValue("uri");
            Value theLabel = bs.getValue("label");
            if (theURI == null || !(theURI instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'uri' was null or not a URI type in termHandler: "+theURI);
            else {
                result.add(new AccessGrant.Term((URI)theURI,
                    (theLabel != null && theLabel instanceof Literal) ? ((Literal)theLabel).getLabel() : null));
            }
        }
    }

    // Grant query result handler: stash tuple query results in List<Grant>
    // Looks for bindings of "uri" and "label"
    private static class grantHandler extends TupleQueryResultHandlerBase
    {
        private List<AccessGrant> result = new ArrayList<AccessGrant>();

        private RepositoryConnection rc = null;

        // subject of these grants
        private URI resource = null;

        protected grantHandler(RepositoryConnection rc, URI resource)
        {
            super();
            grantHandler.this.rc = rc;
            grantHandler.this.resource = resource;
        }

        // columns: namedGraphURI, namedGraphLabel, typeURI, typeLabel, anon
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value agent = bs.getValue("agent");
            Value agentLabel = bs.getValue("agentLabel");
            Value agentType = bs.getValue("agentType");
            Value agentTypeLabel = bs.getValue("agentTypeLabel");
            Value access = bs.getValue("access");
            Value accessLabel = bs.getValue("accessLabel");
            Value ggraph = bs.getValue("graph");
            if (agent == null || !(agent instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'agent' was null or not a URI type in grantHandler: "+agent);
            else if (access == null || !(access instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "The value for 'access' was null or not a URI type in grantHandler: "+access);
            else {
                // kludge: if agentType not set, grab repo:Agent
                if (agentType == null)
                    agentType = REPO.AGENT;
                // is this grant from the repo ontology, hence immutable??
                boolean builtin = ggraph != null && REPO.NG_REPO_ONTOLOGY.equals((URI)ggraph);
                log.debug("getGrants: Adding Grant(agent="+agent+
                    ", agentLabel="+agentLabel+", agentType="+agentType+
                    ", agentTypeLabel="+agentTypeLabel+
                    ", access="+access+", accessLabel="+accessLabel+
                    ", builtin="+builtin);
                result.add(new AccessGrant((URI)agent,
                    (agentLabel != null && agentLabel instanceof Literal) ? ((Literal)agentLabel).getLabel() : null,
                    (URI)agentType,
                    (agentTypeLabel != null && agentTypeLabel instanceof Literal) ? ((Literal)agentTypeLabel).getLabel() : null,
                    (URI)access,
                    (accessLabel != null && accessLabel instanceof Literal) ? ((Literal)accessLabel).getLabel() : null,
                    builtin));
            }
        }
    }
}
