package org.eaglei.repository.util;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import java.io.IOException;
import java.io.OutputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Date;
import java.util.TimeZone;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.datatype.DatatypeConstants;

import org.openrdf.query.Dataset;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.datatypes.XMLDatatypeUtil;

import org.eaglei.repository.status.BadRequestException;

/**
 * Utility methods
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Utils
{
    private static Logger log = LogManager.getLogger(Utils.class);

    /**
     * The canonical file copy loop.
     * Closes streams when done.
     *
     * @param in source, a {@link java.io.InputStream} object.
     * @param out destination, a {@link java.io.OutputStream} object.
     * @throws java$io$IOException if anything goes wrong.
     */
    public static void copyStream(InputStream in, OutputStream out)
        throws IOException
    {
        byte[] buf = new byte[8192];
        int len;
        while ((len = in.read(buf)) > 0) {
            out.write(buf, 0, len);
        }
        in.close();
        out.close();
    }

    /**
     * Formatted human-readable view of Dataset contents which works
     * even when some members are null.  Needed because Sesame's toString()
     * breaks on a null URI, even though that is allowed and necessary(!)
     * for temporary fix of inferencing context problem..
     *
     * @param ds a {@link org.openrdf.query.Dataset} object.
     * @return multi-line prettyprint in a {@link java.lang.String} object.
     */
    public static String prettyPrint(Dataset ds)
    {
        StringBuilder result = new StringBuilder();

        for (URI u : ds.getDefaultGraphs())
            result.append("FROM ").append(u == null ? "{null}" : "<"+u.toString()+">").append("\n");
        for (URI u : ds.getNamedGraphs())
            result.append("FROM NAMED ").append(u == null ? "{null}" : "<"+u.toString()+">").append("\n");
        if (result.length() > 0)
            result.deleteCharAt(result.length()-1);
        return result.toString();
    }

    /**
     * Test whether a string is a well-formed-enough absolute URI that
     * it will not cause createURI to fail.
     *
     * @param s possible URI
     * @return a boolean, true if s is a valid URI.
     */
    public static boolean isValidURI(String s)
    {
        try {
            new URIImpl(s);
            return true;
        } catch (IllegalArgumentException e) {
            return false;
        }
    }

    /**
     * Translates string to URI with sanity check.
     *
     * @param s possible URI
     * @param argName name of parameter for error messages
     * @param required true if arg is required
     * @return a URI
     */
    public static URI parseURI(String s, String argName, boolean required)
    {
        if (s == null) {
            if (required)
                throw new BadRequestException("Missing required argument: "+argName);
            else
                return null;
        }
        try {
            return new URIImpl(s);
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Argument '"+argName+"' is not a valid URI: "+e);
        }
    }

    /**
     * Parse a query arg value that is supposed to belong to an Enum
     * set.  If a value is required, throw the BadRequest exception if
     * not present, and in any case throw BadRequest if it is not
     * a valid enumerated value.
     * NOTE: This _ought_ to be shared in utils or some such but then all
     * the arg enums would haev to be public, and they shouldn't be.
     * @param et the enumerated type class
     * @param name value of the argument
     * @param argName name of the argument, e.g. parameter name
     * @param required when true this arg MUST have a value.
     * @param defaultValue default to subsitute when arg is not required and missing
     * @return the enum or null only if required is false and there is no value.
     * @throws BadRequestException if arg is missing or illegal
     */
    public static Enum parseKeywordArg(Class et, String name, String argName, boolean required, Enum defaultValue)
    {
        if (name == null) {
            if (required)
                throw new BadRequestException("Missing required argument: "+argName);
            else
                return defaultValue;
        }
        try {
            return Enum.valueOf(et, name);
        } catch (IllegalArgumentException e) {
            try {
                throw new BadRequestException("Illegal value for '"+argName+"', must be one of: "+
                  Arrays.deepToString((Enum[])et.getMethod("values").invoke(null)));
            } catch (NoSuchMethodException ee) {
                log.error("Failed generating arglist for exception, ",ee);
                throw new BadRequestException("Illegal value for '"+argName+"'");
            } catch (IllegalAccessException ee) {
                log.error("Failed generating arglist for exception, ",ee);
                throw new BadRequestException("Illegal value for '"+argName+"'");
            } catch (java.lang.reflect.InvocationTargetException ee) {
                log.error("Failed generating arglist for exception, ",ee);
                throw new BadRequestException("Illegal value for '"+argName+"'");
            }
        }
    }

    // values of a boolean arg for use by special keyword parser
    private enum BooleanArg
    {
        true1 ("yes", true),
        true2 ("true", true),
        true3 ("t", true),
        false1 ("no", false),
        false2 ("false", false),
        false3 ("nil", false);

        String label = null;
        boolean truth = false;

        private BooleanArg(String nm, boolean v)
        {
            label = nm;
            truth = v;
        }
        public String toString()
        {
            return label;
        }
    }

    /**
     * Special query arg parser for args with boolean values.
     * If a value is "required", throw the BadRequest exception if
     * not present, and in any case throw BadRequest if it is not
     * a valid enumerated value.  Use the enum as a convenient table
     * of strings and values.
     * @param name value of the argument
     * @param argName name of the argument, e.g. parameter name
     * @param required when true this arg MUST have a value.
     * @param defaultValue default to subsitute when arg is not required and missing
     * @return the boolean result of parsed or defaulted arg
     * @throws BadRequestException if arg is missing or illegal
     */
    public static boolean parseBooleanParameter(String name, String argName, boolean required, boolean defaultValue)
    {
        BooleanArg b = null;
        if (name == null) {
            if (required)
                throw new BadRequestException("Missing required argument: "+argName);
            else
                b = defaultValue ? BooleanArg.true1 : BooleanArg.false1;
        } else {
            for (BooleanArg ba : BooleanArg.values()) {
                if (ba.label.equals(name)) {
                    b = ba;
                    break;
                }
            }
            if (b == null)
                throw new BadRequestException("Illegal value for '"+argName+"', must be one of: "+
                  Arrays.deepToString(BooleanArg.values()));
        }
        return b.truth;
    }

    public static XMLGregorianCalendar parseXMLDate(String raw)
    {
        XMLGregorianCalendar result = null;
        try {
            result = XMLDatatypeUtil.parseCalendar(raw);
            if (result.getDay() == DatatypeConstants.FIELD_UNDEFINED)
                throw new BadRequestException("This date/time argument is incomplete, it must include at least complete date: "+raw);
            if (result.getHour() == DatatypeConstants.FIELD_UNDEFINED)
                result.setHour(0);
            if (result.getMinute() == DatatypeConstants.FIELD_UNDEFINED)
                result.setMinute(0);
            if (result.getSecond() == DatatypeConstants.FIELD_UNDEFINED)
                result.setSecond(0);
            if (result.getTimezone() == DatatypeConstants.FIELD_UNDEFINED) {
                TimeZone td = TimeZone.getDefault();
                int tzMinutes = td.getRawOffset()/(1000*60);
                if (td.inDaylightTime(new Date()))
                    tzMinutes += td.getDSTSavings()/(1000*60);
                // XXX only for TZ debugging
                //log.debug("Tweaking by default timezone minutes = "+tzMinutes);
                result.setTimezone(tzMinutes);
            }
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Illegal date format in this date/time argument: "+e.toString());
        }
        if (log.isDebugEnabled())
            log.debug("parseXMLDate: given string = \""+raw+"\", interpreted as = "+result+
                      ", as java.util.Date = "+result.toGregorianCalendar().getTime().toString());
        return result;
    }

}
