package org.eaglei.repository;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.ConflictException;
import org.eaglei.repository.status.BadRequestException;
import java.util.List;
import java.util.Date;
import java.util.Set;
import java.util.HashSet;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Value;
import org.openrdf.model.Statement;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.Dataset;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.servlet.WithRepositoryConnection;

/**
 * Object Model for "fake workflow", RDF annotations on a resource
 * instance that takes the place of a full workflow implementation
 * for "cycle 1".
 *
 * A FakeFlow instance is associated with a resource instance.
 * All of the statements implementing workflow are in the Metadata graph.
 *
 * @author Larry Stone
 * Started May 17, 2010
 * @version $Id: $
 */
public class FakeFlow extends TupleQueryResultHandlerBase
{
    private static Logger log = LogManager.getLogger(FakeFlow.class);

    // default fwf state for newly-created instances
    private static final URI DEFAULT_STATE = REPO.WFS_DRAFT;

    // cache of all states - may not be needed after all.
    private static Set<URI> wfStates = null;

    /* immutable, eagle-i resource instance to which workflow state applies */
    private URI resource = null;

    // workflow state
    private URI state = null;
    private String stateLabel = null;

    // current "owner" of the workflow on this instance, i.e. claimant.
    private URI owner = null;
    private String ownerLabel = null;

    // initialize static set of WFS URIs that needs RDF conn.
    private static void init(RepositoryConnection rc)
    {
        if (wfStates == null) {
            wfStates = new HashSet<URI>();
            try {
                RepositoryResult<Statement> rr = null;
                try {
                    rr = rc.getStatements(null, RDF.TYPE, REPO.WORKFLOW_STATE, false);
                    while (rr.hasNext()) {
                        Resource sub = rr.next().getSubject();
                        if (sub instanceof URI)
                            wfStates.add((URI)sub);
                        else
                            log.warn("Got non-URI in workflow state query: "+sub);
                    }
                } finally {
                    rr.close();
                }
            } catch (RepositoryException e) {
                log.error("Failed finding Workflow State instances.");
            }
        }
    }

    private FakeFlow(URI resource)
    {
        super();
        this.resource = resource;
    }

    /**
     * Predicate that is true if uri is one of the workflow states.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if given URI is subject of a Workflow state.
     */
    public static boolean isWorkflowState(HttpServletRequest request, URI uri)
    {
        return isWorkflowState(WithRepositoryConnection.get(request), uri);
    }

    /**
     * Predicate that is true if uri is one of the workflow states.
     * Optional version called with RepositoryConnection instead of Request.
     *
     * @param rc a {@link org.openrdf.repository.RepositoryConnection} object.
     * @param uri a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if given URI is subject of a Workflow state.
     */
    public static boolean isWorkflowState(RepositoryConnection rc, URI uri)
    {
        init(rc);
        return wfStates.contains(uri);
    }

    /**
     * Predicate that is true if this is a predicate managed by fake workflow
     *
     * @param uri a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if URI is one of the fake workflow predicates.
     */
    public static boolean isWorkflowPredicate(URI uri)
    {
        return REPO.HAS_WORKFLOW_STATE.equals(uri) ||
               REPO.HAS_WORKFLOW_OWNER.equals(uri);
    }


    // get workflow info about an instance
    private static final String fwfQuery =
        "SELECT * WHERE { \n"+
          "?instance <"+REPO.HAS_WORKFLOW_STATE+"> ?state . \n"+
          "?state <"+RDFS.LABEL+"> ?stateLabel \n"+
          "OPTIONAL { ?instance <"+REPO.HAS_WORKFLOW_OWNER+"> ?owner . \n"+
          " OPTIONAL { ?owner <"+RDFS.LABEL+"> ?ownerLabel } } }";

    /**
     * Get current workflow on a resource if any, or null if not found.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param resource the subject {@link org.openrdf.model.URI} of the eagle-i resource instance.
     * @return workflow as a {@link org.eaglei.repository.FakeFlow} object, or null if none found.
     */
    public static FakeFlow find(HttpServletRequest request, URI resource)
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            Dataset ds = SPARQL.getInternalAndMetadataGraphs(request);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, fwfQuery);
            q.setBinding("instance", resource);
            q.setDataset(ds);
            q.setIncludeInferred(false);
            //XXX log.debug("FakeFlow SPARQL query = "+fwfQuery);
            //XXX log.debug("Dataset for FakeFlow SPARQL query = "+Utils.prettyPrint(ds));
            FakeFlow result = new FakeFlow(resource);
            q.evaluate(result);
            if (result.state == null) {
                log.debug("NO fake workflow properties found on Resource Instance="+resource);
                return null;
            } else {
                log.debug("Found fake workflow state="+result.state+" for Resource Instance="+resource);
                return result;
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new InternalServerErrorException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException(e);
        }
    }


    /**
     * {@inheritDoc}
     *
     * Query handler to pick up what SHOULD be the single result..
     */
    public void handleSolution(BindingSet bs)
        throws TupleQueryResultHandlerException
    {
        Value state = bs.getValue("state");
        Value stateLabel = bs.getValue("stateLabel");
        Value owner = bs.getValue("owner");
        Value ownerLabel = bs.getValue("ownerLabel");
        if (state != null && state instanceof URI)
            this.state = (URI)state;
        else
            log.warn("Got strange result with no State for uri="+resource);
        if (stateLabel != null && stateLabel instanceof Literal)
            this.stateLabel = ((Literal)stateLabel).getLabel();
        if (owner != null && owner instanceof URI)
            this.owner = (URI)owner;
        if (ownerLabel != null && ownerLabel instanceof Literal)
            this.ownerLabel = ((Literal)ownerLabel).getLabel();
    }

    /**
     * Create fake workflow state for given resource.
     * There MUST NOT be a wf state already.
     * Initial state is the default, initial owner is current user.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param resource the subject {@link org.openrdf.model.URI} of the eagle-i resource instance.
     * @return new workflow as a {@link org.eaglei.repository.FakeFlow} object, NEVER null.
     */
    public static FakeFlow create(HttpServletRequest request, URI resource)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // sanity check: do not add an extra workflow state
            if (rc.hasStatement(resource, REPO.HAS_WORKFLOW_STATE, null, false, REPO.NG_METADATA))
                throw new InternalServerErrorException("This resource instance already has a workflow state: "+resource);

            FakeFlow result = new FakeFlow(resource);
            result.state = DEFAULT_STATE;
            result.owner = Access.getPrincipalURI(request);
            rc.add(resource, REPO.HAS_WORKFLOW_STATE, result.state, REPO.NG_METADATA);
            rc.add(resource, REPO.HAS_WORKFLOW_OWNER, result.owner, REPO.NG_METADATA);
            log.debug("Created FakeFlow entries for resource instance= "+resource+", state="+result.state+", owner="+result.owner);
            return result;
        } catch (RepositoryException e) {
            log.error("Failed creating FakeFlow entry: ",e);
            throw new InternalServerErrorException("Failed creating FakeFlow: ",e);
        }
    }

    /**
     * Remove this workflow state from its resource instance.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     */
    public void clear(HttpServletRequest request)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(resource, REPO.HAS_WORKFLOW_STATE, null, REPO.NG_METADATA);
            rc.remove(resource, REPO.HAS_WORKFLOW_OWNER, null, REPO.NG_METADATA);
        } catch (RepositoryException e) {
            log.error("Failed clearing FakeFlow: ",e);
            throw new InternalServerErrorException("Failed clearing FakeFlow: ",e);
        }
    }

    /**
     * Advance workflow state to the given new state.  This
     * also contains any side effects, hardcoded -- e.g.
     * moving instance to the "published" graph.
     * Also must hardcode all checks, e.g. cannot jump from draft to publsihed.
     * NOTE: this DOES NOT commit, caller must call rc.commit()
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param newState {@link org.openrdf.model.URI} of new workflow state
     */
    public void promote(HttpServletRequest request, URI newState)
    {
        try {
            // crude access check
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (!Access.isSuperuser(request))
                throw new ForbiddenException("Fakeflow promotion is only permitted to Administrators.");
             
            // "Promote" to Draft (usually demotion from Curation):
            //  - check: must be in Curation
            if (REPO.WFS_DRAFT.equals(newState)) {
                if (!REPO.WFS_CURATION.equals(state))
                    throw new ConflictException("May not make this transition from current Fakeflow state.");

            // Promote to Curation:
            //  - check: must be in draft
            } else if (REPO.WFS_CURATION.equals(newState)) {
                if (!REPO.WFS_DRAFT.equals(state))
                    throw new ConflictException("May not make this transition from current Fakeflow state.");
             
            // Promote to Published:
            // - must be in Curation or Withdrawn
            // - side effect: move instance properties to Published graph
            } else if (REPO.WFS_PUBLISHED.equals(newState)) {
                if (!(REPO.WFS_CURATION.equals(state) ||
                      REPO.WFS_WITHDRAWN.equals(state)))
                    throw new ConflictException("May not make this transition from current Fakeflow state.");
                moveToGraph(rc, REPO.NG_PUBLISHED);
             
            // Promote to Withdrawn:
            // - must be in Published
            // - side effect: move instance properties to Withdrawn graph
            } else if (REPO.WFS_WITHDRAWN.equals(newState)) {
                if (!REPO.WFS_PUBLISHED.equals(state))
                    throw new ConflictException("May not make this transition from current Fakeflow state.");
                moveToGraph(rc, REPO.NG_WITHDRAWN);
             
            } else {
                throw new BadRequestException("There are no transitions to the given workflow state.");
            }
             
            // Common side-effects:
            //  - clear the owner
            //  - update the state
            rc.remove(resource, REPO.HAS_WORKFLOW_STATE, null, REPO.NG_METADATA);
            rc.remove(resource, REPO.HAS_WORKFLOW_OWNER, null, REPO.NG_METADATA);
            rc.add(resource, REPO.HAS_WORKFLOW_STATE, newState, REPO.NG_METADATA);
            new Provenance(resource).setModified(request, new Date());
            log.debug("Completed transition to workflow state: "+newState);

        } catch (RepositoryException e) {
            String msg = "Failed in promote() for resource inst="+resource;
            log.error(msg);
            throw new InternalServerErrorException(msg,e);
        }
    }

    /**
     * Lets user assert a claim on this workflow state.
     * Just clears the old claim if user URI is null.
     *
     * NOTE: this DOES NOT commit, caller must call rc.commit()
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param newOwner new owner as a :Person {@link org.openrdf.model.URI} object.
     */
    public void claim(HttpServletRequest request, URI newOwner)
    {
        // crude access check
        if (!Access.isSuperuser(request))
            throw new ForbiddenException("Fakeflow claim is only permitted to Administrators.");

        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (rc.hasStatement(resource, REPO.HAS_WORKFLOW_OWNER, null, false, REPO.NG_METADATA))
                rc.remove(resource, REPO.HAS_WORKFLOW_OWNER, null, REPO.NG_METADATA);
            if (newOwner != null)
                rc.add(resource, REPO.HAS_WORKFLOW_OWNER, newOwner, REPO.NG_METADATA);
        } catch (RepositoryException e) {
            String msg = "Failed in claim() for resource inst="+resource;
            log.error(msg);
            throw new InternalServerErrorException(msg,e);
        }
    }

    // move resource's properties to new named graph
    // (i.e. to implement different access control and query response)
    private void moveToGraph(RepositoryConnection rc, URI newGraph)
    {
        URI homeGraph = null;
        try {

            // 1. get resource's home graph, ideally where its rdf:type
            //    statements live - but be careful to avoid inferred ones!
            RepositoryResult<Statement> rr = null;
            try {
                rr = rc.getStatements(resource, RDF.TYPE, null, false);
                while (rr.hasNext()) {
                    Statement s = rr.next();
                    Resource ctx = s.getContext();

                    // XXX FIXME: under current sesame-FC inferencing kludge,
                    // skip all types in null context since they are inferred.
                    log.debug("Found statement: "+resource+" rdf:type "+s.getObject()+", in graph "+ctx);
                    if (ctx != null && ctx instanceof URI) {
                        homeGraph = (URI)ctx;
                        break;
                    }
                }
            } finally {
                rr.close();
            }
            if (homeGraph == null) {
                log.error("Failed to find home graph for resource instance="+resource);
                throw new InternalServerErrorException("Failed to find home graph for resource instance="+resource);
            } else if (homeGraph.equals(newGraph)) {
                log.warn("Tried to move resource inst="+resource+" to the same named graph: "+homeGraph);
                return;
            }

            // 2. Move all statements to new graph.
            //   Need to create a List because removing them might
            //   mess up RepoResult's lazy evaluation.
            List<Statement> stms = null;
            try {
                rr = rc.getStatements(resource, null, null, false, homeGraph);
                stms = rr.asList();
            } finally {
                rr.close();
            }
            for (Statement s : stms) {
                rc.remove(s, homeGraph);
                rc.add(s, newGraph);
            }
        } catch (RepositoryException e) {
            String msg = "Failed moving statments for resource inst="+resource+", from="+homeGraph+", to="+newGraph;
            log.error(msg);
            throw new InternalServerErrorException(msg,e);
        }
    }

    /**
     * <p>Getter for the field <code>resource</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getResource()
    {
        return resource;
    }
    /**
     * <p>Getter for the field <code>state</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getState()
    {
        return state;
    }
    /**
     * <p>Getter for the field <code>owner</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getOwner()
    {
        return owner;
    }
    /**
     * <p>Getter for the field <code>stateLabel</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getStateLabel()
    {
        return stateLabel == null ? state.getLocalName() : stateLabel;
    }
    /**
     * <p>Getter for the field <code>ownerLabel</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getOwnerLabel()
    {
        return ownerLabel == null ? owner.getLocalName() : ownerLabel;
    }
}
