package org.eaglei.repository.servlet;

import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.sql.Connection;
import java.sql.SQLException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.naming.NamingException;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileItemHeaders;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.memory.MemoryStore;
import org.openrdf.model.Literal;
import org.openrdf.model.BNode;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFParser;
import org.openrdf.rio.RDFHandler;
import org.openrdf.rio.helpers.RDFHandlerWrapper;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.repository.RepositoryException;

import org.eaglei.repository.Access;
import org.eaglei.repository.Formats;
import org.eaglei.repository.DataRepository;
import org.eaglei.repository.NamedGraph;
import org.eaglei.repository.Provenance;
import org.eaglei.repository.User;
import org.eaglei.repository.View;
import org.eaglei.repository.admin.AuthUser;
import org.eaglei.repository.rid.RIDGenerator;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Structured export and import of resource instances OR user instances.
 * The same servlet implementation is called by both /export and /import,
 * they should set the init parameter "import" or "export" appropriately.
 * e.g.
 * <pre>
 * <servlet>
 *   <servlet-name>Export</servlet-name>
 *   <servlet-class>org.eaglei.repository.servlet.ImportExport</servlet-class>
 *   <init-param>
 *     <param-name>export</param-name>
 *     <param-value>true</param-value>
 *   </init-param>
 * </servlet>
 * </pre>
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class ImportExport extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(ImportExport.class);

    // values of 'duplicate' arg - must be public for parseKeyword.
    public enum DuplicateArg { abort, ignore, replace };

    // values of 'graph' arg
    public enum NewGraphArg { abort, create };

    // values of 'type' arg
    public enum TypeArg { resource, user };

    // special property only used in export and import of users to hold
    // RDBMS password entry
    private static final URI EXPORT_AUTH_PASSWORD = new URIImpl(REPO.NAMESPACE + "exportAuthPassword");

    // special property only used in export and import of users to
    // indicate which type of user authentication implements this user
    private static final URI EXPORT_AUTH_TYPE = new URIImpl(REPO.NAMESPACE + "exportAuthType");

    // special indtance only used in export and import of users to
    // indicate the builtin authentication (RDBMS) type
    private static final URI EXPORT_AUTH_TYPE_BUILTIN = new URIImpl(REPO.NAMESPACE + "exportAuthType_Builtin");

    // get contents of AuthUser entries to rebuild the RDBMS
    private static final String importUserGetAuthUsers =
        "SELECT * WHERE { \n"+
        "GRAPH <"+REPO.NG_INTERNAL+"> { "+
        " ?authUser <"+EXPORT_AUTH_TYPE+"> <"+EXPORT_AUTH_TYPE_BUILTIN+">; "+
        " <"+REPO.HAS_PRINCIPAL_NAME+"> ?username; "+
        " <"+EXPORT_AUTH_PASSWORD+"> ?password "+
        " OPTIONAL { ?authUser <"+REPO.HAS_ROLE+"> ?su "+
        "   FILTER(?su = <"+REPO.ROLE_SUPERUSER+">)}}}";

    // get RDF statements from user import
    private static final String importUserGetStatements =
        "SELECT ?g ?s ?p ?v ?username WHERE { \n"+
        "GRAPH <"+REPO.NG_USERS+"> { ?s a <"+REPO.PERSON+"> }\n"+
        "OPTIONAL { GRAPH <"+REPO.NG_INTERNAL+"> { "+
        "    ?s <"+REPO.HAS_PRINCIPAL_NAME+"> ?username}}\n"+
        "GRAPH ?g {?s ?p ?v \n"+
        "  FILTER((?g = <"+REPO.NG_INTERNAL+"> && isURI(?s) &&\n"+
        "         (?p = <"+REPO.HAS_ROLE+"> || ?p = <"+REPO.HAS_PRINCIPAL_NAME+">)) ||\n"+
        "        (?g = <"+REPO.NG_USERS+">))}}";

    // ----------- Instance variables

    // identify which kind of servlet this is
    private boolean isImport = false;
    private boolean isExport = false;

    /**
     * {@inheritDoc}
     *
     * Configure this servlet as either import or export mode.
     */
    public void init(ServletConfig sc)
        throws ServletException
    {
        super.init(sc);

        // if this is not the export servlet, don't implement GET:
        isExport = sc.getInitParameter("export") != null;
        isImport = sc.getInitParameter("import") != null;

        if (!(isExport || isImport))
            log.error("Servlet was initialized without either import or export mode set, THIS IS BAD.");
    }

    /**
     * {@inheritDoc}
     *
     * GET the contents of a graph - for EXPORT only
     * Query Args:
     *  - format = MIME type (overrides content-type)
     *  - view = view to query (mutually excl. w/workspace)
     *  - workspace = workspace to query (mutually excl. w/workspace)
     *     (NOTE: default to view = USER for type=resource)
     *  - type=(resource|user) - what to import
     *  - include="URI ..." or "username ..." if type=user
     *  - exclude="URI ..." or "username ..." if type=user
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        if (!isExport)
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "GET is not implemented by this service");

        String rawFormat = null;
        String rawView = null;
        String rawWorkspace = null;
        String rawType = null;
        String include = null;
        String exclude = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("format"))
                        rawFormat = item.getString();
                    else if (ifn.equals("view"))
                        rawView = item.getString();
                    else if (ifn.equals("workspace"))
                        rawWorkspace = item.getString();
                    else if (ifn.equals("type"))
                        rawType = item.getString();
                    else if (ifn.equals("include"))
                        include = item.getString();
                    else if (ifn.equals("exclude"))
                        exclude = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            rawFormat = request.getParameter("format");
            rawView = request.getParameter("view");
            rawWorkspace = request.getParameter("workspace");
            rawType = request.getParameter("type");
            include = request.getParameter("include");
            exclude = request.getParameter("exclude");
        }

        // defaulting and sanity-checking
        TypeArg type = (TypeArg)Utils.parseKeywordArg(TypeArg.class, rawType, "type", true, null);
        View view = (View)Utils.parseKeywordArg(View.class, rawView, "view", false, null);
        URI workspace = null;
        if (rawWorkspace != null) {
            if (Utils.isValidURI(rawWorkspace))
                workspace = new URIImpl(rawWorkspace);
            else
                throw new BadRequestException("Workspace is not a valid URI: "+rawWorkspace);
        }

        // sanity check, then set default view if needed
        if (workspace != null && view != null)
            throw new BadRequestException("Only one of the 'workspace' or 'view' args may be specified.");
        else if (workspace == null && view == null)
            view = (type == TypeArg.resource ? View.USER_RESOURCES : View.USER);

        Set<String> includes = parseXCludeList(include);
        Set<String> excludes = parseXCludeList(exclude);

        // sanity check format, also must be a quad format
        String mimeType = Formats.negotiateRDFContent(request, rawFormat, RDFFormat.TRIG.getDefaultMIMEType());
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mimeType);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mimeType+"\"");
        }
        if (!format.supportsContexts()) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "Format does not support quad (graph) encoding: "+format);
        }

        try {
            log.debug("Output serialization format = "+format);
            response.setContentType(mimeType);
            RDFHandler realOut = Rio.createWriter(format, response.getOutputStream());
            RDFHandler out = new mergeHandler(realOut);
            realOut.startRDF();
             
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
             
            // export of users
            if (type == TypeArg.user) {

                // access check
                if (!Access.isSuperuser(request))
                    throw new ForbiddenException("This request requires administrator privileges.");

                // pass 1 - users with RDF presnece
                Map<String,AuthUser> authUsers = AuthUser.findAllAsMap();
                for (User u : User.findAll(request)) {
                    URI uu = u.getURI();
                    String uus = uu.toString();
                    String un = u.getUsername();
                    AuthUser au = authUsers.remove(un);
                    if (excludes.contains(un) || excludes.contains(uus))
                        log.debug("SKIP USER because of exclude: "+u);
                    else if (includes.isEmpty() ||
                             includes.contains(un) || includes.contains(uus)) {
                        log.debug("EXPORT USER: "+uu);
                        // XXX maybe add NG_METADATA graph here? nothing there YET..
                        rc.exportStatements(uu, null, null, false, out,
                            REPO.NG_USERS, REPO.NG_INTERNAL);

                        if (au == null) {
                            log.warn("User is NOT IN RDBMS, so no password: username="+un);
                        } else {
                            // synthetic statement: identify user type
                            out.handleStatement(new ContextStatementImpl(
                                uu, EXPORT_AUTH_TYPE, EXPORT_AUTH_TYPE_BUILTIN, REPO.NG_INTERNAL));
                             
                            // synthetic statement: add the password from RDBMS
                            out.handleStatement(new ContextStatementImpl(
                                uu, EXPORT_AUTH_PASSWORD, new LiteralImpl(au.getPassword(), XMLSchema.STRING), REPO.NG_INTERNAL));
                        }
                    } else
                        log.debug("SKIP USER because of include: "+u);
                }

                // pass 2: RDBMS-only users
                // create blank node for each with essential properties
                for (AuthUser au : authUsers.values()) {
                    BNode subject = vf.createBNode();
                    out.handleStatement(new ContextStatementImpl(subject,
                        EXPORT_AUTH_TYPE,
                        EXPORT_AUTH_TYPE_BUILTIN,
                        REPO.NG_INTERNAL));
                    out.handleStatement(new ContextStatementImpl(subject,
                        REPO.HAS_PRINCIPAL_NAME,
                        new LiteralImpl(au.getUsername(), XMLSchema.STRING),
                        REPO.NG_INTERNAL));
                    out.handleStatement(new ContextStatementImpl(subject,
                        EXPORT_AUTH_PASSWORD,
                        new LiteralImpl(au.getPassword(), XMLSchema.STRING),
                        REPO.NG_INTERNAL));
                }

            // export of instances
            } else {
                //XXX not done yet
                throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Resource export not implemented yet.");
            }
            realOut.endRDF();

        } catch (NamingException e) {
            throw new InternalServerErrorException(e);
        } catch (SQLException e) {
            throw new InternalServerErrorException(e);
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * {@inheritDoc}
     *
     * Import a collection of Users *or* Instances from an export serialization
     * generated by this service.
     * POST is for IMPORT only.
     *
     * Args:
     *  - format = MIME type (overrides content-type)
     *  - duplicate=(abort|ignore|replace)
     *  - transform=yes|no - no default
     *  - graph=(abort|create) -- what to do when import would create new graph
     *  - type=(resource|user) - what to import
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        // GET implements the export function, even for a POST.
        if (isExport) {
            doGet(request, response);
            return;
        }
        if (!isImport)
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Servlet must be configured for import or export.");

        String rawFormat = null;
        String rawDuplicate = null;
        String rawTransform = null;
        String rawNewGraph = null;
        String rawType = null;
        String include = null;
        String exclude = null;
        String contentType = null;
        Reader content = null;

        // if we got here through POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                File tmp = (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir");
                if (tmp == null)
                    throw new InternalServerErrorException("Cannot find servlet context attr = \"javax.servlet.context.tempdir\"");
                upload.setFileItemFactory(new DiskFileItemFactory(100000, tmp));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("format"))
                        rawFormat = item.getString();
                    else if (ifn.equals("duplicate"))
                        rawDuplicate = item.getString();
                    else if (ifn.equals("transform"))
                        rawTransform = item.getString();
                    else if (ifn.equals("newgraph"))
                        rawNewGraph = item.getString();
                    else if (ifn.equals("type"))
                        rawType = item.getString();
                    else if (ifn.equals("include"))
                        include = item.getString();
                    else if (ifn.equals("exclude"))
                        exclude = item.getString();
                    else if (ifn.equals("content")) {
                        content = new InputStreamReader(item.getInputStream());
                        contentType = item.getContentType();
                        log.debug("Got content stream, MIME type = "+contentType);
                    } else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            rawFormat = request.getParameter("format");
            rawDuplicate = request.getParameter("duplicate");
            rawTransform = request.getParameter("transform");
            rawNewGraph = request.getParameter("newgraph");
            rawType = request.getParameter("type");
            include = request.getParameter("include");
            exclude = request.getParameter("exclude");
            String cs = request.getParameter("content");
            if (cs != null)
                content = new StringReader(cs);
        }

        // defaulting and sanity-checking
        TypeArg type = (TypeArg)Utils.parseKeywordArg(TypeArg.class, rawType, "type", true, null);
        DuplicateArg duplicate = (DuplicateArg)Utils.parseKeywordArg(DuplicateArg.class, rawDuplicate, "duplicate", false, DuplicateArg.abort);
        NewGraphArg newGraph = (NewGraphArg)Utils.parseKeywordArg(NewGraphArg.class, rawNewGraph, "newgraph", false, NewGraphArg.abort);
        boolean transform = Utils.parseBooleanParameter(rawTransform, "transform", true, false);
        Set<String> includes = parseXCludeList(include);
        Set<String> excludes = parseXCludeList(exclude);
        if (rawFormat == null)
            rawFormat = contentType;

        // sanity check format
        if (rawFormat == null) {
            throw new BadRequestException("Missing required argument: format (or content-type on input)");
        }
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(rawFormat);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+rawFormat+"\"");
        }
        if (!format.supportsContexts()) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "Format does not support quad (graph) encoding: "+format);
        }
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        log.debug("Input serialization format = "+format);
        try {

            // ------------- import of users
            if (type == TypeArg.user) {

                // access check
                if (!Access.isSuperuser(request))
                    throw new ForbiddenException("This request requires administrator privileges.");

                // read document into memory repo so we can query it without polluting main repo
                Repository mr = null;
                RepositoryConnection mrc = null;
                try {
                    mr = new SailRepository(new MemoryStore());
                    mr.initialize();
                    mrc = mr. getConnection();
                    mrc.add(content, "", format);

                    TupleQuery q = mrc.prepareTupleQuery(QueryLanguage.SPARQL, importUserGetAuthUsers);
                    log.debug("SPARQL query PASS1 against internal memory repo = "+importUserGetAuthUsers);
                    q.setDataset(User.getUserDataset());
                    q.setIncludeInferred(false);
                    // c is a single transaction for all AuthUser changes.
                    Connection c = AuthUser.startTransaction();
                    authUserPass1Handler ah = new authUserPass1Handler(c, rc, transform, duplicate, includes, excludes);
                    try {
                        // pass 1 - AuthUsers, basic description
                        q.evaluate(ah);

                        // pass 2 - gather all roles and user metadata
                        q = mrc.prepareTupleQuery(QueryLanguage.SPARQL, importUserGetStatements);
                        log.debug("SPARQL query PASS2 against internal memory repo = "+importUserGetStatements);
                        q.setDataset(User.getUserDataset());
                        q.setIncludeInferred(false);
                        q.evaluate(new authUserPass2Handler(ah));

                        // XXX NOTE:  sort of a race condition here, but
                        // it can't be helped.  If the Sesame commit
                        // goes south, the RDBMS data will be
                        // inconsitent but we can live with that.
                        AuthUser.commitTransaction(c);
                        rc.commit();
                        c = null;
                    } finally {
                        if (c != null) {
                            AuthUser.abortTransaction(c);
                            c = null;
                        }
                    }

                } catch (handlerBadRequest e) {
                    throw new BadRequestException(e.getMessage());
                } catch (NamingException e) {
                    log.error("Failed in IMPORT USER: ",e);
                    throw new InternalServerErrorException(e);
                } catch (SQLException e) {
                    log.error("Failed in IMPORT USER: ",e);
                    throw new InternalServerErrorException(e);
                } finally {
                    if (mrc != null && mrc.isOpen()) {
                        mrc.rollback();
                        mrc.close();
                    }
                    if (mr != null)
                        mr.shutDown();
                }

            } else {
              throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "import instances");
            }
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    // parse excludes or includes list - space separated URI or string values
    private Set<String> parseXCludeList(String s)
    {
        Set<String> result = new HashSet<String>();
        if (s != null) {
            for (String e : s.split("\\s+"))
                result.add(e);
        }
        return result;
   }

    // merge several "streams" of output statements into one output
    private static class mergeHandler extends RDFHandlerWrapper
    {
        private mergeHandler(RDFHandler defer)
        {
            super(defer);
        }
        public void startRDF()
        {
            // do nothing
        }
        public void endRDF()
        {
            // do nothing
        }
    }

    // Pass 1 of user import:
    //  - filter users to import
    //  - create RDBMS entry for selected users
    //  - prepare map of URI to username for second pass
    private static class authUserPass1Handler extends TupleQueryResultHandlerBase
    {
        // map of user URI to principal username
        private RepositoryConnection rc;
        private Connection rdbms;
        private boolean transform;
        private DuplicateArg duplicate;
        private Set<String> includes;
        private Set<String> excludes;
        private int count = 0;
        // usernames which have been allowed to be imported
        private Set<String> allow = new HashSet<String>();

        public authUserPass1Handler(Connection c, RepositoryConnection r,
                boolean t, DuplicateArg d, Set<String> inc, Set<String> exc)
        {
            super();
            rdbms = c;
            rc = r;
            transform = t;
            duplicate = d;
            includes = inc;
            excludes = exc;
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value vsubject = bs.getValue("authUser");
            Value vusername = bs.getValue("username");
            Value vpassword = bs.getValue("password");
            boolean su = bs.getValue("su") != null;

            log.debug("authUserPass1Handler: Got subj="+vsubject+", username="+
              vusername+", password="+(vpassword==null?null:"[password]")+", su="+su);
            // sanity check
            if (vsubject == null || vusername == null || vpassword == null)
                throw new handlerBadRequest("Bad export file content: missing one of the required data: "+
                        "subject="+vsubject+", username="+vusername+", password="+(vpassword==null?null:"[password]"));
            try {

                // filter: check include, exclude, duplicates
                String username = vusername.stringValue();
                URI subject = vsubject instanceof URI ? (URI)vsubject : null;
                String ssubject = vsubject.stringValue();
                if (excludes.contains(username) || excludes.contains(ssubject)) {
                    log.debug("PASS1: SKIP USER import because of exclude: username="+username+", or subject="+ssubject);
                    return;
                } else if (!(includes.isEmpty() ||
                           includes.contains(username) ||
                           includes.contains(ssubject))) {
                    log.debug("PASS1: SKIP USER import because of include: username="+username+", or subject="+ssubject);
                    return;
                }
                 
                // is this a duplicate username (principal)?
                AuthUser au = AuthUser.find(username);
                if (au != null) {
                    log.debug("PASS1: FOUND DUPLICATE PRINCIPAL, user principal="+username);
                    if (duplicate == DuplicateArg.ignore)
                        return;
                    else if (duplicate == DuplicateArg.abort)
                        throw new handlerBadRequest("Import contains a duplicate user for username="+username);
                }
                 
                // is this a duplicate of the user URI?
                if (subject != null && rc.hasStatement(subject, REPO.HAS_PRINCIPAL_NAME, null, false, REPO.NG_INTERNAL)) {
                    log.debug("PASS1: FOUND DUPLICATE SUBJECT, user subject="+subject);
                    if (duplicate == DuplicateArg.ignore)
                        return;
                    else if (duplicate == DuplicateArg.abort)
                        throw new handlerBadRequest("Import contains a duplicate user for URI="+subject);
                }
                 
                // finally, create or replace user:
                if (au != null) {
                    au.setPassword(vpassword.stringValue());
                    au.setIsSuperuser(su);
                    au.update(rdbms);
                } else {
                    AuthUser.create(rdbms, username, vpassword.stringValue(), su);
                }
                allow.add(username);
                ++count;
                log.debug("Added new user in PASS1: username="+username+", URI="+subject);
            } catch (NamingException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            } catch (SQLException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            } catch (RepositoryException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            }
        }
    }

    // pass 2 - install all the user instance and metadata
    private static class authUserPass2Handler extends TupleQueryResultHandlerBase
    {
        // map of user URI to principal username
        private authUserPass1Handler pass1;

        // imported subject to transformed subject - also used as flag that
        // we've seen this subject before and it is not to be filtered out.
        private Map<URI,URI> subject2uri = new HashMap<URI,URI>();

        public authUserPass2Handler(authUserPass1Handler p1)
        {
            super();
            pass1 = p1;
        }

        public void endQueryResult()
        {
            log.info("SUMMARY: Added and/or replaced: "+pass1.count+" RDBMS entries, and "+subject2uri.size()+" User Descriptions.");
        }

        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value vgraph = bs.getValue("g");
            Value vsubject = bs.getValue("s");
            Value vpredicate = bs.getValue("p");
            Value vobject = bs.getValue("v");
            Value vusername = bs.getValue("username");
            ValueFactory vf = pass1.rc.getValueFactory();

            // sanity check
            if (vsubject == null || vgraph == null || vpredicate == null || vobject == null)
                throw new handlerBadRequest("Bad export file content: missing one of the required data: "+
                            "subj="+vsubject+", pred="+vpredicate+", obj="+vobject+", graph="+vgraph);
            log.debug("authUserPass2Handler: Got subj="+vsubject+", pred="+vpredicate+", obj="+vobject.stringValue()+", graph="+vgraph);
            try {

                // filter: check include, exclude, if not already passed
                URI subject = vsubject instanceof URI ? (URI)vsubject : null;
                String username = vusername == null ? null : vusername.stringValue();

                // skip if already denied in pass1
                if (username != null && !pass1.allow.contains(username))
                    return;

                if (!subject2uri.containsKey(subject)) {
                    String ssubject = vsubject.stringValue();
                    if (pass1.excludes.contains(username) || pass1.excludes.contains(ssubject)) {
                        log.debug("PASS2: SKIP USER import because of exclude: username="+username+", or subject="+ssubject);
                        return;
                    } else if (!(pass1.includes.isEmpty() ||
                               pass1.includes.contains(username) ||
                               pass1.includes.contains(ssubject))) {
                        log.debug("PASS2: SKIP USER import because of include: username="+username+", or subject="+ssubject);
                        return;
                    }

                    // Is there a duplicate of the user URI?
                    // If replacing, and NOT transforming,remove all old statements.
                    if (subject != null && pass1.rc.hasStatement(subject, REPO.HAS_PRINCIPAL_NAME, null, false, REPO.NG_INTERNAL)) {
                        log.debug("PASS2: FOUND DUPLICATE SUBJECT, user subject="+subject);
                        if (pass1.duplicate == DuplicateArg.ignore)
                            return;
                        else if (pass1.duplicate == DuplicateArg.abort)
                            throw new handlerBadRequest("Import contains a duplicate user for URI="+subject);
                        else if (pass1.duplicate == DuplicateArg.replace) {
                            if (!pass1.transform) {
                                pass1.rc.remove(subject, null, null, REPO.NG_INTERNAL, REPO.NG_USERS);
                                log.debug("Removing all statements about user URI="+subject+" in NG_Users");
                            }
                        } else
                            throw new handlerBadRequest("Unknown state of 'duplicate' arg, duplicate="+pass1.duplicate);
                    }

                    // Was there already a Person object for this
                    // username?  If so, remove it:
                    RepositoryResult<Statement> rr =
                        pass1.rc.getStatements(null, REPO.HAS_PRINCIPAL_NAME, vusername, false, REPO.NG_INTERNAL);
                    try {
                        while (rr.hasNext()) {
                            Resource uu = rr.next().getSubject();
                            log.debug("PASS2: Clearing out existing Person with principal="+username+", subject="+uu);
                            pass1.rc.remove(uu, null, null, REPO.NG_INTERNAL, REPO.NG_USERS);
                        }
                    } finally {
                        rr.close();
                    }
                     
                    // record that we've been here so next results for this
                    // subject skip the filter test, and use the same transform
                    if (pass1.transform)
                        subject2uri.put(subject, vf.createURI(DataRepository.getInstance().getDefaultNamespace(),
                                              RIDGenerator.getInstance().newID().toString()));
                    else
                        subject2uri.put(subject, subject);
                }
                URI newsubj = subject2uri.get(subject);
                pass1.rc.add(newsubj, (URI)vpredicate, vobject, (URI)vgraph);
                log.debug("PASS2: Added statement ("+newsubj+", "+vpredicate+", "+vobject.stringValue()+", "+vgraph+")");
            } catch (RepositoryException e) {
                log.error("Failed in PASS1: ",e);
                throw new TupleQueryResultHandlerException(e);
            }
        }
    }

    // Marker exception, tells caller to throw a BadRequest error from a tuple result handler
    private static class handlerBadRequest extends TupleQueryResultHandlerException
    {
        public handlerBadRequest(String msg)
        {
            super(msg);
        }
    }
}
