package org.eaglei.repository.servlet;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.Query;
import org.openrdf.query.BooleanQuery;
import org.openrdf.query.GraphQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.query.resultio.BooleanQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.QueryResultIO;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.ValueFactory;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;

import org.eaglei.repository.Formats;
import org.eaglei.repository.Access;
import org.eaglei.repository.View;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;

/**
 * SPARQL query protocol service
 * see http://www.w3.org/TR/rdf-sparql-protocol/
 *
 * TODO: finish, more doc
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class SparqlProtocol extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(SparqlProtocol.class);

    // query arg names defined in SPARQL/Protocol (others are nonstandard)
    private static final String P_QUERY = "query";
    private static final String P_NAMED_GRAPH = "named-graph-uri";
    private static final String P_DEFAULT_GRAPH = "default-graph-uri";

    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        this.doPost(request,response);
    }

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);

        String rawquery = null;
        String namedGraph[] = null;
        String defaultGraph[] = null;
        String format = null;
        String rawview = null;
        String rawinferred = null;
        View view = null;
        String workspace = null;
        String time = null;
        long startMs = System.currentTimeMillis();

        // if POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            List<String> ngl = new ArrayList<String>();
            List<String> dgl = new ArrayList<String>();

            try {
                ServletFileUpload upload = new ServletFileUpload();
                upload.setFileItemFactory(new DiskFileItemFactory(100000,
                  (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir")));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals(P_QUERY))
                        rawquery = item.getString();
                    else if (ifn.equals(P_DEFAULT_GRAPH)) {
                        String s = item.getString();
                        if (s.length() > 0)
                            dgl.add(s);
                    } else if (ifn.equals("format"))
                        format = item.getString();
                    else if (ifn.equals(P_NAMED_GRAPH)) {
                        String s = item.getString();
                        if (s.length() > 0)
                            ngl.add(s);
                    } else if (ifn.equals("view"))
                        rawview = item.getString();
                    else if (ifn.equals("workspace"))
                        workspace = item.getString();
                    else if (ifn.equals("time"))
                        time = item.getString();
                    else if (ifn.equals("inferred"))
                        rawinferred = item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }
            if (ngl.size() > 0)
                namedGraph = ngl.toArray(new String[ngl.size()]);
            if (dgl.size() > 0)
                defaultGraph = dgl.toArray(new String[dgl.size()]);

        // gather args from input params instead
        } else {
            rawquery = request.getParameter(P_QUERY);
            // sanity check: elide empty list or list of one empty string which CAN be generated e.g. by Web form
            namedGraph = request.getParameterValues(P_NAMED_GRAPH);
            if (namedGraph != null && (namedGraph.length == 0 || namedGraph[0].length() == 0))
                namedGraph = null;
            // sanity check: elide empty list or list of one empty string which CAN be generated e.g. by Web form
            defaultGraph = request.getParameterValues(P_DEFAULT_GRAPH);
            if (defaultGraph != null && (defaultGraph.length == 0 || defaultGraph[0].length() == 0))
                defaultGraph = null;
            format = request.getParameter("format");
            rawview = request.getParameter("view");
            workspace = request.getParameter("workspace");
            time = request.getParameter("time");
            rawinferred = request.getParameter("inferred");
        }

        // sanity check
        if (rawquery == null || rawquery.equals("")) {
            throw new BadRequestException("Missing required arg: "+P_QUERY);
        }

        // Beware: Some HTML forms can submit 0-length string for "view" etc.
        if (rawview != null && rawview.length() > 0)
            view = View.parseView(rawview);

        if (workspace != null && workspace.length() == 0)
            workspace = null;

        // sanity check: workspace, view, and default/named graphs args
        // are all mutually exclusive:
        int dsCount = 0;
        if (namedGraph != null || defaultGraph != null)
            ++dsCount;
        if (workspace != null)
            ++dsCount;
        if (view != null)
            ++dsCount;
        if (dsCount > 1) {
            throw new BadRequestException("Only one of the arguments for 'view', 'workspace' or explicit named/default graph URIs maybe specified.");

        // XXX dike this if we decide to implement a default view..
        } else if (dsCount == 0) {
            throw new BadRequestException("Missing required argument: Exactly one of the arguments 'view', 'workspace' or explicit named/default graph URIs must be specified.");
        }

        // include inferred statements? true by default.  NON-STANDARD arg.
        boolean inferred = Utils.parseBooleanParameter(rawinferred, "inferred", false, true);

        // parse the SPARQL query and handle each different form (SELECT, ASK, etc)
        try {
            // set up dataset - either explicit list, or view
            // XXX note that sesame only seems to notice default graphs..
            ValueFactory vf = rc.getValueFactory();
            DatasetImpl ds = new DatasetImpl();
            if (defaultGraph != null || namedGraph != null) {
                if (defaultGraph != null) {
                    for (String n : defaultGraph) {
                        URI gu = vf.createURI(n);
                        if (Access.hasPermission(request, gu, Access.READ)) {
                            log.debug("Adding Default Graph to the dataset: "+n);
                            ds.addDefaultGraph(gu);
                        } else {
                            throw new ForbiddenException("Read access denied to named graph: "+gu.toString());
                        }
                    }
                }
                if (namedGraph != null) {
                    for (String n : namedGraph) {
                        URI gu = vf.createURI(n);
                        if (Access.hasPermission(request, gu, Access.READ)) {
                            log.debug("Adding Named Graph to the dataset: "+n);
                            ds.addNamedGraph(gu);
                        } else {
                            throw new ForbiddenException("Read access denied to named graph: "+gu.toString());
                        }
                    }
                }
            } else if (view != null) {
                View.addGraphs(request, ds, view);
            } else if (workspace != null) {
                if (!Utils.isValidURI(workspace))
                    throw new BadRequestException("Workspace URI must be a legal absolute URI: "+workspace);
                View.addWorkspaceGraphs(request, ds, new URIImpl(workspace));
            }

            // sanity check: do not let the query proceed with empty dataset
            // because this would inadvertently give access to ALL named graphs.
            if (ds.getDefaultGraphs().isEmpty() && ds.getNamedGraphs().isEmpty()) {
                throw new InternalServerErrorException("Empty dataset for query, this should not happen.");
            }

            Query q = rc.prepareQuery(QueryLanguage.SPARQL, rawquery);
            q.setIncludeInferred(inferred);

            // For "null" view, do NOT set the explicit dataset at all.
            // This allows SPARQL's "GRAPH" operator to work, and otherwise
            // the effect is the same as an explicit dataset with a
            // graph of null -- it gives access to *all* contexts.
            if (View.NULL.equals(view))
                log.debug("Null view, leave query's Dataset unset!");

            else {
                log.debug("Dataset for SPARQL query = "+Utils.prettyPrint(ds));
                q.setDataset(ds);
            }

            // Set the query timeout if a value was given.
            // XXX FIXME maybe there ought to be a default, at least for non-Superuser
            if (time != null && time.length() > 0) {
                try {
                    int tl = Integer.parseInt(time);
                    q.setMaxQueryTime(tl);
                    log.debug("Setting query time limit to "+String.valueOf(tl)+" seconds.");
                } catch (NumberFormatException e) {
                    throw new BadRequestException("Value for 'time' must be a decimal integer; this is unacceptable: "+time);
                }
            }

            // XXX FIXME should set max query time here

            // SELECT query
            if (q instanceof TupleQuery) {
                String mimeType = Formats.negotiateTupleContent(request, format, Formats.DEFAULT_TUPLE_MIMETYPE);
                TupleQueryResultFormat tqf = QueryResultIO.getWriterFormatForMIMEType(mimeType);
                if (tqf == null) {
                    log.error("Failed to get tuple query format, for mime="+mimeType);
                    throw new ServletException("Failed to get tuple query format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(mimeType);
                    ((TupleQuery)q).evaluate(QueryResultIO.createWriter(tqf, response.getOutputStream()));
                }

            // ASK query
            } else if (q instanceof BooleanQuery) {
                String mimeType = Formats.negotiateBooleanContent(request, format, Formats.DEFAULT_BOOLEAN_MIMETYPE);
                BooleanQueryResultFormat bf = BooleanQueryResultFormat.forMIMEType(mimeType);
                if (bf == null) {
                    log.error("Failed to get boolean serialization format, for mime="+mimeType);
                    throw new ServletException("Failed to get boolean serialization format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(mimeType);
                    QueryResultIO.createWriter(bf, response.getOutputStream()).write(((BooleanQuery)q).evaluate());
                }

            // CONSTRUCT, DESCRIBE queries
            } else if (q instanceof GraphQuery) {
                String mimeType = Formats.negotiateRDFContent(request, format, Formats.DEFAULT_RDF_MIMETYPE);
                RDFFormat rf = Formats.RDFOutputFormatForMIMEType(mimeType);
                if (rf == null) {
                    log.error("Failed to get RDF serialization format, for mime="+mimeType);
                    throw new ServletException("Failed to get RDF serialization format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(mimeType);
                    ((GraphQuery)q).evaluate(Rio.createWriter(rf, response.getOutputStream()));
                }

            } else {
                log.error("Unrecognized query type! This should never happen.");
                throw new BadRequestException("Unrecognized query type! This should never happen.");
            }
        } catch (MalformedQueryException e) {
            log.info("Rejecting malformed query.");
            throw new BadRequestException("Malformed query: "+e.toString());
        // this "should never happen", means the SAIL or repo does not support generic prepareQuery()
        } catch (UnsupportedOperationException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        } finally {
            // log time message with a truncated fragment of query so
            // there is some hope of matching it up against the application.
            String qtrunc = rawquery;
            Pattern qp = Pattern.compile("(select|ask|construct).*", Pattern.CASE_INSENSITIVE);
            Matcher qm = qp.matcher(rawquery);
            if (qm.find())
                qtrunc = rawquery.substring(qm.start());
            log.info("SPARQL protocol request completed in "+
                     String.format("%,d mSec. query=%.120s (...)", System.currentTimeMillis()-startMs, qtrunc));
        }
    }
}
