package org.eaglei.repository.model;

import java.util.Iterator;
import java.util.HashSet;
import java.util.Set;
import java.security.Principal;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Resource;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.Binding;
import org.openrdf.query.BindingSet;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandler;
import org.openrdf.query.BooleanQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.WithRepositoryConnection;

/**
 * Access control for the repository.
 *
 * The Access class is two things:
 *  1. Enumerated type describing the type of operation the user is allowed to do.
 *     These correspond to REPO constants (and maybe should move there?)
 *  2. A collection of static utility methods to:
 *     a. Answer access-control questions
 *     b. Manage the records of access grants, including import/export
 *     (These arguably don't belong in this enum class but that's how the
 *      code evolved and moving the static methods to a new class isn't
 *      really any more clear..)
 *
 * Access permission is computed as follows:
 *  1. Does current user have the Superuser role?  If so, always "yes".
 *  2. Is there a direct grant, e.g. <resource> :has___Access <user> ?
 *  3. Indirect role grant?   e.g. <resource> :has___Access <role>, and
 *     user asserts that role (i.e. <user> :hasRole <role> )
 *  NOTES:
 *   - Roles are NOT hierarchical, each role is independent.
 *   - ALL users have :Role_Authenticated and :Role_Anonymous asserted
 *     invisibly (materialized but managed automatically)
 *   - A session without a logged-in user is identified as :Role_Anonymous
 *
 * Started April, 2010
 *
 * @author Larry Stone
 * @version $Id: $
 */
public enum Access
{
    /** Types of access to be granted */
    READ   (REPO.HAS_READ_ACCESS),
    ADD    (REPO.HAS_ADD_ACCESS),
    REMOVE (REPO.HAS_REMOVE_ACCESS),
    ADMIN  (REPO.HAS_ADMIN_ACCESS);

    // the related property URI for this access type
    private URI uri = null;

    // all the URIs of values for efficient membership checks.
    private static final Set<URI> valueURIs= new HashSet<URI>();
    static {
        for (Access a : values()) {
            valueURIs.add(a.uri);
        }
    }

    // query string used in hasPermission()
    // Expect to have bindings for: ?user, ?access, and ?resource
    private static final String hasPermissionQuery = makeAccessQuery("resource", "ASK", null);

    // dataset of internal graphs plus NG_Users to get user's rdf:type
    public static final DatasetImpl ACCESS_DATASET = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        ACCESS_DATASET.addDefaultGraph(User.USER_GRAPH);
    }

    private static final Logger log = LogManager.getLogger(Access.class);

    // constructor for enum with uri
    private Access(URI uri)
    {
        this.uri = uri;
    }

    /**
     * Predicate testing whether a URI is a valid access grant property.
     *
     * @param uri the uri to test
     * @return true if uri is the URI value of an access grant keyword.
     */
    public static boolean isAccessPredicate(URI uri)
    {
        return valueURIs.contains(uri);
    }

    /**
     * Get the URI referenced by this access type.
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }

    @Override
    public String toString()
    {
        return name().toUpperCase();
    }

    /**
     * <p>hasPermission - predicate, general permission test.</p>
     * Does current user have the indicated
     * permission on this resource?  See the general formula and rules
     * for computing access in comments at the head of this class.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param subject the object being tested for access
     * @param pred the type of access
     * @return a boolean, true if access was granted.
     */
    public static boolean hasPermission(HttpServletRequest request, Resource subject, Access pred)
    {
        if (Authentication.isSuperuser(request)) {
            if (log.isDebugEnabled())
                log.debug("Superuser elides check: hasPermission("+subject+", "+pred+") => true");
            return true;
        }

        try {
            URI pu = Authentication.getPrincipalURI(request);
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // XXX FIXME could optimize later as "prepared" BooleanQuery,
            // XXX maybe store in app context; is is bound to the RepositoryConnection
            // XXX  Parameterized prepared query - should be able to
            // XXX re-use prepared query object so long as Repository doesn't change.

            // Build SPARQL ASK query to get permission;
            // This is like an SQL prepared statement, setBinding plugs
            // values into the variables.  Variables are:
            //  ?user - agent seeking access (user or role)
            //  ?access - uri of access type/predicate
            //  ?resource - uri of object/subject
            BooleanQuery q = rc.prepareBooleanQuery(QueryLanguage.SPARQL, hasPermissionQuery);
            q.setIncludeInferred(true);
            q.setDataset(ACCESS_DATASET);
            q.clearBindings(); // needed if re-using query
            q.setBinding("user", pu);
            q.setBinding("access", pred.uri);
            q.setBinding("resource", subject);
            boolean result = SPARQL.evaluateBooleanQuery(hasPermissionQuery, q);
            if (log.isDebugEnabled()) {
                log.debug("hasPermission("+subject+", "+pred+", "+pu+") => "+result);
            }
            return result;
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in access check: ",e);
        }
    }

    /**
     * Special case access predicate on User objects.
     * Does CURRENT AUTHENTICATED user have permission TO MODIFY the
     * User object associated with this username?
     * True if it matches the current logged-in user, or we are superuser.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal (i.e. RDBMS username, value of :hasPrincipal), a {@link java.lang.String} object.
     * @return a boolean, true if permission is gratned.
     */
    public static boolean hasPermissionOnUser(HttpServletRequest request, String username)
    {
        Principal p = request.getUserPrincipal();
        return Authentication.isSuperuser(request) ||
               (username != null && p != null && username.equals(p.getName()));
    }

    /**
     * Filters results of query by what the current user has indicated permission
     * on.. Resource (URI) expected to be in  variable named "?{name}"  so
     * this SPARQL pattern group fragment (in "{ }") can be combined with the
     * rest of the query.
     *
     * See hasPermission() for algorithm to figure permission.  The only
     * difference is that this does NOT work for superuser.  (It could
     * be added if there is a need.)
     *
     * Results are returned by calling tuple query handler.
     *
     *   ***** WARNING *****
     * DO NOT call this if you are superuser! It will not work.
     * Test for superuser before calling this filter and use alternate query.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param principal URI of user or role being checked for permission
     * @param name bare name of the variable in query containing URI to test for
     *        access. NOTE: 'name' MUST be a variable in 'results' list
     * @param results query results clause, i.e. SELECT <results> WHERE ...
     * @param patternGroup query fragment
     * @param pred type of access being tested, a {@link org.eaglei.repository.Access} object.
     * @param dataset the dataset on whcih to operate, MUST not be null
     * @param handler a {@link org.openrdf.query.TupleQueryResultHandler} object.
     */
    public static void filterByPermission(HttpServletRequest request,
                           URI aprincipal, String name, String results,
                           String patternGroup, Access pred, Dataset dataset,
                           BindingSet bindings, TupleQueryResultHandler handler)
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            URI principal = aprincipal == null ? Authentication.getPrincipalURI(request) : aprincipal;

            // Parameterized prepared query - should be able to
            // re-use prepared query object so long as Repository doesn't change.
            // replacement because there's no equiv. of a Prepared Statement..
            // This is like an SQL prepared statement, setBinding plugs
            // values into the variables and seems to work..

            // XXX maybe optimize later by storing "prepared" BooleanQuery,
            //  maybe in app context. it depends on the sesame repo obj.
            String qs = makeAccessQuery(name, "SELECT "+results+" WHERE", patternGroup);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setIncludeInferred(true);
            q.setDataset(dataset);
            q.clearBindings(); // needed if re-using query
            q.setBinding("user", principal);
            q.setBinding("access", pred.uri);
            if (bindings != null && bindings.size() > 0) {
                for (Iterator<Binding> bi = bindings.iterator(); bi.hasNext();) {
                    Binding b = bi.next();
                    q.setBinding(b.getName(), b.getValue());
                }
            }
            SPARQL.evaluateTupleQuery(qs, q, handler);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new InternalServerErrorException("Failed in access check: ",e);
        }
    }

    // cons up a sparql triple pattern to test access.
    // XXX note the parentheses don't balance
    private static String makeAccessQuery(String resourceName, String prologue, String patternGroup)
    {
        StringBuilder result = new StringBuilder();
        result.append(prologue).append(" { ");
        if (patternGroup != null)
            result.append(patternGroup);

        // user is in a role that has access
        result.append("{ { ?user <").append(REPO.HAS_ROLE)
              .append("> ?r . ?").append(resourceName).append(" ?access ?r }\n");

        // direct grant to user (or Role if login == ANONYMOUS)
        result.append(" UNION { ?").append(resourceName).append(" ?access ?user } } }");
        return result.toString();
    }
}
