package org.eaglei.repository.model;

import java.io.IOException;
import java.util.Set;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.openrdf.rio.RDFFormat;
import org.openrdf.repository.RepositoryConnection;

import org.eaglei.repository.util.DuplicateArg;

/**
 * All object models that can be subjects of the /export and /import
 * services must implement this interface.  Since hte process of
 * serializing and deserializing complete, self-contained instances of
 * the data is mostly unique logic, it belongs in a separate class.
 *
 * @author Larry Stone
 * Started March, 2011
 */
public interface Transporter
{
    /**
     * Check that current authenticated user is authorized for this
     * operation; some export requests require Superuser privilege.
     * It is expected to throw a runtime HTTP status error, e.g.
     * ForbiddenException, when access is denied.
     * @param request the HTTP request object from servlet container
     */
    void authorizeExport(HttpServletRequest request)
        throws ServletException;

    /**
     * Check that current authenticated user is authorized for this
     * operation; some import requests require Superuser privilege.
     * It is expected to throw a runtime HTTP status error, e.g.
     * ForbiddenException, when access is denied.
     * @param request the HTTP request object from servlet container
     */
    void authorizeImport(HttpServletRequest request)
        throws ServletException;

    /**
     * Serializes the selected objects to the HTTP response.
     *
     * @param request the HTTP request object from servlet container
     * @param response the HTTP response object from servlet container
     * @param format the chosen output format
     * @param includes restricted set of instances to include in the output; default is all
     * @param excludes set of instances to exclude from the output, overrides includes
     */
    void doExport(HttpServletRequest request, HttpServletResponse response,
                       RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException;

    /**
     *
     * @param request the HTTP request object from servlet container
     * @param response the HTTP response object from servlet container
     * @param content separate, temporary Sesame repository containing input RDF graph
     * @param includes restricted set of instances allow to be ingested; default is all
     * @param excludes set of instances to exclude from the ingest; overrides includes
     * @param duplicate directive on how to handle duplicates of instances already in repository
     * @param transform when true, transform the instance URI into new "local" URI
     * @param ignoreACL when true, do not ingest the access control list (ACL), i.e. access grants
     */
    void doImport(HttpServletRequest request, HttpServletResponse response,
                       RepositoryConnection content,
                       Set<String> includes, Set<String> excludes,
                       DuplicateArg duplicate,
                       boolean transform, boolean ignoreACL)
        throws ServletException, IOException;
}
