package org.eaglei.repository.model.workflow;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.HashSet;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.NumericLiteralImpl;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.model.ValueFactory;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.model.WritableObjectModel;
import org.eaglei.repository.Configuration;
import org.eaglei.repository.model.User;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.REPO;

/**
 * WorkflowTransition object model, reflects the :WorkflowTransition object in
 * RDF database.
 *
 * @author Larry Stone
 * Started October, 2010
 * @version $Id: $
 */
public final class WorkflowTransition extends WritableObjectModel
{
    private static Logger log = LogManager.getLogger(WorkflowTransition.class);

    private URI uri;                  // subject URI: required
    private String label = null;      // value of rdfs:label (recommended)
    private String comment = null;    // value of rdfs:comment
    private URI workspace = null;     // related workspace if any
    private URI initialState = null;        // initial workflow state (req)
    private URI finalState = null;          // final workflow state   (req)
    private WorkflowAction action = null;   // instance of workflow action to call
    private Value actionParameter = null;   // transition-speicifc parameter
    private Literal order = null;           // optional ordering metric
    private String workspaceLabel = null;   // r/o cache of label
    private String initialLabel = null;     // r/o cache of label
    private String finalLabel = null;       // r/o cache of label

    // SPARQL Query to get all transitions and labels..
    // bind ?uri to get a single transition object.
    private static final String transitionQueryPrefix =
          "{?uri a <"+REPO.WORKFLOW_TRANSITION+"> ; \n"+
          " <"+RDFS.LABEL+"> ?label ;\n"+
          " <"+REPO.INITIAL+"> ?initial; \n"+
          " <"+REPO.FINAL+"> ?final. \n"+
          " OPTIONAL { ?uri <"+REPO.ACTION+"> ?action . \n"+
          "   OPTIONAL { ?uri <"+REPO.ACTION_PARAMETER+"> ?actionParameter }} \n"+
          " OPTIONAL { ?uri <"+RDFS.COMMENT+"> ?comment }\n"+
          " OPTIONAL { ?uri <"+REPO.WORKSPACE+"> ?workspace \n"+
          "   OPTIONAL { ?workspace <"+RDFS.LABEL+"> ?workspaceLabel }}\n"+
          " OPTIONAL { ?initial <"+RDFS.LABEL+"> ?initialLabel }\n"+
          " OPTIONAL { ?final <"+RDFS.LABEL+"> ?finalLabel }\n"+
          " OPTIONAL { ?uri <"+REPO.ORDER+"> ?order } \n";
    private static final String transitionQuerySuffix =
          "ORDER BY ?order ?label";

    /**
     * SPARQL Query to get all transitions and labels..
     * bind ?uri to get a single transition object.
     * Used by Transporter class to grovel objects out of import.
     */
    public static final String SIMPLE_TRANSITION_QUERY =
          "SELECT DISTINCT * WHERE "+transitionQueryPrefix + /* { */ "}" + transitionQuerySuffix;

    private WorkflowTransition(URI uri, String label, String comment, URI initialState, URI finalState)
    {
        super();
        this.uri = uri;
        this.label = label;
        this.comment = comment;
        this.initialState = initialState;
        this.finalState = finalState;
    }

    /**
     * Get the WorkflowTransition for a known URI,
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri URI of the transition
     * @return a {@link org.eaglei.repository.WorkflowTransition} object
     * @throws javax.servlet.ServletException if any.
     */
    public static WorkflowTransition find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        List<WorkflowTransition> result =
          (List<WorkflowTransition>)findAccessibleByAttributes(request, uri, null, null, null);
        if (result.isEmpty())
            throw new NotFoundException("There is no Transition of URI="+uri);
        return result.get(0);
    }

    /**
     * Get all Transitions in an aesthetic order
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all transitions in a {@link java.lang.Collection} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<WorkflowTransition> findAll(HttpServletRequest request)
        throws ServletException
    {
        return findAccessibleByAttributes(request, null, null, null, null);
    }

    /**
     * Get all Transitions matching criteria dictated by args; null is
     * a wildcard, so all nulls means get all transactions.  Note that
     * this does NOT inherently do access control since it only shows
     * internal metadata.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param single if not null, then only find transition whose subject matches that URI
     * @param initialStateMatch filter only transitions with given initial state
     * @param workspaceMatch filter only transitions on either ALL workspaces or this one
     * @param accessible filter transitions accessible to this user or Role URI
     * @return all matching transitions in a {@link java.lang.Collection} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<WorkflowTransition> findAccessibleByAttributes(HttpServletRequest request,
                                              URI single,
                                              URI initialStateMatch,
                                              URI workspaceMatch,
                                              URI accessible)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            // make up the pattern group
            StringBuilder qb = new StringBuilder(transitionQueryPrefix);
            if (workspaceMatch != null) {
                qb.append(" FILTER (!BOUND(?workspace) || ?workspace = <")
                  .append(workspaceMatch).append(">)\n");
            }
            qb.append(/* { */ "}");

            List<WorkflowTransition> result = new ArrayList<WorkflowTransition>();
            WFTHandler hdl = new WFTHandler(result);

            // is the accessor URI a superuser, either raw role or enabled user?
            boolean superuser = false;
            if (accessible != null) {
                if (REPO.ROLE_SUPERUSER.equals(accessible)) {
                    superuser = true;
                } else {
                    User au = User.find(request, accessible);
                    superuser =  au != null && au.isSuperuser();
                }
                if (superuser)
                    log.debug("findAccessibleByAttr: Eliding filter because accessor is superuser");
            }
            if (accessible == null || superuser) {
                // turn pattern group into the full query
                qb.insert(0, "SELECT DISTINCT * WHERE ");
                qb.append(transitionQuerySuffix);
                String qs = qb.toString();
                TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
                q.clearBindings();
                if (single != null)
                    q.setBinding("uri", single);
                if (initialStateMatch != null)
                    q.setBinding("initial", initialStateMatch);
                q.setDataset(SPARQL.InternalGraphs);
                q.setIncludeInferred(false);
                SPARQL.evaluateTupleQuery(qs, q, hdl);
            } else {
                MapBindingSet bs = new MapBindingSet(2);
                if (single != null)
                    bs.addBinding("uri", single);
                if (initialStateMatch != null)
                    bs.addBinding("initial", initialStateMatch);
                Access.filterByPermission(request, accessible, "uri", " * ",
                    qb.toString(), Access.READ, SPARQL.InternalGraphs, bs, hdl);
            }
            log.debug("findAccessibleByAttributes: Got result count="+String.valueOf(result.size()));
            return result;
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /**
     * <p>getURI - getter</p>
     *
     * @return URI subject of the WorkflowTransition, a {@link org.openrdf.model.URI} object.
     */
    @Override
    public URI getURI()
    {
        return uri;
    }

    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return label of the WorkflowTransition, a {@link java.lang.String} object.
     */
    @Override
    public String getLabel()
    {
        return label == null ? uri.stringValue() : label;
    }

    /**
     * <p>Getter for the field <code>comment</code>.</p>
     *
     * @return comment of the WorkflowTransition, a {@link java.lang.String} object, MIGHT be null.
     */
    public String getComment()
    {
        return comment;
    }

    /** Getter for field workspace
     *
     * @return the workspace (can be null)
     */
    public URI getWorkspace()
    {
        return workspace;
    }

    /** Getter for field workspaceLabel
     *
     * @return workspaceLabel - can be null
     */
    public String getWorkspaceLabel()
    {
        return workspaceLabel;
    }

    /** Getter for initial transition
     *
     * @return initial transition URI, should never be null
     */
    public URI getInitial()
    {
        return initialState;
    }

    /** Getter for label of initial transition
     *
     * @return label of initial transition, should not be null but might be
     */
    public String getInitialLabel()
    {
        return initialLabel;
    }

    /** Getter for final transition uri
     *
     * @return final transition uri, should never be null
     */
    public URI getFinal()
    {
        return finalState;
    }

    /** Getter for label of final transition
     *
     * @return label should not be null
     */
    public String getFinalLabel()
    {
        return finalLabel;
    }

    /** getter for order
     *
     * @return order as string, or null if none set
     */
    public String getOrder()
    {
        return order == null ? null : Utils.valueAsString(order);
    }

    /** Getter for action
     *
     * @return action - instance of specificed class, or null if not set
     */
    public WorkflowAction getAction()
    {
        return action;
    }

    /** predicate true if action indicates an error
     *
     * @return true if there was an error setting action (e.g. import of illegal class)
     */
    public boolean isErrorAction()
    {
        return action instanceof ErrorAction;
    }

    /** Getter for action parameter
     *
     * @return parameter value or null if none set
     */
    public Value getActionParameter()
    {
        return actionParameter;
    }


    /**
     * <p>Make a new WorkflowTransition</p> including creating a URI for it.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param prefURI
     * @param label short name for Transition. SHOULD be non-null
     * @param comment longer description of WorkflowTransition may be null
     * @param finalState
     * @param initialState
     * @return
     * @throws javax.servlet.ServletException if any.
     */
    public static WorkflowTransition create(HttpServletRequest request, URI prefURI, String label, String comment, URI initialState, URI finalState)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to create a new Transition.");

        // sanity checks
        if (label == null || label.trim().length() == 0)
            throw new BadRequestException("Label is required when creating a new Transition.");
        if (initialState == null || finalState == null)
            throw new BadRequestException("Initial and Final WorkflowStates are required when creating a new Transition.");

            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            // forge a new subject URI
            URI subject = (prefURI != null) ? prefURI :
                                vf.createURI(Configuration.getInstance().getDefaultNamespace(),
                                                RIDGenerator.getInstance().newID().toString());
            WorkflowTransition result = new WorkflowTransition(subject, label, comment, initialState, finalState);
            log.debug("Created new transition: "+result);
            result.setDirty(true);
            result.write(request, false);
            return result;
    }

    /**
     * <p>Write the properties of a WorkflowTransition into RDF database.
     * Used when creating new instance (either creat() or from import)
     * to record the RDF. </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param makeURI option to create a new subject URI.
     * @return
     * @returns the subject URI (this is necessary if it made up a new one)
     * @throws javax.servlet.ServletException if any.
     */
    public URI write(HttpServletRequest request, boolean makeURI)
        throws ServletException
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            if (makeURI)
                uri = vf.createURI(Configuration.getInstance().getDefaultNamespace(),
                                          RIDGenerator.getInstance().newID().toString());
            log.debug("Writing to RDF database: "+this);
            rc.add(uri, RDF.TYPE, REPO.WORKFLOW_TRANSITION, REPO.NG_INTERNAL);
            rc.add(uri, RDFS.LABEL, vf.createLiteral(label), REPO.NG_INTERNAL);
            if (comment != null)
                rc.add(uri, RDFS.COMMENT, vf.createLiteral(comment), REPO.NG_INTERNAL);
            rc.add(uri, REPO.INITIAL, initialState, REPO.NG_INTERNAL);
            rc.add(uri, REPO.FINAL, finalState, REPO.NG_INTERNAL);
            if (workspace != null)
                rc.add(uri, REPO.WORKSPACE, workspace, REPO.NG_INTERNAL);
            if (action != null)
                rc.add(uri, REPO.ACTION, vf.createLiteral(action.getClass().getName()), REPO.NG_INTERNAL);
            if (actionParameter != null)
                rc.add(uri, REPO.ACTION_PARAMETER, actionParameter, REPO.NG_INTERNAL);
            if (order != null)
                rc.add(uri, REPO.ORDER, order, REPO.NG_INTERNAL);
            return uri;
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /** Obliterate all trace of this transition.
     *
     * @param request
     * @throws ServletException
     */
    public void delete(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to delete a Transition.");
        log.debug("Deleting: "+this);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(uri, null, null, REPO.NG_INTERNAL);
            setDirty(true);
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>Setter for the field <code>label</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param val
     * @throws javax.servlet.ServletException if any.
     */
    public void setLabel(HttpServletRequest request, String val)
        throws ServletException
    {
        if (val == null || val.length() == 0)
            throw new BadRequestException("Label must be a non-empty string.");
        label = val;
        setMetadataInternal(request, RDFS.LABEL, new LiteralImpl(val));
    }

    /** Setter for comment field - null clears the comment.
     *
     * @param request
     * @param val new value
     * @throws ServletException
     */
    public void setComment(HttpServletRequest request, String val)
        throws ServletException
    {
        comment = val;
        setMetadataInternal(request, RDFS.COMMENT, val == null ? null : new LiteralImpl(val));
    }

    /** Setter for workspace field - null clears it
     *
     * @param request
     * @param val new value (null to clear it)
     * @throws ServletException
     */
    public void setWorkspace(HttpServletRequest request, URI val)
        throws ServletException
    {
        workspace = val;
        setMetadataInternal(request, REPO.WORKSPACE, val);
    }

    /** Setter for initial staet
     *
     * @param request
     * @param val new value, must be URI of workflow state
     * @throws ServletException
     */
    public void setInitial(HttpServletRequest request, URI val)
        throws ServletException
    {
        if (val == null)
            throw new BadRequestException("New state value may not be null");
        initialState = val;
        setMetadataInternal(request, REPO.INITIAL, val);
    }

    /** setter for final state
     *
     * @param request
     * @param val new value, must be workflow stte
     * @throws ServletException
     */
    public void setFinal(HttpServletRequest request, URI val)
        throws ServletException
    {
        if (val == null)
            throw new BadRequestException("New state value may not be null");
        finalState = val;
        setMetadataInternal(request, REPO.FINAL, val);
    }

    /** setter for order
     *
     * @param request
     * @param val new value, should be string encoding of integer or null to clear it
     * @throws ServletException
     */
    public void setOrder(HttpServletRequest request, String val)
        throws ServletException
    {
        Literal litVal = null;
        try {
            if (val != null)
                litVal = new NumericLiteralImpl(Integer.decode(val), XMLSchema.INTEGER);
        } catch (NumberFormatException e) {
            log.debug("setOrder: got non-integer value = \""+val+"\"");
            if (val != null)
                litVal = new LiteralImpl(val);
        }
        order = litVal;
        setMetadataInternal(request, REPO.ORDER, litVal);
    }

    /** Setter for action class
     *
     * @param request
     * @param val new value, fully-qualifed name of a class implementing WorkflowAction
     * @throws ServletException
     */
    public void setAction(HttpServletRequest request, String val)
        throws ServletException
    {
        if (val != null) {
            try {
                Object newAction = Class.forName(val).newInstance();
                if (newAction instanceof WorkflowAction)
                    action = (WorkflowAction)newAction;
                else
                    throw new BadRequestException("Action class does not implement WorkflowAction: "+val);
            } catch (Exception e) {
                throw new BadRequestException("Failed to instantiate Action class: "+val+": "+e);
            }
        }
        setMetadataInternal(request, REPO.ACTION, val == null ? null : new LiteralImpl(val));
    }

    /**
     *  Setter for ActionParameter
     * @param request
     * @param str new value - may be stringified URI or literal; parse out URI or literal.
     * @throws ServletException
     */
    public void setActionParameter(HttpServletRequest request, String str)
        throws ServletException
    {
        Value val = null;
        if (str != null) {
            if (Utils.isValidURI(str))
                val = new URIImpl(str);
            else
                val = new LiteralImpl(str);
        }
        setActionParameter(request, val);
    }

    /** Setter for action parameter
     *
     * @param request
     * @param val new value or null to clear it.
     * @throws ServletException
     */
    public void setActionParameter(HttpServletRequest request, Value val)
        throws ServletException
    {
        actionParameter = val;
        setMetadataInternal(request, REPO.ACTION_PARAMETER, val);
    }

    // update one of the metadata properties.
    private void setMetadataInternal(HttpServletRequest request, URI property, Value newVal)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to modify Transitions.");
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(uri, property, null, REPO.NG_INTERNAL);
            if (newVal != null) {
                rc.add(uri, property, newVal, REPO.NG_INTERNAL);
                log.debug("Setting Transition uri="+uri+", property="+property+" to value="+newVal);
            } else
                log.debug("Clearing Transition uri="+uri+", property="+property);
            setDirty(true);
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * Tuple query result handler to gather list of all transitions (or one);
     * removes duplicates since data errors can produce them.
     * Note this is public since it is shared by TransportWorkflowTransition
     */
    public static class WFTHandler extends TupleQueryResultHandlerBase
    {
        private List<WorkflowTransition> result = null;
        private Set<URI> uniqueURI = new HashSet<URI>();

        /** Constructor */
        WFTHandler(List<WorkflowTransition> r)
        {
            super();
            result = r;
        }

        /** {@inheritDoc} */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("uri");
            if (newURI instanceof URI) {
                URI u = (URI)newURI;
                Value rlabel = bs.getValue("label");
                String label = (rlabel instanceof Literal) ?
                           ((Literal)rlabel).getLabel() : "";
                Value rcomment = bs.getValue("comment");
                String comment = null;
                if (rcomment instanceof Literal)
                    comment = ((Literal)rcomment).getLabel();
                Value rinitial = bs.getValue("initial");
                if (!(rinitial instanceof URI))
                    log.error("Should not get null or non-URI result for initial in transitionHandler: "+rinitial);
                URI initialState = (URI)rinitial;
                Value rfinal = bs.getValue("final");
                if (!(rfinal instanceof URI))
                    log.error("Should not get null or non-URI result for final in transitionHandler: "+rfinal);
                URI finalState = (URI)rfinal;
                // Check for duplicate results, which do NOT get eliminated:
                if (uniqueURI.contains(u)) {
                    log.warn("SKIPPING duplicate SPARQL query result for transition uri="+u+", label="+label+", initial="+initialState+" , final="+finalState);
                    return;
                }
                WorkflowTransition t = new WorkflowTransition(u, label, comment, initialState, finalState);
                // optional workspace
                Value rworkspace = bs.getValue("workspace");
                if (rworkspace instanceof URI)
                    t.workspace = (URI)rworkspace;
                // optional Action
                Value raction = bs.getValue("action");
                // optional labels
                Value rwl = bs.getValue("workspaceLabel");
                if (rwl != null)
                    t.workspaceLabel = Utils.valueAsString(rwl);
                Value ril = bs.getValue("initialLabel");
                if (ril != null)
                    t.initialLabel = Utils.valueAsString(ril);
                Value rfl = bs.getValue("finalLabel");
                if (rfl != null)
                    t.finalLabel = Utils.valueAsString(rfl);
                Value rord = bs.getValue("order");
                if (rord instanceof Literal)
                    t.order = (Literal)rord;
                t.actionParameter = bs.getValue("actionParameter");
                if (raction instanceof Literal) {
                    String saction = Utils.valueAsString(raction);
                    try {
                        Object na = Class.forName(saction).newInstance();
                        if (na instanceof WorkflowAction) {
                            t.action = (WorkflowAction)na;
                        } else {
                            t.action = new ErrorAction("The configured Action class does not implement WorkflowAction: "+saction);
                            log.warn("For transaction="+t+", the configured Action class does not implement WorkflowAction: "+saction);
                        }
                    } catch (Exception e) {
                        t.action = new ErrorAction("Failed instantiating action class="+saction+": "+e);
                        log.warn("For transaction="+t+", failed instantiating action class="+saction+": "+e);
                    }
                }
                result.add(t);
                uniqueURI.add(u);
            } else {
                log.error("Should not get null or non-URI result in transtionHandler: "+newURI);
            }
        }
    }

    /**
     * <p>toString</p>
     *
     * @return a detailed {@link java.lang.String} representation of the object.
     */
    @Override
    public String toString()
    {
        return "<#Transition: uri="+uri.toString()+
            ", label="+label+">";
    }

    /** {@inheritDoc} Transitions are equal if their subject URI is the same. */
    @Override
    public boolean equals(Object other)
    {
        return other instanceof WorkflowTransition && uri.equals(((WorkflowTransition)other).uri);
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode()
    {
        return uri.hashCode();
    }
}
