package org.eaglei.repository.servlet;

import org.eaglei.repository.util.WithRepositoryConnection;
import java.io.Reader;
import java.io.FileReader;
import java.io.File;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.MalformedURLException;
import javax.xml.datatype.XMLGregorianCalendar;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;
import javax.servlet.ServletConfig;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.GraphQuery;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.OpenRDFException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;

import org.jdom.Namespace;
import org.jdom.Element;
import org.jdom.Document;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.transform.JDOMResult;
import org.jdom.transform.JDOMSource;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.Configuration;
import org.eaglei.repository.Lifecycle;
import org.eaglei.repository.util.Formats;
import org.eaglei.repository.HasContentCache;
import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.Provenance;
import org.eaglei.repository.model.NamedGraph;
import org.eaglei.repository.model.NamedGraphType;
import org.eaglei.repository.model.View;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Dissemination of the content for an Eagle-I Resource representation instance.
 * Exact content varies by the format:  if HTML is requested or
 * negotiated, output only includes properties that should be visible to
 * end users.  It is basic unstyled HTML since styles are applied via UI
 * customization.
 *  When an RDF serialization format is selected, output is a graph of
 * *all* relevant statements that authenticated user is entitled to know.
 *
 * Query Args:
 *   uri - alternate method of specifying URI
 *   format - override content negotiation with this MIME type
 *   view - use this view as source of RDF data; default is 'user'
 *
 * The generated XHTML includes the following classes on relevant tags,
 * to assist in styling and transformation:
 *      eaglei_resourceLabel
 *      eaglei_resourceURI
 *      eaglei_resourceProperties
 *      eaglei_resourceProperty
 *      eaglei_resourcePropertyTerm
 *      eaglei_resourcePropertyValue
 *      eaglei_resourceTypes
 *      eaglei_resourceType
 *      rdf_uriLocalName
 *      rdf_literal
 *      rdf_type_{XSD-type-localname}
 *      rdf_value
 *
 * @author Larry Stone
 * @version $Id: $
 */
@HasContentCache
public class Disseminate extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Disseminate.class);

    // source of TrAX transformers
    private static final TransformerFactory transformerFactory = TransformerFactory.newInstance();

    // used by JDOM to generate XHTML tags
    private static final String REPO_XML = "http://eagle-i.org/xsd/repo/";
    private static final Namespace REPO_XML_NS = Namespace.getNamespace(REPO_XML);

    // canonical XSI namespace
    private static final Namespace XSI_NAMESPACE = Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");

    // path of schema document, relative to context, depends on webapp hierarchy
    private static final String INSTANCE_SCHEMA_PATH = "repository/schemas/instance.xsd";

    // all label predicate URIs to look for (in resource instance stms)
    private static final URI labelPredicate[] = DataModel.LABEL_PREDICATE.getArrayOfURI();

    // switch controlling whether contact-hiding behavior is on.
    private static final boolean hideContactsEnabled = Boolean.parseBoolean(Configuration.getInstance().getConfigurationProperty("eaglei.repository.hideContacts", Boolean.FALSE.toString()));

    // URLs of JavaScript files to include in generated HTML.
    // There is a default which can be overridden by a configuration value
    // for Configuration.INSTANCE_JS, which is a comma-separated
    // list of URLs
    private static final String defaultScriptURLs[] = {
        "http://ajax.googleapis.com/ajax/libs/jquery/1.4.1/jquery.min.js",
        "/repository/styles/i.js"
    };
    private static final String scriptURLs[] =
        Configuration.getInstance().getConfigurationPropertyArray(Configuration.INSTANCE_JS, defaultScriptURLs);

    private boolean authenticated;

    /**
     * {@inheritDoc}
     *
     * Configure this servlet instance as "authenticated" or not.
     */
    @Override
    public void init(ServletConfig sc)
        throws ServletException
    {
        super.init(sc);

        // default is non-authenitcated
        String a = sc.getInitParameter("authenticated");
        authenticated = a != null && Boolean.parseBoolean(a);
    }

    /**
     * Invalidate any in-memory cache of RDF data.
     */
    public static void decache()
    {
    }

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String format = getParameter(request, "format", false);
        String rawView = getParameter(request, "view", false);
        URI uri = getParameterAsURI(request, "uri", false);
        URI workspace = getParameterAsURI(request, "workspace", false);
        boolean noinferred = isParameterPresent(request, "noinferred");
        boolean forceRDF = isParameterPresent(request, "forceRDF");
        boolean forceXML = isParameterPresent(request, "forceXML");

        // deduce resource URI from arg or path
        if (uri == null) {
            String pi = request.getPathInfo();
            if (pi == null || pi.length() == 0)
                throw new BadRequestException("Missing required instance URI to disseminate.");
            uri = Utils.parseURI(Configuration.getInstance().getDefaultNamespace() + pi.substring(1),
                                 "Resource URI", true);
        }
        log.debug("Got requested resource uri="+uri.stringValue());

        // sanity check: cannot specify view and workspace
        if (workspace != null && rawView != null)
            throw new BadRequestException("The 'view' and 'workspace' arguments are mutually exclusive.  Choose only one.");

        // more sanity and security checking: to be reasonably sure
        // the URI refers to a resource instance, make sure there is
        // a rdf:type statement about it in a graph of type Public or WOrkspace
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // configuration sanity check
            if (labelPredicate == null)
                throw new ServletException("Configuration failure, no label predicates found.");

            // populate dataset from workspace or view, or user's
            // visible graphs by default.
            DatasetImpl ds = new DatasetImpl();
            View view = null;
            if (workspace != null) {
                View.addWorkspaceGraphs(request, ds, workspace);
            } else {
                view = (rawView == null) ? View.USER : View.parseView(rawView);
                if (view == null)
                    throw new BadRequestException("Unknown view: "+rawView);
                View.addGraphs(request, ds, view);
            }

            //  Use prettyPrint because ds.toString() breaks when a default graph is null
            //  which is possible for the 'null' view.  don't ask, it's ugly.
            if (log.isDebugEnabled())
                log.debug("Dataset derived from initial 'view' or 'workspace' args =\n"+Utils.prettyPrint(ds,2));

            // Heuristic to check whether URI is subject of a legitimate
            // resource: does it have a rdf:type statement in a named
            // graph that is expected to contain only resources?
            // Unfortunately SPARQL can't act on context URIs so we
            // have to do it the hard way:
            RepositoryResult<Statement> rr = null;
            boolean goodResource = false;
            boolean exists = false;
            boolean visible = false;
            boolean any = false;
            URI contexts[] = View.NULL.equals(view) ? new URI[1] :
                             ds.getDefaultGraphs().toArray(new URI[ds.getDefaultGraphs().size()]);
            URI homeGraph = null;
            try {
                rr = rc.getStatements(uri, RDF.TYPE, null, false, contexts);

                // if no types are visible, check ALL graphs just to be sure:
                if (!rr.hasNext()) {
                    log.debug("Probing for rdf:type in ALL graphs, after failing in allowed ones");
                    exists = rc.hasStatement(uri, RDF.TYPE, null, false);
                    any = rc.hasStatement(uri, null, null, false, contexts);
                }
                while (rr.hasNext()) {
                    Statement s = rr.next();
                    Resource ctx = s.getContext();
                    log.debug("Found statement: "+uri+" rdf:type "+s.getObject()+", in graph "+ctx);
                    exists = true;

                    // shortcut: test for read access by checking if context
                    //  URI is listed in Dataset, instead of slower call
                    //  to  Access.hasPermission(request, (URI)ctx, REPO.NAMED_GRAPH, Access.READ)
                    if (ctx instanceof URI &&
                          // XXX FIXME: temp kludge to accomodate NULL view:
                          (View.NULL.equals(view) ?
                           Access.hasPermission(request, (URI)ctx, Access.READ) :
                           ds.getDefaultGraphs().contains((URI)ctx))) {
                        visible = true;
                        NamedGraph cng = NamedGraph.find(request, (URI)ctx);
                        NamedGraphType cngt = cng == null ? null : cng.getType();
                        if (cngt == NamedGraphType.published || cngt == NamedGraphType.workspace) {
                            log.debug("...approving resource because of rdf:type statement in graph="+ctx);
                            goodResource = true;
                            homeGraph = (URI)ctx;
                            break;
                        }
                    }
                }
            } finally {
                rr.close();
            }
            log.debug("Probe results: exists="+exists+", any="+any+", visible="+visible+", goodResource="+goodResource);

            // NOTE: Deny it "exists" if user does not have read permission so
            //  we do not give away information they would otherwise not know.
            // Leave a log message in case admin wants to diagnose the 404
            //  so they'll know it was caused by access failure
            if (exists && !visible) {
                log.info("Resource exists but is not visible to user, so returning 404.  User="+Authentication.getPrincipalURI(request)+", resource="+uri);
                throw new NotFoundException("Subject not found in this repository: "+uri.toString());
            
            // Give a cryptic hint if there are some statements found but no rdf:type
            // This is usually caused by all rdf:type's being removed when
            // instance is deleted, and metadata persists.
            } else if (!exists) {
                if (any)
                    throw new HttpStatusException(HttpServletResponse.SC_GONE, "Subject was deleted or is not a well formed eagle-i resource: "+uri.toString());
                else
                    throw new NotFoundException("Subject not found in this repository: "+uri.toString());

            // We can read it, but it isn't in a Published or Workspace graph.
            // This tries to protect the URI resolution service from
            // being abused to romp through the triplestore..
            // NOTE: allow superuser to abuse Disseminate since it may
            // help him/her browse the repository.
            } else if (!goodResource || homeGraph == null) {
                if (!Authentication.isSuperuser(request))
                    throw new NotFoundException("Subject exists but is not available as an Eagle-I resource: "+uri.toString());
            }

            // Is HTML an acceptable format? format arg is override, otherwise check accepts
            String mimeType = Formats.negotiateHTMLorRDFContent(request, format);

            // XXX this is a terrible kludge, but that's the nature of MIME..
            // see http://www.w3.org/TR/xhtml-media-types also RFC3236
            boolean html = !forceRDF &&
                           (mimeType.equals("text/html") ||
                            mimeType.equals("application/xhtml+xml"));
            String contentType = Utils.makeContentType(html ? "application/xhtml+xml": mimeType, "UTF-8");

            log.debug("Preparing query for authenticated="+authenticated+", hideContactsEnabled="+hideContactsEnabled);
            StringBuilder query = new StringBuilder();
            if (html) {
                query.append("SELECT DISTINCT ?g ?p ?v");
                for (int i = 0; i < labelPredicate.length; ++i) {
                    query.append(" ?pl").append(String.valueOf(i)).
                          append(" ?vl").append(String.valueOf(i));
                }
            } else {
                query.append("CONSTRUCT { ?subject ?p ?v . "/*}*/);
                for (int i = 0; i < labelPredicate.length; ++i) {
                    query.append(" ?p <").append(labelPredicate[i]).
                      append("> ?pl").append(String.valueOf(i)).append(" .").
                    append(" ?v <").append(labelPredicate[i]).
                      append("> ?vl").append(String.valueOf(i)).append(" .").
                    append(" ?vo <").append(labelPredicate[i]).
                      append("> ?vol").append(String.valueOf(i)).append(" .");
                }
                query.append(/*{*/" ?v ?vp ?vo }\n");
            }
            query.append(" WHERE { \n"/*}*/);
            query.append("{ GRAPH ?g { ?subject ?p ?v } \n"/*}*/);
            for (int i = 0; i < labelPredicate.length; ++i) {
                query.append(" OPTIONAL { ?p <"/*}*/).append(labelPredicate[i]).
                    append("> ?pl").append(String.valueOf(i)).append(/*{*/" } \n").
                  append(" OPTIONAL { ?v <"/*}*/).append(labelPredicate[i]).
                    append("> ?vl").append(String.valueOf(i)).append(/*{*/" } \n");
            }

            // pick up embedded class instances for CONSTRUCT result
            if (!html) {
                query.append("OPTIONAL { ?v a ?vt . ?vt <"/*}*/).
                      append(DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()).append("> <").
                      append(DataModel.EMBEDDED_INSTANCE_OBJECT.toString()).append("> . ");

                if (noinferred) {
                    query.append("GRAPH ?eg ");
                }
                // add all properties of the embedded instance
                query.append("{?v ?vp ?vo}\n");
                if (noinferred) {
                    query.append(" FILTER( ?eg != <"/*)*/).append(REPO.NG_INFERRED).append(/*(*/">)\n");
                }
        // XXX THIS IS BAD, need other metric than just authenticated.
        //  ideally some kind of ACL on props, but that's probly too expensive.
                if (!authenticated) {
                    addPropertyFilters(query, "vp", !authenticated);
                }
                // add labels of object properties of the EI...where does this end?!
                for (int i = 0; i < labelPredicate.length; ++i) {
                    query.append(" OPTIONAL { ?vo <"/*}*/).append(labelPredicate[i]).
                      append("> ?vol").append(String.valueOf(i)).append(/*{*/" } \n");
                }

                // close the OPTIONAL
                query.append(/*{*/"}\n");
            }
        // XXX THIS IS BAD, need other metric than just authenticated.
        //  ideally some kind of ACL on props, but that's probly too expensive.
            if (!authenticated) {
                addPropertyFilters(query, "p", !authenticated);
            }
            query.append(/*{*/" }");
            if (noinferred) {
                query.append(" FILTER( ?g != <"/*)*/).
                      append(REPO.NG_INFERRED).append(/*(*/">)\n");
            }
            query.append(/*{*/"}");
            if (html) {
                query.append(" ORDER BY ?p ?v");
            }

            // Dataset to use in results query: unless there is an explicit
            // view arg, compose it out of metadata + onto + home graph.
            DatasetImpl qds = null;
            if (rawView == null && workspace == null) {
                qds = new DatasetImpl();
                View.addWorkspaceGraphs(request, qds, homeGraph);
            } else {
                qds = ds;
            }
            if (log.isDebugEnabled()) {
                log.debug("Home graph = "+homeGraph);
            }
            if (html) {
                Element assertedTypes = new Element("asserted-types", REPO_XML_NS);
                Element inferredTypes = new Element("inferred-types", REPO_XML_NS);
                Element properties = new Element("properties", REPO_XML_NS);
                Element metadata = new Element("metadata", REPO_XML_NS);

                String qs = query.toString();
                TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
                q.setBinding("subject", uri);
                // if specified view arg was "null", DO NOT set the dataset:
                if (!View.NULL.equals(view))
                    q.setDataset(qds);
                PropsHandler handler = new PropsHandler(assertedTypes, inferredTypes, properties, metadata);
                SPARQL.evaluateTupleQuery(qs, q, handler);

                Element root = new Element("instance", REPO_XML_NS);
                root.addNamespaceDeclaration(Namespace.XML_NAMESPACE);
                root.addNamespaceDeclaration(XSI_NAMESPACE);
                root.setAttribute("schemaLocation",
                  REPO_XML+" "+makeFileURL(request, INSTANCE_SCHEMA_PATH), XSI_NAMESPACE);
                root.addContent(makeResource(uri, handler.getInstanceLabel()));
                root.addContent(new Element("permission", REPO_XML_NS).
                                  setAttribute("write", String.valueOf(Access.hasPermission(request, uri, Access.ADD))).
                                  setAttribute("admin", String.valueOf(Access.hasPermission(request, uri, Access.ADMIN))));
                root.addContent(assertedTypes);
                root.addContent(inferredTypes);
                root.addContent(properties);
                root.addContent(metadata);
                Document result = new Document(root);
                    if (!result.hasRootElement())
                        log.error("No root element in doc, this will be bad.");
                     
                // set last-modified header if available
                try {
                    XMLGregorianCalendar mc = handler.getModified();
                    if (mc != null) {
                        log.debug("Got last-modified date = "+mc.toString());
                        response.addDateHeader("Last-Modified",
                                mc.toGregorianCalendar().getTime().getTime());
                    }
                } catch (IllegalArgumentException e) {
                    log.warn("Failed to parse dcterms:created or dcterms:modified as date; value="+handler.modified);
                }

                // do we have optional XSL stylesheet to transform result?
                // if path is relative, look in webapp, otherwise, on filesystem
                String xslPath = Configuration.getInstance().getConfigurationProperty(Configuration.INSTANCE_XSLT);
                if (xslPath != null && !forceXML) {
                    log.debug("Transforming result with XSLT in file = "+xslPath);
                    File xslFile = new File(xslPath);
                    Reader xis = xslFile.isAbsolute() ?
                                        new FileReader(xslFile) :
                                        Lifecycle.getInstance().getWebappResourceAsReader(xslPath);
                    Transformer xf = transformerFactory.newTransformer(new StreamSource(xis));
                    xf.setParameter("__repo_version", Configuration.getInstance().getProjectVersion());
                    String css = Configuration.getInstance().getConfigurationProperty(Configuration.INSTANCE_CSS);
                    if (css != null)
                        xf.setParameter("__repo_css", css);
                    String logo = Configuration.getInstance().getConfigurationProperty("eaglei.repository.logo");
                    if (logo != null)
                        xf.setParameter("__repo_logo", logo);
                    String repoTitle = Configuration.getInstance().getConfigurationProperty("eaglei.repository.title");
                    if (repoTitle != null)
                        xf.setParameter("__repo_title", repoTitle);
                    JDOMResult xresult = new JDOMResult();
                    xf.transform(new JDOMSource(result), xresult);
                    result = xresult.getDocument();

                // emit the un-transformed XML
                // XXX IWBNI there were an option to add xslt annotation like
                //  <?xml-stylesheet type="text/xsl" href="{url-of-stylesheet}"?>
                // Note that we *don't* do this generally because (a) browsers
                // silently ignore cross-site URLs; (b) cannot pass params
                // to the XSL stylesheet this way.
                } else {
                    log.debug("No XSLT stylesheet configured or forceXML, returning raw XML.");

                    // XXX should add a charset, ideally "charset=utf-8"..
                    contentType = "application/xml";
                }

                XMLOutputter out = new XMLOutputter();
                out.setFormat(Format.getPrettyFormat());
                response.setContentType(contentType);
                out.output(result, response.getOutputStream());

            // For RDF dissemination, just serialze CONSTRUCT query results.
            } else {
                RDFFormat rf = Formats.getRDFOutputFormatForMIMEType(mimeType);
                if (rf == null) {
                    log.error("Failed to get RDF serialization format, for mime="+mimeType);
                    throw new ServletException("Failed to get RDF serialization format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(contentType);
                    Provenance p = new Provenance(uri);
                    String modified = p.getField(request, DCTERMS.MODIFIED);
                    if (modified != null) {
                        XMLGregorianCalendar mc = Utils.parseXMLDate(modified);
                        log.debug("Got last-modified date = "+mc.toString());
                        response.addDateHeader("Last-Modified", mc.toGregorianCalendar().getTime().getTime());
                    }
                    String gqs = query.toString();
                    GraphQuery q = rc.prepareGraphQuery(QueryLanguage.SPARQL, gqs);
                    q.setBinding("subject", uri);
                    q.setDataset(qds);
                    SPARQL.evaluateGraphQuery(gqs, q, Rio.createWriter(rf, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));
                }
            }
        } catch (TransformerException e) {
            log.error("Failed transforming instance result, uri="+uri, e);
            throw new ServletException("Failed in XSL Transformation:",e);

        } catch (MalformedQueryException e) {
            log.error("Malformed query generated internally: ",e);
            throw new ServletException("Malformed query generated internally: "+e.toString(), e);

        // this "should never happen", means the SAIL or repo does not support generic prepareQuery()
        } catch (UnsupportedOperationException e) {
            log.error(e);
            throw new ServletException(e);

        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // add clauses to eliminate properties that should not be shown
    // call from within a triple pattern.
    private void addPropertyFilters(StringBuilder query, String propName, boolean hide)
    {
        String hideObj = propName + "_hide";
        String contactObj = propName + "_contact";
        if (hide) {
            query.append(" OPTIONAL { ?"/*}*/).append(propName).
             append(" <").append(DataModel.HIDE_PROPERTY_PREDICATE.toString()).append("> ?").
             append(hideObj).append("\n");
            query.append("  FILTER(?"/*)*/).append(hideObj).append(" = <").
             append(DataModel.HIDE_PROPERTY_OBJECT.toString()).append(/*{(*/">)}\n");
        }
        if (hideContactsEnabled) {
            query.append(" OPTIONAL { ?"/*}*/).append(propName).
             append(" <").append(DataModel.CONTACT_PROPERTY_PREDICATE.toString()).append("> ?").
             append(contactObj).append("\n");
            query.append("  FILTER(?"/*)*/).append(contactObj).append(" = <").
             append(DataModel.CONTACT_PROPERTY_OBJECT.toString()).append(/*{(*/">)}\n");
        }
        query.append("FILTER(!(BOUND(?"/*)))*/).append(hideObj).append(/*(*/")");
        if (hideContactsEnabled) {
            query.append(" || BOUND(?"/*)*/).append(contactObj).append(/*(*/")");
        }
        query.append(/*((*/"))\n");
    }

    // add each property-value result to JDOM model of HTML output,
    // except the rdfs:label value becomes instance title.
    private final class PropsHandler extends TupleQueryResultHandlerBase
    {
        // resulting list tags
        private Element props;
        private Element provenance;
        private Element directTypes;
        private Element inferredTypes;

        // previous solution's values with which to remove duplicates
        // caused by duplicate labels..
        private Value lastP = null;
        private Value lastV = null;

        // resource label
        private String label = null;

        // last-modified date
        private Literal modified = null;

        private PropsHandler(Element directTypes, Element inferredTypes,  Element props, Element provenance)
        {
            super();
            PropsHandler.this.directTypes = directTypes;
            PropsHandler.this.inferredTypes = inferredTypes;
            PropsHandler.this.props = props;
            PropsHandler.this.provenance = provenance;
        }

        // get the label found in the query results
        private String getInstanceLabel()
        {
            return label;
        }

        private XMLGregorianCalendar getModified()
        {
            return modified == null ? null : modified.calendarValue();
        }

        /**
         * {@inheritDoc}
         *
         * Expect vars: ?p ?v ?pl0...n, ?vl0..n
         */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value p = bs.getValue("p");
            Value v = bs.getValue("v");
            if (p == null || v == null)
                return;

            // skip duplicate record - these are very likely caused by
            // values with duplicate labels, e.g.
            //  <http://purl.org/obo/owl/NCBITaxon#NCBITaxon_9031>
            //   <http://purl.obolibrary.org/obo/IAO_0000111>
            //   "chicken" .
            //  <http://purl.org/obo/owl/NCBITaxon#NCBITaxon_9031>
            //   <http://purl.obolibrary.org/obo/IAO_0000111>
            //   "Gallus gallus" .
            // .. makes the sparql query yield 2 tuple results for one prop.
            if (p.equals(lastP) && v.equals(lastV)) {
                log.debug("Skipping duplicate record pred="+p+", value="+v);
                return;
            }
            lastP = p;
            lastV = v;

            log.debug("handleSolution: Gathering labels for predicate="+p+", value="+v);
            Value vl = null;
            for (int i = 0; i < labelPredicate.length; ++i) {
                String key = "vl"+String.valueOf(i);
                if (bs.hasBinding(key)) {
                    vl = bs.getValue(key);
                    if (log.isDebugEnabled())
                        log.debug("handleSolution: "+key+"=\""+vl+"\"");
                    break;
                }
            }
            String vls = vl == null ? null : Utils.valueAsString(vl);

            // when predicate is rdfs:label, just set the label
            Value g = bs.getValue("g");
            if (RDFS.LABEL.equals(p)) {
                label = Utils.valueAsString(v);
                log.debug("Setting instance's rdfs:label = "+label);
            }

            // if type, add to asserted or inferred tyeps (is graph inferred?)
            else if (RDF.TYPE.equals(p)) {
                Element te = makeResource((URI)v, vls);
                if (REPO.NG_INFERRED.equals(g)) {
                    inferredTypes.addContent(te);
                } else {
                    directTypes.addContent(te);
                }
                     
            // add to general properties OR metadata
            } else {
                Element predicate = new Element("predicate", REPO_XML_NS);
                Element object = new Element("object", REPO_XML_NS);
                Element property = new Element("property", REPO_XML_NS).
                                       addContent(predicate).
                                       addContent(object);

                Value pl = null;
                for (int i = 0; i < labelPredicate.length; ++i) {
                    String key = "pl"+String.valueOf(i);
                    if (bs.hasBinding(key)) {
                        pl = bs.getValue(key);
                        if (log.isDebugEnabled())
                            log.debug("handleSolution: p="+p+", "+key+"=\""+pl+"\"");
                        break;
                    }
                }
                predicate.addContent(makeResource((URI)p, pl == null ? null : Utils.valueAsString(pl)));

                if (v instanceof URI) {
                    object.addContent(makeResource((URI)v, vls));

                } else {
                    Element lit = new Element("literal", REPO_XML_NS).
                                    addContent(Utils.valueAsString(v));
                    if (v instanceof Literal) {
                        URI vdt = ((Literal)v).getDatatype();
                        if (vdt != null)
                            lit.setAttribute("type", vdt.toString());
                        String lang = ((Literal)v).getLanguage();
                        if (lang != null)
                            lit.setAttribute("lang", lang, Namespace.XML_NAMESPACE);
                    }
                    object.addContent(lit);
                }

               // choose either provenance or value properties, by graph:
                if (REPO.NG_METADATA.equals(g)) {
                    provenance.addContent(property);
                    if (DCTERMS.MODIFIED.equals(p)) {
                        if (v instanceof Literal) {
                            modified = (Literal)v;
                        } else {
                            log.warn("Unexpected non-Literal value of dcterms:modified, value="+v.toString());
                        }
                    }
                } else {
                    props.addContent(property);
                }
            }
        }
    }

    // Create the resource element for a URI.
    // If no label supplied, default to the last element of the URI path
    private Element makeResource(URI uri, String label)
    {
        return new Element("resource", REPO_XML_NS).
                     setAttribute("uri", uri.stringValue()).
                     addContent(label == null ? uri.getLocalName() : label);
    }

    // create a URL referring to a file in the webapp, e.g. the schema document
    private String makeFileURL(HttpServletRequest request, String path)
        throws ServletException
    {
        try {
            StringBuilder result = new StringBuilder();
            URL from = new URL(request.getRequestURL().toString());
            result.append(from.getProtocol()).append("://").append(from.getHost());
            if (from.getPort() >= 0)
                result.append(":").append(String.valueOf(from.getPort()));
            result.append("/").append(path);
            return result.toString();
        } catch (MalformedURLException e) {
            log.error("Failed reading request URL", e);
            throw new ServletException("Failed reading request URL", e);
        }
    }
}
