package org.eaglei.repository.servlet;

import org.eaglei.repository.util.WithRepositoryConnection;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.HashSet;
import java.util.Set;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.TupleQuery;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.BooleanLiteralImpl;
import org.openrdf.model.BNode;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.OpenRDFException;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.QueryResultIO;
import org.openrdf.query.TupleQueryResultHandler;
import org.openrdf.query.impl.MapBindingSet;

import org.eaglei.repository.Lifecycle;
import org.eaglei.repository.util.Formats;
import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.NamedGraph;
import org.eaglei.repository.model.NamedGraphType;
import org.eaglei.repository.model.View;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Harvest service - retrieves listing of eagle-i resource instances in order
 * to "harvest" metadata to build and maintain external indexes.
 * The optional "from" argument helps the client maintain an index efficiently
 * by reporting ONLY the resource instances which have changed from the
 * indicated timestamp onward, e.g. since the last index update.  The most
 * common case, where nothing has changed since the last update, is
 * heavily optimized to avoid querying the triplestore at all.
 * The optional 'after' argument, mutually exclusive with 'from', is just
 * like from only the time restriction statrts after rather than on hte
 *  given timestamp.
 *
 * When given time bounds, /harvest reports not only which resources have
 * been updated, but also on resources which have been deleted or otherwise
 * removed from view (e.g. withdrawn).
 *
 * Query Args:
 *   format - override content negotiation with this MIME type
 *   view - use this view as source of RDF data; default is 'user'
 *   workspace - URI of workspace, mutually excl with view
 *   from - optional start date, inclusive
 *   after - optional start date, exclusive
 *   detail=(identifier|full) - no default
 *
 * @author Larry Stone
 * @version $Id: $
 * Started June 2010
 */
public class Harvest extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Harvest.class);

    /** values of 'detail' arg */
    public enum DetailArg { identifier, full };

    // URI prefix to indicate a deleted URI; just append the whole URI.
    private static final String DELETED_PREFIX = "info:/deleted#";

    private static final String column[] = { "subject", "predicate", "object"};
    private static List<String> columnNames = null;
    static {
        columnNames = Arrays.asList(column);
    }

    // ----------- Queries for detail=idenifier from=NULL
    // Get only top-level resources at detail=identifier (no embeds)
    // Expect to bind ?graph to the workspace/published graph
    private static final String detailIdNoTimeQuery =
        "SELECT DISTINCT ?subject WHERE {\n"+
        " GRAPH ?graph {?subject a ?typ}\n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?subject <"+DCTERMS.MODIFIED+"> ?mod}}";

    // Get ONLY the embedded instances in ?graph
    // Expect to bind ?graph to the workspace/published graph
    private static final String detailIdNoTimeEmbedsQuery =
        "SELECT DISTINCT ?subject WHERE {\n"+
        " GRAPH ?graph {?parent a ?typ ; ?pp ?subject }\n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?parent <"+DCTERMS.MODIFIED+"> ?mod}\n"+
        " ?subject a ?eit . ?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+">}";

    // ------------------ Queries for detail=full from=NULL
    // query to get *all* resources at detail=full
    // all the misery with ?graph is to get the instances on a specific
    // published/workspace graph along with JUST THEIR inferred types.
    // Need to bind: ?graph, includeInferred must be true on the query
    private static final String detailFullNoTimeQueryStart =
        "SELECT ?subject ?predicate ?object WHERE { \n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?subject <"+DCTERMS.MODIFIED+"> ?mod}\n";

    private static final String detailFullNoTimeQueryEnd =
        " OPTIONAL { ?predicate <"+DataModel.HIDE_PROPERTY_PREDICATE.toString()+"> ?hidep\n"+
        "            FILTER(?hidep = <"+DataModel.HIDE_PROPERTY_OBJECT.toString()+">)}\n"+
        " OPTIONAL { ?predicate <"+DataModel.CONTACT_PROPERTY_PREDICATE.toString()+"> ?contactp\n"+
        "            FILTER(?contactp = <"+DataModel.CONTACT_PROPERTY_OBJECT.toString()+">)}\n"+
        " FILTER(!(BOUND(?hidep) || BOUND(?contactp))) } ORDER BY ?subject";

    private static final String detailFullNoTimeQueryNoInferred =
        detailFullNoTimeQueryStart+
        " GRAPH ?graph {?subject a ?typ; ?predicate ?object} \n"+
        detailFullNoTimeQueryEnd;

    private static final String detailFullNoTimeQueryWithInferred =
        detailFullNoTimeQueryStart+
        " {GRAPH ?graph {?subject a ?typ; ?predicate ?object}} \n"+
        " UNION \n"+
        " {GRAPH ?graph {?subject a ?typ}\n"+
        "  GRAPH <"+REPO.NG_INFERRED+"> {?subject ?predicate ?object}}\n"+
        detailFullNoTimeQueryEnd;

    // Queries to get Embedded Instances - detail=full
    private static final String detailFullNoTimeQueryEmbedsStart =
        "SELECT ?subject ?predicate ?object WHERE { \n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?parent <"+DCTERMS.MODIFIED+"> ?mod}\n"+
        " GRAPH ?graph {?parent a ?ptyp; ?eip ?subject . ?subject a ?typ }\n"+
        " ?subject a ?eit . ?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+">\n";
        /*}*/

    private static final String detailFullNoTimeQueryEmbedsNoInferred =
        detailFullNoTimeQueryEmbedsStart+
        " GRAPH ?graph {?subject a ?typ; ?predicate ?object} \n"+
        detailFullNoTimeQueryEnd;

    private static final String detailFullNoTimeQueryEmbedsWithInferred =
        detailFullNoTimeQueryEmbedsStart+
        " {GRAPH ?graph {?subject a ?typ; ?predicate ?object}} \n"+
        " UNION \n"+
        " {GRAPH ?graph {?subject a ?typ}\n"+
        "  GRAPH <"+REPO.NG_INFERRED+"> {?subject ?predicate ?object}}\n"+
        detailFullNoTimeQueryEnd;

    // ------------------ Queries for detail=full from=TIMESTAMP
    // First portion of queries to get DELETED mod-time-bounded
    // resources at detail=identifier.  See versions below with and without
    // Withdrawn to accomodate workspace=:NG_Withdrawn.
    // Need to bind: ?from
    private static final String deletedFromTimeQueryProlog =
        "SELECT DISTINCT ?subject WHERE {\n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?subject <"+DCTERMS.MODIFIED+"> ?mod}\n"+
        " FILTER(?mod >= ?from)\n"+
        " OPTIONAL{ GRAPH ?g {?subject a ?t}} \n";
        /*}*/

    // Get both deleted AND withdrawn instances in time-bound query
    // Need to bind: ?from
    private static final String deletedAndWithdrawnFromTimeQuery =
        /*{*/
        deletedFromTimeQueryProlog +
        "  FILTER(!bound(?t) || ?g = <"+REPO.NG_WITHDRAWN+">) }\n"+
        "ORDER BY ?subject";
    // XXX is the ORDER BY necessary?

    // Get ONLY deleted instances in time-bound query -
    // don't worry about withdrawn because Dataset excludes that NG anyway.
    // Need to bind: ?from
    private static final String deletedNotWithdrawnFromTimeQuery =
        /*{*/
        deletedFromTimeQueryProlog +
        "  FILTER(!bound(?t)) }\n"+
        "ORDER BY ?subject";
    // XXX is the ORDER BY necessary?

    // ----------- detail=identifier
    // query to get mod-time-bounded resources at detail=identifier
    // Need to bind: ?graph, ?from
    private static final String identifierFromTimeQuery =
        "SELECT DISTINCT ?subject WHERE \n"+
        "{ GRAPH <"+REPO.NG_METADATA+"> { ?subject <"+DCTERMS.MODIFIED+"> ?mod}\n"+
        " FILTER( ?mod >= ?from )"+
        " GRAPH ?graph {?subject a ?type}} ORDER BY ?mod";

    // ----------- detail=full
    // query to get mod-time-bounded resources at detail=full --
    // shared prologue used by both inferred and non-inferred queries.
    private static final String fullFromTimeQueryStart =
        "SELECT DISTINCT ?subject ?predicate ?object WHERE {\n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?subject <"+DCTERMS.MODIFIED+"> ?mod}\n";

    // query to get mod-time-bounded resources at detail=full --
    // shared prologue used by both inferred and non-inferred queries.
    private static final String fullFromTimeQueryEnd =
        " OPTIONAL { ?predicate <"+DataModel.HIDE_PROPERTY_PREDICATE.toString()+"> ?hidep\n"+
        "            FILTER(?hidep = <"+DataModel.HIDE_PROPERTY_OBJECT.toString()+">)}\n"+
        " OPTIONAL { ?predicate <"+DataModel.CONTACT_PROPERTY_PREDICATE.toString()+"> ?contactp\n"+
        "            FILTER(?contactp = <"+DataModel.CONTACT_PROPERTY_OBJECT.toString()+">)}\n"+
        " FILTER(?mod >= ?from && !(BOUND(?hidep) || BOUND(?contactp))) } ORDER BY ?subject ?mod";

    // query to get mod-time-bounded resources at detail=full
    // Need to bind: ?graph, ?from
    private static final String fullFromTimeWithInferredQuery =
        fullFromTimeQueryStart+
        " {{GRAPH <"+REPO.NG_INFERRED+"> {?subject ?predicate ?object}}\n"+
        "   UNION\n"+
        "  {GRAPH ?graph {?subject a ?type; ?predicate ?object}}} \n"+
        fullFromTimeQueryEnd;

    private static final String fullFromTimeNoInferredQuery =
        fullFromTimeQueryStart+
        " GRAPH ?graph {?subject a ?type; ?predicate ?object} \n"+
        fullFromTimeQueryEnd;

    //-------- detail=full, From=X queries for Embedded Instances, include after parents

    // common prefix of queries
    private static final String fullEIsFromTimeQueryStart =
        "SELECT DISTINCT ?subject ?predicate ?object WHERE {\n"+
        " GRAPH <"+REPO.NG_METADATA+"> { ?parent <"+DCTERMS.MODIFIED+"> ?mod}\n"+
        " GRAPH ?graph {?parent a ?parentType; ?ppred ?subject . \n"+
        "                ?subject ?predicate ?object} \n"+
        " ?subject a ?eit . ?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+">\n";
        /*}*/

    // get properties of Embedded Instances (ONLY) which changed since time:
    // this version includes inferred
    private static final String fullEIsFromTimeNoInferredQuery =
        fullEIsFromTimeQueryStart+
        " GRAPH ?graph {?subject a ?type; ?predicate ?object}\n"+
        fullFromTimeQueryEnd;

    // get properties of Embedded Instances (ONLY) which changed since time:
    // this version includes inferred
    private static final String fullEIsFromTimeWithInferredQuery =
        fullEIsFromTimeQueryStart+
        " {{GRAPH <"+REPO.NG_INFERRED+"> {?subject ?predicate ?object}}\n"+
        "   UNION\n"+
        "  {GRAPH ?graph {?subject a ?type; ?predicate ?object}}} \n"+
        fullFromTimeQueryEnd;

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        // cache the current last-mod date *before* making any queries
        // to avoid possible race condition (spotted by dbw, 041811)
        Date lm = Lifecycle.getInstance().getLastModified();

        request.setCharacterEncoding("UTF-8");
        String format = getParameter(request, "format", false);
        String rawView = getParameter(request, "view", false);
        URI workspace = getParameterAsURI(request, "workspace", false);
        boolean inferred = getParameterAsBoolean(request, "inferred", false, false);
        boolean embedded = getParameterAsBoolean(request, "embedded", false, false);
        String rawfrom = getParameter(request, "from", false);
        String rawafter = getParameter(request, "after", false);
        DetailArg detail = (DetailArg)getParameterAsKeyword(request, "detail", DetailArg.class, null, true);

        // sanity check - 'until' not impl. yet
        if (isParameterPresent(request, "until"))
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "The 'until' arg is not implemented yet.");

        // sanity-check after & from; if 'after', add a millisecond..
        if (rawafter != null && rawfrom != null)
            throw new BadRequestException("The 'from' and 'after' args are mutually exclusive.");
        XMLGregorianCalendar from = null;
        if (rawfrom != null) {
            from = Utils.parseXMLDate(rawfrom);
        } else if (rawafter != null) {
            try {
                from = Utils.parseXMLDate(rawafter);
                from.add(DatatypeFactory.newInstance().newDuration(1L));
            } catch (DatatypeConfigurationException e) {
                throw new BadRequestException("Failed converting 'after' arg: ",e);
            }
        }

        // sanity check - 'inferred' not allowed here
        if (detail == DetailArg.identifier && inferred)
            throw new BadRequestException("The 'inferred' arg is not allowed when detail = identifier.");

        // sanity check: cannot specify view and workspace
        View view = null;
        if (workspace != null && rawView != null)
            throw new BadRequestException("The 'view' and 'workspace' arguments are mutually exclusive.  Choose only one.");
        if (rawView == null) {
            view = View.PUBLISHED_RESOURCES;
        } else {
            view = View.parseView(rawView);
            if (view == null)
                throw new BadRequestException("Unknown view: "+rawView);
        }

        long startMs = System.currentTimeMillis();
        try {
            // 'out' is final output handler - chosen by 'format' arg
            String mimeType = Formats.negotiateTupleContent(request, format);
            TupleQueryResultFormat tqf = QueryResultIO.getWriterFormatForMIMEType(mimeType);
            if (tqf == null) {
                throw new InternalServerErrorException("Failed to get tuple query format that SHOULD have been available, for mime="+mimeType);
            }
            response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
            TupleQueryResultHandler out = QueryResultIO.createWriter(tqf, response.getOutputStream());

            // Set last-modified header to repo's last-mod time so
            // client can gauge their next incremental request from this time
            // Also add more precise timestamp since HTTP dates only have
            // 1-second granularity, but our timekeeping is to the mSec.
            response.addDateHeader("Last-Modified", lm.getTime());
            response.addHeader("X-Precise-Last-Modified", Utils.makePreciseHTTPDate(lm));

            // Optimization: if nothing has changed since the 'from' time,
            // return an empty document.
            if (from != null) {
                if (lm.before(from.toGregorianCalendar().getTime())) {
                    log.debug("Optimizing result since last-modified mark is earlier than from: last-mod = "+lm);
                    out.startQueryResult(columnNames);
                    out.endQueryResult();
                    return;
                } else
                    log.debug("Going ahead with query, last-modified mark is after 'from': last-mod = "+lm);
            }
            harvest(request, detail, from, inferred, embedded, view, workspace, out);

            // XXX Maybe make this log.debug if polling bloats logs..
            // but it shouldn't, since most /harvest calls with from=X
            // will be optimized out without a SPARQL query and it is
            // valuable to have this in the log to watch for excessive
            // query overhead.
            log.info("SPARQL query for /harvest request completed in "+
                     String.format("%,d mSec.", System.currentTimeMillis()-startMs));
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // do the actual logic of producing the response
    private void harvest(HttpServletRequest request,
                         DetailArg detail, XMLGregorianCalendar from,
                         boolean inferred, boolean embedded, View view,
                         URI workspace, TupleQueryResultHandler out)
        throws  OpenRDFException, ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();

        // Construct a dataset for objects to report on from
        // workspace or view, published resource graphs by default.
        DatasetImpl ds = new DatasetImpl();
        if (workspace != null) {
            View.addWorkspaceGraphs(request, ds, workspace);
        } else {
            View.addGraphs(request, ds, view);
        }
        //  Use prettyPrint because ds.toString() breaks when a default graph is null
        //  which is possible for the 'null' view.  don't ask, it's ugly.
        if (log.isDebugEnabled())
            log.debug("Dataset derived from initial 'view' or 'workspace' args = \n"+Utils.prettyPrint(ds, 2));

        // Now, further filter out just the Published
        // and Workspace graphs, save in resGraphs:
        Set<URI> resGraphs = new HashSet<URI>();
        // XXX artifact: resDS used to be different..
        DatasetImpl resDS = ds;
        for (URI g : ds.getDefaultGraphs()) {
            // XXX KLUDGE: skip NG_Users graph, we don't want repo-users.
            //  unfortunately, it comes with Workspace datasets too.
            if (!REPO.NG_USERS.equals(g)) {
                NamedGraph ng = NamedGraph.find(request, g);
                if (ng == null)
                    throw new InternalServerErrorException("Failed to resolve named graph = "+g);
                NamedGraphType ngt = ng.getType();
                if (ngt == NamedGraphType.published || ngt == NamedGraphType.workspace) {
                    resGraphs.add(g);
                }
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Base (View/Workspace) Dataset ds = \n"+Utils.prettyPrint(ds, 2));
            log.debug("Resource Dataset resDS = \n"+Utils.prettyPrint(resDS, 2));
        }

        // no time bounds: get *all* resource identifiers and/or data:
        if (from == null) {
            TupleQuery q = null;
            TupleQuery qe = null;
            String qs = null;
            String qes = null;
            if (detail == DetailArg.identifier) {
                if (log.isDebugEnabled()) {
                    log.debug("Getting IDENTIFIER (ALL times, detail="+detail+") and "+
                              "(ALL times, detail="+detail+", embedded instances)");
                }
                qs = detailIdNoTimeQuery;
                if (embedded) {
                    qes = detailIdNoTimeEmbedsQuery;
                }

            // 'full' pattern is more complex, since we have to query
            // each graph of interest separately; get the graphs
            // from resource dataset (BEFORE adding inferred)
            } else {
                qs = inferred ? detailFullNoTimeQueryWithInferred : detailFullNoTimeQueryNoInferred;
                String qsEmbeds = inferred ? detailFullNoTimeQueryEmbedsWithInferred : detailFullNoTimeQueryEmbedsNoInferred;
                if (log.isDebugEnabled()) {
                    log.debug("Getting TOPLEVEL RESOURCES (ALL times, detail=full, inferred="+inferred+"), and "+
                              "EMBEDDED INSTANCES (ALL times, detail=full, inferred="+inferred+")");
                }
                qes = qsEmbeds;
            }
            q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);

            // common query execution over selected graphs:
            // NOTE needs inference for embedded-class matching
            q.setIncludeInferred(true);
            q.setDataset(resDS);
            if (qes != null) {
                qe = rc.prepareTupleQuery(QueryLanguage.SPARQL, qes);
                qe.setIncludeInferred(true);
                qe.setDataset(resDS);
            }
            out.startQueryResult(detail == DetailArg.identifier ?
                             columnNames.subList(0,1) : columnNames);
            for (URI graph : resGraphs) {
                if (log.isDebugEnabled())
                    log.debug("Querying Resource Instances from graph="+graph);
                q.clearBindings();
                q.setBinding("graph", graph);
                SPARQL.evaluateTupleQuery(qs, q, new StatementsOnlyWrapper(out));
                if (qe != null) {
                    qe.clearBindings();
                    qe.setBinding("graph", graph);
                    SPARQL.evaluateTupleQuery(qes, qe, new StatementsOnlyWrapper(out));
                }
            }
            out.endQueryResult();

        // from = TIMESTAMP; get results for args WITH a ?from limit:
        } else {

            // First, query for URIs of Deleted and Withdrawn resources;
            // NOTE resDS dataset must include NG_Metadata.

           // choose query based on whether NG_Withdrawn is in the dataset:
           // when it *isn't* (which is likely) we can use a simpler query:
           String qs =  resGraphs.contains(REPO.NG_WITHDRAWN) ?
                        deletedNotWithdrawnFromTimeQuery :
                        deletedAndWithdrawnFromTimeQuery;
            log.debug("Getting DELETED and WITHDRAWN resources....");
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setDataset(resDS);
            q.setIncludeInferred(false);
            q.setBinding("from", vf.createLiteral(from));
            out.startQueryResult(detail == DetailArg.identifier ?
                             columnNames.subList(0,1) : columnNames);
            SPARQL.evaluateTupleQuery(qs, q, new DeletedHandler(out, detail));

            // Now get the recently-modified resource instances.
            qs = (detail == DetailArg.identifier) ? identifierFromTimeQuery :
                             (inferred ? fullFromTimeWithInferredQuery : fullFromTimeNoInferredQuery);
            q = rc.prepareTupleQuery(QueryLanguage.SPARQL, qs);
            q.setDataset(resDS);
            q.setIncludeInferred(true);

            // when detail=full, add embedded instances.
            String qes = null;
            TupleQuery qe =  null;
            if (detail == DetailArg.full) {
                qes = inferred ? fullEIsFromTimeWithInferredQuery : fullEIsFromTimeNoInferredQuery;
                qe = rc.prepareTupleQuery(QueryLanguage.SPARQL, qes);
                qe.setDataset(resDS);
                qe.setIncludeInferred(true);
            }
            Literal lfrom = vf.createLiteral(from);
            for (URI graph : resGraphs) {
                if (log.isDebugEnabled())
                    log.debug("Getting non-deleted Resource Instances modified since="+from+", from graph="+graph);
                q.clearBindings();
                q.setBinding("from", lfrom);
                q.setBinding("graph", graph);
                SPARQL.evaluateTupleQuery(qs, q, new StatementsOnlyWrapper(out));
                if (qe != null) {
                    qe.clearBindings();
                    qe.setBinding("from", lfrom);
                    qe.setBinding("graph", graph);
                    SPARQL.evaluateTupleQuery(qes, qe, new StatementsOnlyWrapper(out));
                }
            }
            out.endQueryResult();
        }
    }

    // transform query results for query that returns DELETED subjects
    private static final class DeletedHandler extends StatementsOnlyWrapper
    {
        private DetailArg detail = null;
        private MapBindingSet bs = null;

        private DeletedHandler(TupleQueryResultHandler d, DetailArg dt)
        {
            super(d);
            detail = dt;
        }

        /** {@inheritDoc} */
        @Override
        public void startQueryResult(List<String> bn)
            throws TupleQueryResultHandlerException
        {
            List<String> cols = detail == DetailArg.identifier ?
                                 columnNames.subList(0,1) : columnNames;
            bs = new MapBindingSet(cols.size());
        }

        /** {@inheritDoc} */
        @Override
        public void handleSolution(BindingSet nbs)
            throws TupleQueryResultHandlerException
        {
            Value sub = nbs.getValue("subject");
            if (!(sub instanceof BNode)) {
                bs.clear();
                log.debug("DeletedHandler.handleSolution: Got result subject="+sub);
                if (detail == DetailArg.identifier)
                    bs.addBinding(column[0], new URIImpl(DELETED_PREFIX+sub.stringValue()));
                else {
                    bs.addBinding(column[0], sub);
                    bs.addBinding(column[1], REPO.IS_DELETED);
                    bs.addBinding(column[2], BooleanLiteralImpl.TRUE);
                }
                defer.handleSolution(bs);
            }
        }
    }

    // wrap a tuple handler and skip its start/stop methods
    private static class StatementsOnlyWrapper extends TupleQueryResultHandlerBase
    {
        protected TupleQueryResultHandler defer = null;

        protected StatementsOnlyWrapper(TupleQueryResultHandler d)
        {
            super();
            defer = d;
        }

        /** {@inheritDoc} */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            defer.handleSolution(bs);
        }
    }
}
