package org.eaglei.repository.servlet;

import org.eaglei.repository.util.WithRepositoryConnection;
import java.io.IOException;
import java.io.FilterInputStream;
import java.io.InputStream;
import java.io.File;
import java.io.FileWriter;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.jar.JarInputStream;
import java.util.jar.JarEntry;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.URI;
import org.openrdf.model.ValueFactory;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFParseException;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFParser;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.model.AccessGrant;
import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.NamedGraph;
import org.eaglei.repository.model.NamedGraphType;
import org.eaglei.repository.model.Provenance;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.util.AppendingRDFHandler;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Report on and load the data model ontology from various sources.
 *
 * @author Larry Stone
 * Started June, 2010
 * @version $Id: $
 */
public final class Model extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Model.class);

    // keys for "source" arg - only jar matters
    private static final String SOURCE_JAR = "jar";
    private static final String SOURCE_MODEL = "model";

    /** {@inheritDoc} */
    @Override
    public void init() {
    }

    // Returns path to a OWL file in the jar which is also the source of
    // the key owl:versionInfo statement; since we have to know that anyway
    // might as well leverage it to get the jar loaded.
    private String getJarTopLevelPath()
    {
       return DataModel.VERSION_INFO_SOURCE.getString();
    }
    
    // version of data model ontology currently loaded in repo
    private String getLoadedVersion(RepositoryConnection rc)
        throws RepositoryException
    {
        return Utils.getVersionInfo(rc, DataModel.VERSION_INFO_URI.getURI(),
                                        DataModel.GRAPH_URI.getURI());
    }

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");

       /***XXX
        //XXX built-in smoke test for version compare
        tryVersionNewer("0", "1", false);
        tryVersionNewer("1", "0", true);
        tryVersionNewer("1.1", "1", true);
        tryVersionNewer("0.1", "0.0.1", true);
        tryVersionNewer("0.1", "0.0.1 extra crap here", true);
        tryVersionNewer("0.1 extra crap here too", "0.0.1 extra crap here", true);
        tryVersionNewer("0.7", "0.6.2", true);
        tryVersionNewer("0.7", "0.7.2", false);
        tryVersionNewer("0.a.7", "0.7.2", false);
        tryVersionNewer("", "0.7.2", false);
        tryVersionNewer("", "", false);
        XXX***/

        // sanity check
        if (isParameterPresent(request, "action"))
            throw new BadRequestException("The 'action' argument is not allowed in the GET method, use POST.");

        String format = getParameter(request, "format", false);
        log.debug("/model GET, format="+format);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // get the current version of datamodel ontology
            String loadedVersion = getLoadedVersion(rc);
            if (loadedVersion == null)
                loadedVersion = "(not loaded or not in ontology)";

            // get version from jena model
            String availableModelVersion = getAvailableModelVersionInfo();
            if (availableModelVersion == null)
                availableModelVersion = "(failed or not in ontology)";

            // get version from Jar in War
            String availableJarVersion = getAvailableJarVersionInfo();
            if (availableJarVersion == null)
                availableJarVersion = "(failed or not in Jar or War)";

            ArrayList<BindingSet> results = new ArrayList<BindingSet>(1);
            MapBindingSet bs = new MapBindingSet(2);
            ValueFactory vf = rc.getValueFactory();
            bs.addBinding("loaded", vf.createLiteral(loadedVersion));
            bs.addBinding("availableJar", vf.createLiteral(availableJarVersion));
            bs.addBinding("availableModel", vf.createLiteral(availableModelVersion));
            results.add(bs);
            SPARQL.sendTupleQueryResults(request, response, format, results);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        boolean force = false;
        long modTime = -1;
        String action = getParameter(request, "action", true);
        String source = getParameter(request, "source", true);
        log.debug("/model POST, action="+action+", source="+source);

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        URI context = DataModel.GRAPH_URI.getURI();

        if (action.equals("load"))
            force = true;
        else if (!action.equals("update"))
            throw new BadRequestException("Argument action must be one of: 'load', 'update'");

        // check access
        if (!(Access.hasPermission(request, context, Access.ADD) &&
              Access.hasPermission(request, context, Access.REMOVE))) {
            throw new ForbiddenException("User is not permitted to replace the data model ontology graph");
        }

        // sanity check on source, "model" is now very deprecated
        if (source.equals(SOURCE_MODEL)) {
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Loading or updating from the OntModel source is no longer implemented.");
        }

        try {
            boolean created = rc.size(context) == 0;
            String loadedVersion = getLoadedVersion(rc);
            String availableVersion = source.equals(SOURCE_JAR) ?
                                        getAvailableJarVersionInfo() :
                                        getAvailableModelVersionInfo();
            String sourceLoaded = null;
             
            log.debug("availableVersion="+availableVersion+", loadedVersion="+loadedVersion);
            if (availableVersion == null)
                throw new ServletException("Failed to get version of available data model ontology from chosen source, perhaps failed to load ontology.  Try force-loading instead.");
             
            if (force || loadedVersion == null || versionNewer(availableVersion, loadedVersion)) {
                if (log.isDebugEnabled()) {
                    if (force)
                        log.debug("Forced reload of data model ontology.");
                    else
                        log.debug("Updating data model ontology: loaded="+loadedVersion+", available="+availableVersion);
                }
                rc.clear(context);

                // load from OWL files in maven project jar:
                if (source.equals(SOURCE_JAR)) {
                    String toplevel = getJarTopLevelPath();
                    String jar = getDataModelJarSpec(toplevel);
                    modTime = loadDataModelFromJar(rc, context, toplevel, jar);
                    sourceLoaded = jar;

                } else
                    throw new BadRequestException("Argument source must be one of: 'jar'");

                // set named graph admin MD - force creation if necessary
                NamedGraph ng = NamedGraph.findOrCreate(request, context, true);
                ng.setType(request, NamedGraphType.ontology);
                ng.setLabel(request, "eagle-i Data Model Ontology");
                ng.update(request);

                // make it world-readable
                AccessGrant.addGrant(request, context, REPO.ROLE_ANONYMOUS, Access.READ.getURI());

                // set provenance like any other graph
                Date now = new Date();
                Provenance p = new Provenance(context);
                if (created)
                    p.setCreated(request, now);
                else
                    p.setModified(request, now);
                Date sourceModified = null;
                if (modTime != -1)
                    sourceModified = new Date(modTime);
                p.setSource(request, sourceLoaded, sourceModified);
                rc.commit();
                response.setStatus(HttpServletResponse.SC_CREATED);
            }
            else
                response.setStatus(HttpServletResponse.SC_OK);
        } catch (NumberFormatException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // get version of ontology in jar - i.e. maven version
    private String getAvailableJarVersionInfo()
        throws ServletException, IOException
    {
        String toplevel = getJarTopLevelPath();
        String jarURLspec = getDataModelJarSpec(toplevel);
        log.debug("Getting maven version from jar at: "+jarURLspec);
        return Utils.getMavenVersionFromJar(jarURLspec);
    }

    // get versionInfo from "available" data model onto, i.e. EIOntModel
    private String getAvailableModelVersionInfo()
    {
        // XXX very deprecated now
        return null;
    }

    // Returns the resource URL of the jar file containing a given "toplevel"
    // OWL file.  Assumes there is only one such file in the search path, or
    // else that you don't care which jar it comes from.
    private String getDataModelJarSpec(String toplevel)
        throws ServletException
    {
        URL ero = this.getClass().getClassLoader().getResource(toplevel);
        if (ero == null)
            throw new ServletException("Cannot load data model OWL, since the top-level file \""+toplevel+"\"  is not available as a resource.  Check your configuration.");

        // The rest of this kludge assumes we have a jar-scheme URL, which
        // has the form:   jar:<url>!/<entry>
        // e.g.  jar:file:/..../lib/eagle-i-model-owl-0.4.3.jar!/ero.owl
        // so we can extract the URL of the jar itself:
        String eros = ero.toString();
        if (eros.startsWith("jar:")) {
            Pattern ep = Pattern.compile("jar:([^!]+)!/"+Pattern.quote(toplevel));
            log.debug("Groveling jar URL with pattern=\""+ep.pattern()+"\"");
            Matcher em = ep.matcher(eros);
            if (em.matches()) {
                String jarURLspec = em.group(1);
                log.debug("Got jar URL = "+jarURLspec);
                return jarURLspec;
            } else {
                log.warn("Failed matching pattern=\""+ep.pattern()+"\" against owl resource URL="+eros);
                throw new ServletException("Failed to extract jar's URL from resource (see log for details): "+eros);
            }
        } else {
            log.warn("Cannot load data model OWL, URL for the toplevel file is NOT in Jar scheme: "+eros);
            throw new ServletException("Cannot load data model OWL, URL for the toplevel file is NOT in Jar scheme: "+eros);
        }
    }

    /**
     * Grovel the data model OWL files out of a given jar file.  Jar is
     * identified by a resource URL.
     * NOTE: since we have to commit() after each file, copy all statements
     * into a temporary file first, and then read *that* into the target
     * graph..
     * Returns long integer modification time of sentinal file if found,
     *  or -1 if not found.
     */
    private long loadDataModelFromJar(RepositoryConnection rc, URI context, String toplevel, String jarURLspec)
        throws ServletException, IOException
    {
        long result = -1;

        log.info("Loading data model from jar at: "+jarURLspec);
        JarInputStream jis = null;

        // prepare temp file to accumulate all ontology statements
        File tmpFile = File.createTempFile("datamodel", ".txt",
                         (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir"));
        AppendingRDFHandler ah = new AppendingRDFHandler(Rio.createWriter(RDFFormat.NTRIPLES, new FileWriter(tmpFile)));
        RDFParser op = Rio.createParser(RDFFormat.RDFXML);
        op.setRDFHandler(ah);
        try {
            URL jarURL = new URL(jarURLspec);
            jis = new JarInputStream(jarURL.openStream());
            JarEntry je;
            while ((je = jis.getNextJarEntry()) != null) {
                if (je.isDirectory())
                    continue;
                String fn = je.getName();
                if (fn.endsWith(".owl")) {
                    log.info("..Adding OWL statements from jar entry: "+fn);
                    try {
                        op.parse(new UnclosableInputStream(jis), "");
                    } catch (RDFParseException e) {
                        log.error("Skipping file="+fn+", failed parsing: "+e);
                     // XXX FIXME: this failure should appear in overall servlet status as well
                    }
                    if (fn.equals(toplevel) || fn.equals("/"+toplevel)) {
                        result = je.getTime();
                        log.debug("Found toplevel entry, setting mod time = "+result);
                    }
                } else
                    log.info("..Skipping non-OWL jar file = "+fn);
            }

            // Now read the accumulated statements:
            ah.reallyEndRDF();
            ah = null;
            rc.add(tmpFile, "", RDFFormat.NTRIPLES, context);
            log.info("Loaded merged data model ontology, size = "+String.valueOf(rc.size(context))+" statements.");
        } catch (MalformedURLException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        } finally {
            if (jis != null)
                jis.close();
            if (ah != null) {
                try {
                    ah.reallyEndRDF();
                } catch (RDFHandlerException ie) {
                    log.warn("Failed closing merged ontology file: "+ie);
                }
            }
            tmpFile.delete();
        }
        return result;
    }

    // return true if v1 is "newer" than v2, for version strings
    // consisting of dotted integers.  e.g.  "1.2" is newer than "1",
    //  1.0.2 newer than 1.0, etc.
    // NOTE: IGNORE any trailing data after a space.
    boolean versionNewer(String av1, String av2)
        throws NumberFormatException, ServletException
    {
        String v01[] = av1.trim().split("\\s");
        String v02[] = av2.trim().split("\\s");
        if (!v01[0].matches("^[0-9\\.]+$")) {
            throw new ServletException("Cannot compare badly formatted version number: \""+v01[0]+"\"");
        }
        if (!v02[0].matches("^[0-9\\.]+$")) {
            throw new ServletException("Cannot compare badly formatted version number: \""+v02[0]+"\"");
        }
        String v1[] = v01[0].split("\\.");
        String v2[] = v02[0].split("\\.");
        log.debug("versionNewer: Raw versions = "+Arrays.deepToString(v1)+", "+Arrays.deepToString(v2));
        int len = v1.length > v2.length ? v1.length : v2.length;
        for (int i = 0; i < len; ++i) {
            int c1 =  (i < v1.length) ? Integer.parseInt(v1[i]) : 0;
            int c2 =  (i < v2.length) ? Integer.parseInt(v2[i]) : 0;
            if (c1 > c2) {
                log.debug("versionNewer: "+v01[0]+" is newer than "+v02[0]);
                return true;
            } else if (c2 > c1) {
                log.debug("versionNewer: "+v01[0]+" is older than "+v02[0]);
                return false;
            }
        }
        log.debug("versionNewer: "+v01[0]+" is equal to "+v02[0]);
        return false;
    }

    // XXX uncomment when enabling the builtin testing
  /***XXX
    private void tryVersionNewer(String v1, String v2, boolean expected)
    {
        try {
            boolean result = versionNewer(v1, v2);
            log.debug("Trying versionNewer(\""+v1+"\", \""+v2+"\") => "+(result == expected ? "success" : "FAIL")+", expected="+expected);
        }catch (Exception e) {
            log.error("Failed with exception: ",e);
        }
    }
         ***/

    // InputStream wrapper that ignores close() so RDF reader can
    // read from jar entry without closing the whole jar input stream
    private final class UnclosableInputStream extends FilterInputStream {
        private UnclosableInputStream(InputStream in)
        {
            super(in);
        }

        @Override
        public void close()
        {
        }
    }
}
