package org.eaglei.repository.servlet;

import java.util.ArrayList;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.BooleanLiteralImpl;

import org.eaglei.repository.model.User;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ConflictException;

/**
 * Get status of current authenticated user as tabular result.
 * Returns an empty result when there is no authenticated user.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class WhoAmI extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WhoAmI.class);

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");

        // sanity check - create arg only allowed on POST
        if (isParameterPresent(request,"create"))
            throw new BadRequestException("'create' arg is only allowed with POST method");

        whoami(request, response, getParameter(request, "format", false));
    }

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        boolean create = getParameterAsBoolean(request, "create", false, false);
        String firstName = getParameter(request, "firstname", false);
        String lastName = getParameter(request, "lastname", false);
        String mbox = getParameter(request, "mbox", false);
        String format = getParameter(request, "format", false);

        // make a new User if current authenticated user is "undocumented"
        if (create)
        {
            String login = Authentication.getAuthenticatedUsername(request);
            User u = Authentication.getPrincipalUser(request);
            if (login == null) {
                throw new BadRequestException("No authentication found.");

            // this should not happen
            } else if (u == null) {
                throw new ServletException("Cannot create User for login="+login);

            // no metadata, so create new User
            } else if (u.getURI() == null) {
                u = User.createAsAdministrator(request, login, null);
                if (firstName != null)
                    u.setFirstName(request, firstName);
                if (lastName != null)
                    u.setLastName(request, lastName);
                if (mbox != null)
                    u.setMbox(request, mbox);
                u.commit(request);
                response.setStatus(HttpServletResponse.SC_CREATED);

            // when there is already user metadata, flag a conflict
            } else {
                throw new ConflictException("There is already User metadata for login name = "+login);
            }
        } else {
            whoami(request, response, format);
        }
    }

    // return "whoami" results in response document
    private void whoami(HttpServletRequest request, HttpServletResponse response, String format)
        throws ServletException, java.io.IOException
    {
        ArrayList<BindingSet> result = new ArrayList<BindingSet>();
        MapBindingSet nbs = new MapBindingSet(6);
        User u = Authentication.getPrincipalUser(request);
        log.debug("authenticated User = "+u);
        if (u == null) {
            nbs.addBinding("uri", Authentication.getPrincipalURI(request));
            nbs.addBinding("username", null);
            nbs.addBinding("firstname", null);
            nbs.addBinding("lastname", null);
            nbs.addBinding("mbox", null);
        } else {
            nbs.addBinding("uri", u.getURI());
            nbs.addBinding("username", new LiteralImpl(u.getUsername()));
            nbs.addBinding("firstname", u.getFirstName() == null ? null : new LiteralImpl(u.getFirstName()));
            nbs.addBinding("lastname", u.getLastName() == null ? null : new LiteralImpl(u.getLastName()));
            nbs.addBinding("mbox", u.getMbox() == null ? null : new LiteralImpl(u.getMbox()));
        }
        nbs.addBinding("isSuperuser", Authentication.isSuperuser(request) ?
                       BooleanLiteralImpl.TRUE : BooleanLiteralImpl.FALSE);
        result.add(nbs);
        SPARQL.sendTupleQueryResults(request, response, format, result);
    }
}
