package org.eaglei.repository.servlet.admin;


import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.model.impl.URIImpl;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.model.Access;
import org.eaglei.repository.model.Role;
import org.eaglei.repository.model.User;
import org.eaglei.repository.servlet.RepositoryServlet;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;

/**
 * User Account creation and update.  This servlet exists primarily
 * to serve the user list and edit pages in the admin JSP UI.  Of course
 * it can be used by itself, but beware of weird behavior when its
 * referrer is a .JSP URL - it will indicate success by redirecting back
 * to the JSP.
 *
 * Upon fatal errors it does NOT return to the JSP, it just sends an
 * error status response.  This isn't very user friendly, but neither is
 * the rest of the admin UI.
 *
 * Method: POST only.
 *    Invocation Modes:
 *      - create a new user
 *      - disable a current user - disable login credentials
 *      - reinstate - create new login credentials
 *      - update - just change user metadata
 *
 *    REQUIRED ARGS:
 *     username  - principal name of user to change, defaults to authenticated
 *    OPTIONAL
 *     only_password - if specified, do not change other user metadata
 *     password - new password
 *     password_confirm - must match new password
 *     old_password - current (old) password, required for non-admin user
 *     first - first name, can be ""
 *     last - last name, can be ""
 *     mailbox - mailbox, can be ""
 *     role - multiple values, URI of roles to apply to user
 *
 * @author Larry Stone
 * Started March 2011
 */
public class UpdateUser extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(UpdateUser.class);

    // allowable values for Action arg
    public enum Action { create, update, disable, reinstate };

    // args to pass through to JSP
    private static final String PASS_THRU_ARGS[] =
        { "only_password", "standalone", "username" };

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String username = getParameter(request, "username", true);
        Action action = (Action)getParameterAsKeyword(request, "action", Action.class, null, true);
        boolean onlyPassword = isParameterPresent(request, "only_password");
        String password = getParameter(request, "password", false);
        String passwordConfirm = getParameter(request, "password_confirm", false);
        String oldPassword = getParameter(request, "old_password", false);
        String first = getExactParameter(request, "first", false);
        String last = getExactParameter(request, "last", false);
        String mailbox = getExactParameter(request, "mailbox", false);
        String roleURI[] = getParameters(request, "role", false);

        // general arg sanity checks:
        //  1. the username exists OR we are creating a user
        User user = (username == null) ? null : User.findByUsername(request, username);
        if (action == Action.create) {
            if (user != null) {
                throw new BadRequestException("Cannot create the user '"+username+"', it already exists.");
            } else if (username == null) {
                throw new BadRequestException("Username is required to create a new user.");
            }
        } else if (user == null) {
            throw new BadRequestException("Cannot update user '"+username+"', not found.");
        }

        // 2. password needed? if password_confirm specified, it must match
        if ((action == Action.create || action == Action.reinstate) && password == null) {
            throw new BadRequestException("You must enter a Password when creating or reinstating a user.");
        }
        if (password != null || passwordConfirm != null) {
            if (!(password != null && passwordConfirm != null &&
                  password.equals(passwordConfirm)))
                throw new BadRequestException("Password values do not match.");
        }

        // Access checks: must be admin, unless updating your own user
        // (and non-admin must not update roles, of course...)
        if (!Authentication.isSuperuser(request)) {
            if (action == Action.update) {
                if (!Access.hasPermissionOnUser(request, username))
                    throw new ForbiddenException("Not allowed to modify user: "+username);
            } else {
                throw new ForbiddenException("Only an Administrator may create, disable, or reinstate a user.");
            }
        }

        if (action == Action.create) {
            user = User.create(request, username, password);
            mungUser(request, user, first, last, mailbox);
            mungRoles(request, user, roleURI);
            user.commit(request);
         
        } else if (action == Action.reinstate) {
            user.reinstate(password);
            user.commit(request);

        // update: change metadata and/or password
        } else if (action == Action.update) {
            if (password != null) {
                // non-superuser must supply old password
                if (!Authentication.isSuperuser(request)) {
                    if (oldPassword == null || !user.authenticate(oldPassword))
                        throw new ForbiddenException("Old password does not match, you are not allowed to update your password without it.");
                }
                user.setPassword(request, password);
            }
         
            if (!onlyPassword) {
                mungUser(request, user, first, last, mailbox);
         
                // update roles - let User object figure out diffs.
                if (Authentication.isSuperuser(request)) {
                    mungRoles(request, user, roleURI);
                }
            }
            user.commit(request);
         
        // disable: just delete authentication DB entry, leave metadata
        } else if (action == Action.disable) {
            user.disable();
            user.commit(request);
        }

        // XXX may need to expand this to redirect to other pages

        redirectToJSP(request, response,
                      action.toString()+"d user "+username,
                      PASS_THRU_ARGS);
    }

    // mung user roles, return true if anything chagned
    private boolean mungRoles(HttpServletRequest request, User user, String roleURI[])
        throws ServletException
    {
        Role role[] = new Role[roleURI == null ? 0 : roleURI.length];
        if (roleURI != null) {
            for (int i = 0; i < roleURI.length; ++i)
                role[i] = Role.find(request, new URIImpl(roleURI[i]));
        }
        return user.setRoles(request, role);
    }

    // mung user metadata if specified new values are different
    private boolean mungUser(HttpServletRequest request, User user, String first, String last, String mailbox)
        throws ServletException
    {
        log.debug("Updating user, change to: first='"+first+"', last='"+last+"'");
        boolean changed = false;
        if (wouldChange(first, user.getFirstName())) {
            user.setFirstName(request, first);
            log.debug("Changed first name = "+first);
            changed = true;
        }
        if (wouldChange(last, user.getLastName())) {
            user.setLastName(request, last);
            log.debug("Changed last name = "+last);
            changed = true;
        }
        if (wouldChange(mailbox, user.getMbox())) {
            user.setMbox(request, mailbox);
            log.debug("Changed mbox = "+mailbox);
            changed = true;
        }
        return changed;
    }

    // would arg be a different value for the existing value?
    private static boolean wouldChange(String arg, String existing)
    {
        return arg != null &&
               (existing != null && !existing.equals(arg)) ||
               (existing == null && arg.length() > 0);
    }
}
