package org.eaglei.search.harvest;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIURI;

/**
 * Holds information about change of a single resource. NOT thread safe.
 */
public class ResourceChangeEvent implements Comparable<ResourceChangeEvent>, Serializable {

    public static final long serialVersionUID = 1L;
    
    /**
     * Identifies the type of change.
     * Currently EIOntConstants.IS_DELETED for delete event.
     * Null for any other change event.
     */
    private String changeId;

    /*
     * The associated eagle-i resource instance.
     */
    private EIEntity entity;

    /*
     * Type of the resource. Corresponds to a class in the eagle-i ontology. 
     */
    private EIEntity type;

    /*
     * Resource provider.  May be null.
     */
    private EIURI provider;
    
    /*
     * Institution at which the resource is located.  May be null.
     */
    private EIEntity institution;

    /*
     * Data type properties as defined by the eagle-i ontology.
     */
    private Map<EIDatatypeProperty, Set<String>> datatypeProperties;
    
    /*
     * Object properties as defined by the eagle-i ontology.
     */
    private Map<EIObjectProperty, Set<EIURI>> objectProperties;

    
    private ResourceChangeEvent() {}

    /**
     * Creates a new change event.
     */
    public ResourceChangeEvent(final String changeId, final EIEntity entity) { 
        this.changeId = changeId;
        setEntity(entity);
    }
    
    /**
     * Identifies the type of change.
     * Currently EIOntConstants.IS_DELETED for delete event.
     * Null for any other change event.
     */
    public String getChangeId() {
        return this.changeId;
    }
    
    public boolean isDelete() {
    	return (getChangeId() != null && getChangeId().equals(EIOntConstants.IS_DELETED));
    }

    /**
     * Retrieves the EIEntity representing the resource instance.
     * 
     * @return The EIEntity.
     */
    public EIEntity getEntity() {
        return this.entity;
    }

    /**
     * Sets the EIEntity representing the resource instance
     * @param entity The EIEntity. Cannot be null.
     */
    public void setEntity(final EIEntity entity) {
        assert entity != null;
        this.entity = entity;
    }

    /**
     * Retrieves the EIEntity representing the resource type.
     * 
     * @return The type entity. Corresponds to a class in the eagle-i ontology. 
     */
    public EIEntity getType() {
        return this.type;
    }
    
    /**
     * Sets the EIEntity representing the resource type.
     * 
     * @param type The type entity. Corresponds to a class in the eagle-i ontology. 
     */
    public void setType(final EIEntity type) {
        this.type = type;
    }

    /**
     * Retrieves the resource provider URI.
     * 
     * @return The resource provider URI. May be null if this resource is not associated with a specific provider.
     */
    public EIURI getProvider() {
        return this.provider;
    }
    
    /**
     * Sets the resource provider URI.
     * 
     * @param provider EIURI representing the resource provider. May be null.
     */
    public void setProvider(final EIURI provider) {
        this.provider = provider;
    }

    /**
     * Retrieves the EIEntity representing the resource institution.
     * 
     * @return The institution entity.
     */
    public EIEntity getInstitution() {
        return this.institution;
    }
    
    /**
     * Sets the EIEntity representing the resource institution.
     *
     * @param institution The institution entity. 
     */
    public void setInstitution(EIEntity institution) {
        this.institution = institution;
    }

    /**
     * Returns the URIs (as EIURI instances) for the result data type properties.
     * 
     * @return Set of EIURIs representing a eagle-i ontology data type properties for the
     *         result.
     */
    @SuppressWarnings("unchecked")
    public Set<EIDatatypeProperty> getDataTypeProperties() {
        if (this.datatypeProperties == null) {
            // GWT compiler doesn't like emptySet()
            // return Collections.emptySet();
            return Collections.EMPTY_SET;
        }
        return this.datatypeProperties.keySet();
    }

    /**
     * Adds a property binding for a data type property.
     * 
     * @param property Property URI as an EIURI. Cannot be null.
     * @param value Property value.
     */
    public void addDataTypeProperty(EIDatatypeProperty property, String value) {
        assert property != null;
        assert value != null;
        // Not threadsafe
        if (this.datatypeProperties == null) {
            this.datatypeProperties = new HashMap<EIDatatypeProperty, Set<String>>();
        }
        Set<String> values = this.datatypeProperties.get(property);
        if (values == null) {
            values = new HashSet<String>();
            this.datatypeProperties.put(property, values);
        }
        values.add(value);
    }

    /**
     * Retrieves the values of the specified data type property.
     * 
     * @param property Property URI as an EIURI.
     * @return Values of the property if it has been set or null.
     */
    public Set<String> getDataTypeProperty(EIDatatypeProperty property) {
        assert property != null;
        if (this.datatypeProperties == null) {
            return null;
        }
        return this.datatypeProperties.get(property);
    }
    
    /**
     * Returns the URIs (as EIURI instances) for the result object properties.
     * 
     * @return Set of EIURIs representing a eagle-i ontology properties for the
     *         result.
     */
    @SuppressWarnings("unchecked")
    public Set<EIObjectProperty> getObjectProperties() {
        if (this.objectProperties == null) {
            // GWT compiler doesn't like emptySet()
            // return Collections.emptySet();
            return Collections.EMPTY_SET;
        }
        return this.objectProperties.keySet();
    }

    /**
     * Adds a property binding for an object property.
     * 
     * @param property Property URI as an EIURI. Cannot be null.
     * @param value Property value.
     */
    public void addObjectProperty(EIObjectProperty property, EIURI value) {
        assert property != null;
        assert value != null;
        // Not threadsafe
        if (this.objectProperties == null) {
            this.objectProperties = new HashMap<EIObjectProperty, Set<EIURI>>();
        }
        Set<EIURI> values = this.objectProperties.get(property);
        if (values == null) {
            values = new HashSet<EIURI>();
            this.objectProperties.put(property, values);
        }
        values.add(value);
    }

    /**
     * Retrieves the values of the specified object property.
     * 
     * @param property Property URI as an EIURI.
     * @return Values of the property if it has been set or null.
     */
    public Set<EIURI> getObjectProperty(EIObjectProperty property) {
        assert property != null;
        if (this.objectProperties == null) {
            return null;
        }
        return this.objectProperties.get(property);
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof ResourceChangeEvent)) {
            return false;
        }
        final ResourceChangeEvent other = (ResourceChangeEvent) o;
        if (!other.getEntity().equals(getEntity())) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        return getEntity().hashCode();
    }
    
    public String toString() {
        return getEntity().toString();
    }
    
    @Override
    public int compareTo(ResourceChangeEvent o) {
        // Ordering based on rank, must be consistent w/ equals
        if (this.equals(o)) {
            return 0;
        } else {
            int result = this.getEntity().getLabel().compareTo(o.getEntity().getLabel());
            if (result != 0) {
                return result;
            }

            // same same label, order by URI string
            return this.getEntity().getURI().toString().compareTo(o.getEntity().getURI().toString());
        }
    }
    
    //NB: For serialization tests only.  We need a way to test all the fields in one SearchResult against those from another,
    //as equals() only compares on the entity field. -Clint
    public boolean deepEquals(final Object obj) {
    	//TODO: No idea if this is appropriate
    	final double delta = 0.01; 
    	
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		ResourceChangeEvent other = (ResourceChangeEvent) obj;
		if (entity == null) {
			if (other.entity != null) {
				return false;
			}
		} else if (!entity.equals(other.entity)) {
			return false;
		}
		if (institution == null) {
			if (other.institution != null) {
				return false;
			}
		} else if (!institution.equals(other.institution)) {
			return false;
		}
		if (provider == null) {
			if (other.provider != null) {
				return false;
			}
		} else if (!provider.equals(other.provider)) {
			return false;
		}
		if (datatypeProperties == null) {
			if (other.datatypeProperties != null) {
				return false;
			}
		} else if (!datatypeProperties.equals(other.datatypeProperties)) {
			return false;
		}
		if (objectProperties == null) {
            if (other.objectProperties != null) {
                return false;
            }
        } else if (!objectProperties.equals(other.objectProperties)) {
            return false;
        }		
		if (type == null) {
			if (other.type != null) {
				return false;
			}
		} else if (!type.equals(other.type)) {
			return false;
		}
		return true;
	}
}
