package org.eaglei.search.common;

import java.lang.reflect.Type;

import org.eaglei.model.EIURI;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;

/**
 * 
 * @author Clint Gilbert
 * 
 * Feb 11, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 * Initialize-on-demand idiom (thread-safe lazy-loading)
 */
public final class GsonHolder {
    
    private GsonHolder() {
        //holder only
    }
    
    //NB: Custom serialization config for EIURIs to fix 
    //http://jira.eagle-i.net:8080/browse/SEARCH-153
    public static final Gson Gson = new GsonBuilder().registerTypeAdapter(EIURI.class, EIURIAdapter.Instance).create();
    
    /**
     * 
     * @author Clint Gilbert
     * 
     * May 20, 2010
     *
     * Center for Biomedical Informatics (CBMI)
     * @link https://cbmi.med.harvard.edu/
     *
     * Custom JSON serialization behavior for EIURIs.  
     * Allows Serializing SearchResults and SearchResultSets without too many gymnastics.
     */
    private static final class EIURIAdapter implements JsonSerializer<EIURI>, JsonDeserializer<EIURI> {
        
        public static final EIURIAdapter Instance = new EIURIAdapter();
        private static final String NullInstanceString = new JsonNull().toString();

        private EIURIAdapter() {
            super();
        }
        
        @Override
        public JsonElement serialize(final EIURI uri, final Type type, final JsonSerializationContext context) {
            if (uri == null) {
                return new JsonNull();
            }
            return new JsonPrimitive(uri.toString());
        }

        @Override
        public EIURI deserialize(final JsonElement element, final Type type, final JsonDeserializationContext context) throws JsonParseException {
            String strValue = element.getAsString();
            if (NullInstanceString.equals(strValue)) {
                return null;
            } else {
                return EIURI.create(strValue); 
            }
        }
    }
}
