package org.eaglei.search.provider;

import java.io.IOException;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;

/**
 * SearchProvider augments the results of a search with properties based on searches against other SearchProviders.
 * The query used for these augmentation searches is based on the result entity.
 *
 * @author rfrost
 */
public class SearchResultAugmenter implements SearchProvider {

    private static final Log logger = LogFactory.getLog(SearchResultAugmenter.class);
    private static final boolean DEBUG = logger.isDebugEnabled();

    private SearchProvider nestedProvider;
    private Map<EIEntity, SearchProvider> resultAugmenters;
        
    /**
     * Creates a new SearchProvider that augments each result with additional properties based on a query against
     * another SearchProvider.
     * @param nestedProvider The provider that is queried to retrieve the primary search result set.
     * @param resultAugmenter The provider that is queried to augment each result.
     */
    public SearchResultAugmenter(final SearchProvider nestedProvider, final Map<EIEntity, SearchProvider> resultAugmenters) { 
        this.nestedProvider = nestedProvider;
        this.resultAugmenters = resultAugmenters;
    }
    
    @Override
    public SearchResultSet query(final SearchRequest request) throws IOException {
        final SearchResultSet nestedResults = this.nestedProvider.query(request);
        
        // for each search result (in the page), query the external provider
        for (SearchResult result: nestedResults.getResults()) {
            // create an entity request
            final EIEntity entity = result.getEntity();
            final SearchRequest.Term term = new SearchRequest.Term(entity.getLabel(), entity.getURI());
            final SearchRequest entityRequest = new SearchRequest(term);
    
            for (EIEntity type: resultAugmenters.keySet()) {
                SearchProvider augmenter = resultAugmenters.get(type);
                SearchResultSet augmentingResults = augmenter.query(entityRequest);
                EIURI prop = EIURI.create(type.getLabel());
                for (SearchResult augmentingResult: augmentingResults.getResults()) {
                    result.addDataTypeProperty(prop,augmentingResult.getEntity().getLabel());
                }
            }
        }
        
        return nestedResults;
    }
    
    @Override
    public SearchCounts count(SearchCountRequest request) throws IOException {
        return this.nestedProvider.count(request);
    }    
    
    @Override
    public void init() throws IOException {
        this.nestedProvider.init();
        for (SearchProvider augmenter: resultAugmenters.values()) {
            augmenter.init();
        }
    }    
}
