package org.eaglei.search.provider.rdf;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.query.ResultSetFormatter;

/**
 * SearchProvider implementation that uses the Repository /harvest API.
 * 
 *  @author rfrost
 */
public class RepositoryHarvester extends AbstractRDFProvider {

    private static final Log logger = LogFactory.getLog(RepositoryHarvester.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    // "from" time; lower bound for harvest changes. This is initially null to ensure we get a
    // full dump from the repository. It is then set to the time right before then request until
    // get a change at which point it is updated
    // TODO this currently relies on clock sync between the client and repo server (which will
    // be a non-issue when the client and repo run on the same machine); need to replace the 
    // logic that sets this with code that gets the timestamp from the server
    private Date fromTime = null;
    
    /**
     * Creates a new RepositoryHarvester
     */
    public RepositoryHarvester(final EIOntModel eagleiOntModel, final EIEntity institution,
            final RepositoryConfig repoConfig) {
        super(eagleiOntModel, institution, repoConfig);
    }

    public void init() throws IOException {
        // no-op
    }

    public SearchResultSet query(final SearchRequest request) throws IOException {
        final List<QuerySolution> resultList = harvest();
        final List<SearchResult> results = getSearchResultsFromSPARQLResults(resultList, request);
        request.setMaxResults(results.size());
        // create a SearchResultSet with the correct paginated range 
        final SearchResultSet resultSet = createSearchResultSet(results, request);
        if (DEBUG) {
            if (resultSet.getTotalCount() > 0) {
                logger.debug("Harvest query found " + resultSet.getTotalCount() + " results");
            }
        }
        return resultSet;        
    }
    
    protected QueryExecution getQueryExecution(Query query) {
        return null;
    }
    /**
     * Executes the repository /harvest API and returns the results as a SPARQL ResultSet.
     * @return ResultSet response.
     * @throws IOException Thrown if there is an error.
     */    
    private List<QuerySolution> harvest() throws IOException {
        final HttpClient httpclient = new HttpClient();
        final UsernamePasswordCredentials credentials = new UsernamePasswordCredentials();
        credentials.setUserName(this.repoConfig.getUsername());
        credentials.setPassword(this.repoConfig.getPassword());
        httpclient.getState().setCredentials(AuthScope.ANY, credentials);
        httpclient.getParams().setAuthenticationPreemptive(true);
        final PostMethod method = new PostMethod(this.repoConfig.getUrl());
        //HttpMethodParams params = new HttpMethodParams();
        //method.setParameter("workspace", this.repoConfig.getDefaultGraphURI());
        method.setParameter("view", this.repoConfig.getView());
        method.setParameter("detail", "full");
        if (fromTime != null) {
            final DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
            format.setTimeZone(TimeZone.getTimeZone("GMT"));
            final String dateStr = format.format(fromTime);
            //logger.debug("Date format: " + dateStr);
            method.setParameter("from", dateStr + "Z"); // GMT time 
        } else {
            fromTime = new Date();
        }
        // record the time before we make the call
        final Date timeBeforeCall = new Date();
        if (DEBUG) {
            // Don't include this by default or we'll fill up the logs
            //logger.debug("Harvesting from " + this.repoConfig.getUrl() + " as user " + this.repoConfig.getUsername());
        }
        InputStream is = null;
        try {
            httpclient.executeMethod(method);
            //logger.debug(method.getResponseBodyAsString());
            is = method.getResponseBodyAsStream();
            final ResultSet results = ResultSetFactory.fromXML(is);
            final List<QuerySolution> resultList = ResultSetFormatter.toList(results);
            // if there were results, update the fromTime
            if (!resultList.isEmpty()) {
                this.fromTime = timeBeforeCall;
            }
            //logger.debug("Found " + resultList.size() + " query solutions");
            return resultList;
        } finally {
            method.releaseConnection();
            if (is != null) {
                is.close();
            }
        }        
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#count(org.eaglei.search.provider.SearchCountRequest)
     */
    public SearchCounts count(SearchCountRequest request) throws IOException {
        // TODO return empty counts for now
        return new SearchCounts(request.getRequest());     
    }    
    
}