package org.eaglei.search.provider.rdf;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.EagleIOntConstants;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.search.provider.SearchProviderUtil;
import org.eaglei.services.repository.RepositoryHttpConfig;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFormatter;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.vocabulary.RDF;

/**
 * SearchProvider implementation for a repository containing RDF data.
 * Contains logic to:
 * <ul>
 * <li>Map SearchRequests into SPARQL queries
 * <li>Map SPARQL ResultSets into SearchResults (all QuerySolutions for the same subject are used to build a single
 *     search result).
 * </ul>
 * @author frost
 */
public abstract class AbstractRDFProvider implements SearchProvider {

    private static final Log logger = LogFactory.getLog(AbstractRDFProvider.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    // SPARQL variables 
    
    private static final String SUBJECT= "subject";
    private static final String PREDICATE= "predicate";            
    private static final String OBJECT = "object";

    /**
     * Reference to in-memory copy of the eagle-i ontology model
     */
    protected final EIOntModel eagleiOntModel;
    /**
     * Cache of preferred name property URIs
     */
    protected List<EIURI> prefLabelProperties = new ArrayList<EIURI>();
    /**
     * Institution against which this provider is executing. Not null
     */
    protected final EIEntity institution;
    
    /**
     * Creates a new SearchProvider that executes SPARQL queries against an underlying RDF repository.
     */
    public AbstractRDFProvider(final EIOntModel eagleiOntModel, final EIEntity institution) {
        assert eagleiOntModel != null;
        this.eagleiOntModel = eagleiOntModel;
        this.institution = institution;
        cachePreferredLabelPropURIs();
    }
    
    /*
     * Retrieves and caches the URIs of preferred label properties for reuse during query execution.
     */
    private void cachePreferredLabelPropURIs() {
        List<Property> props = ((JenaEIOntModel) eagleiOntModel).getPrefLabelProperties();
        for (Property prop: props) {
            this.prefLabelProperties.add(EIURI.create(prop.getURI()));
        }
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#init()
     */
    public void init() throws IOException {
        // no-op
    }
        
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request.SearchRequest)
     */
    public SearchResultSet query(final SearchRequest request) throws IOException {
        assert request != null;

        // if there is an institution binding and it is not equal one of the institutions for this provider,
        // return an empty result set
        if (request.getInstitution() != null && this.institution != null) {
            if (!this.institution.getURI().equals(request.getInstitution())) {
                if (DEBUG) {
                    logger.debug("SearchRequest for different institution, returning empty SearchResultSet");
                }
                return new SearchResultSet(request);
            }
        }
        
        // get the SPARQL query
        final String sparql = createSPARQLString(request);

        if (DEBUG) {
            logger.debug("Executing query: " + sparql);
        }
        
        // turn the SearchRequest into an ARQ Query 
        final Query query = QueryFactory.create(sparql);

        // execute the query
        final QueryExecution qe = getQueryExecution(query);
        List<QuerySolution> resultList = null;
        try {
            final ResultSet rs = qe.execSelect();
            resultList = ResultSetFormatter.toList(rs);
        } finally {
            qe.close();
        }

        // turn each QuerySolution into a SearchResult
        final List<SearchResult> uriToResult = getSearchResultsFromSPARQLResults(resultList, request);
        
        // create a SearchResultSet with the correct paginated range 
        final SearchResultSet results = createSearchResultSet(uriToResult, request);
        if (DEBUG) {
            logger.debug("Query executed, found " + results.getTotalCount() + " results");
        }

        return results;
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#count(org.eaglei.search.provider.SearchCountRequest)
     */
    public SearchCounts count(SearchCountRequest request) throws IOException {
        // TODO return empty counts for now
        return new SearchCounts(request.getRequest());     
    }    
    
    /**
     * Create a SPARQL SELECT query against the repository as an ARQ Query.
     * @param request The SearchRequest
     * @return SPARQL Query String.
     */
    public static String createSPARQLString(final SearchRequest request) {
        final StringBuilder sparql = new StringBuilder();

        sparql.append("SELECT DISTINCT");
        sparql.append(" ?" + SUBJECT);
        sparql.append(" ?" + PREDICATE);        
        sparql.append(" ?" + OBJECT);
        
        sparql.append(" WHERE { "); 
        sparql.append(" ?" + SUBJECT);
        sparql.append(" ?" + PREDICATE);        
        sparql.append(" ?" + OBJECT+ ".");
        
        // if type was specified, add that constraint
        // TODO either support use of GRAPH to look for inferred type in a different 
        // graph or change this to directType from rdf:type
        final EIURI type = SearchProviderUtil.getType(request);        
        if (type != null) {
            sparql.append(" ?" + SUBJECT);
            sparql.append(" <" + RDF.type.getURI() + "> ");
            sparql.append(" <" + type.toString() + ">. ");
        }
        sparql.append("}");
        
        return sparql.toString();
    }
    
    /**
     * Gets an ARQ QueryExecution for the Query.
     * @param query Query
     * @return QueryExecution
     */
    protected abstract QueryExecution getQueryExecution(final Query query);
    
    /**
     * Converts the SPARQL ResultSet into SearchResults
     */
    protected List<SearchResult> getSearchResultsFromSPARQLResults(final List<QuerySolution> solns, final SearchRequest request) {

        // map from resource URI to SearchResult that preserves insertion order
        final Map<String, Model> uriToModel = new LinkedHashMap<String, Model>();

        // Iterate over query solutions and convert to SearchResults
        for (QuerySolution soln: solns) {
            final Resource resource = soln.getResource(SUBJECT);
            final String uri = resource.getURI();
            if (!uriToModel.containsKey(uri)) {
                uriToModel.put(uri, ModelFactory.createDefaultModel());
            }
            Model m = uriToModel.get(uri);
            final Resource property = soln.getResource(PREDICATE);
            final RDFNode value = soln.get(OBJECT);
            final Property predicate = m.createProperty(property.getURI());
            m.add(m.createStatement(resource, predicate, value));
        }
        
        final List<SearchResult> results = new ArrayList<SearchResult>(uriToModel.size());
        for (String uri: uriToModel.keySet()) {
            Model m = uriToModel.get(uri);
            final SearchResult result = createSearchResultFromModel(uri, m);
            if (result != null) {
                results.add(result);
            }
        }
        
        return results;
    }
    
    /**
     * Creates a SearchResult from an Model holding only RDF triples associated with that subject URI
     * @return SearchResult or null if it could not be created.
     */
    protected SearchResult createSearchResultFromModel(final String uri, final Model model) {
        
        final Resource resource = model.getResource(uri);

        // create the EIEntity, initially use URI local name for entity name
        final EIEntity resourceEntity = EIEntity.create(EIURI.create(resource.getURI()), resource.getLocalName());
            
        // gets the first asserted type in the model
        // TODO handle multiple asserted types
        final Statement type = resource.getProperty(RDF.type);
        String typeURI = null;
        if (type == null) {
            Property isDeleted = model.getProperty(EagleIOntConstants.IS_DELETED);
            if (isDeleted != null) {
                Statement s = resource.getProperty(isDeleted);
                if (s != null && s.getObject().isLiteral()) {
                    if (((Literal) s.getObject()).getBoolean()) {
                        // has the isDeleted property and the value is true
                        typeURI = EagleIOntConstants.IS_DELETED;
                        EIEntity typeEntity = EIEntity.create(EIURI.create(typeURI), typeURI);
                        return new SearchResult(resourceEntity, typeEntity, null, null);
                    }
                }
            }
        } else {
            typeURI = ((Resource)type.getObject()).getURI();
        }
        if (typeURI == null) {
            logger.error("Resource " + resource.getURI() + " is missing type");
            return null;
        }

        // ensure we have the type in the ontology
        final EIURI typeEIURI = EIURI.create(typeURI);
        // does this have a type from the ontology? 
        if (!eagleiOntModel.isModelClassURI(typeURI)) {
            //logger.error("Resource " + resource.getURI() + " has type " + typeURI + " that does not exist in eagle-i ontology");
            return null;
        }

        // Creates an EIEntity from a URI via look-up against the in-memory Jena ontology model.
        final String typeLabel = eagleiOntModel.getPreferredLabel(typeEIURI);
        final EIEntity typeEntity = EIEntity.create(typeURI, typeLabel);        
            
        // create the SearchResult
        final SearchResult result = new SearchResult(resourceEntity, typeEntity, null, this.institution);
            
        // update the SearchResult with all of the data (add as either data type or object props)
        final List<Statement> statements = model.listStatements(resource, null, (RDFNode) null).toList();
        for (Statement s: statements) {
            final String propertyURI = s.getPredicate().getURI();
            final EIURI eiURI = EIURI.create(propertyURI);
            final RDFNode object = s.getObject();
            if (object.isLiteral()) {
                // if the property object is a literal, add that value
                result.addDataTypeProperty(eiURI, ((Literal) object).getLexicalForm());
            } else if (object.isResource()) {
                // if the property is a resource, get the URI
                final Resource value = (Resource) object;
                // save the entity as a result property
                result.addObjectProperty(eiURI, EIURI.create(value.getURI()));
            }       
        }
        
        // set the label based on the actual label-associated properties
        setLabel(result);
        
        return result;
    }

    /*
     * Sets the label for the specified SearchResult based on the label-associated properties.
     */
    private void setLabel(final SearchResult result) {
        for (EIURI prop: prefLabelProperties) {
            final Set<String> values = result.getDataTypeProperty(prop);
            if (values != null) {
                final EIEntity newEntity = EIEntity.create(result.getEntity().getURI(), values.iterator().next());
                result.setEntity(newEntity);
                break;
            }
        }
    }
    
    /*
     * Creates a SearchResultSet with the right paginated range
     */
    protected SearchResultSet createSearchResultSet(final List<SearchResult> results, final SearchRequest request) {
        // grab the paginated range and update the SearchResultSet
        int i = 0;
        final int start = request.getStartIndex();
        final int max = request.getMaxResults();
        final SearchResultSet resultSet = new SearchResultSet(request);
        for (SearchResult result: results) {
            if (i >= start && i < (start+max)) {
                resultSet.getResults().add(result);
            } 
            i++;
        }
        resultSet.setStartIndex(start);
        resultSet.setTotalCount(i);
        return resultSet;
    }
}
