package org.eaglei.search.provider.rdf;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.datagen.AbstractGenerator;
import org.eaglei.search.datagen.DataGenParams;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.shared.Lock;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * Generator test data (as RDF) using the eagle-i ontology.
 */
public final class RDFGenerator extends AbstractGenerator {

    private static final Log logger = LogFactory.getLog(RDFGenerator.class);

    /**
     * URI for the university class
     */
    public static final String UNIVERSITY_CLASS_URI = EIOntConstants.EAGLE_I_URI + "ERO_0000065";

    private final Model model;
    private final Property directType;

    /**
     * Creates an RDFGenerator that adds all of the RDF to a Jena model that is
     * auto-generated.
     */
    public RDFGenerator(final EIOntModel eagleiOntModel) throws IOException {
    	this(eagleiOntModel, ModelFactory.createDefaultModel());
    }

    /**
     * Creates an RDFGenerator that adds all of the RDF to the specified model.
     * 
     * @param model The Jena model.
     */
    public RDFGenerator(final EIOntModel eagleiOntModel, final Model model) throws IOException {
        super(eagleiOntModel);
        this.model = model;
        this.directType = model.createProperty(EIOntConstants.DIRECT_TYPE_URI);
    }

    /**
     * @return The Jena model that holds the generated RDF.
     */
    public Model getModel() {
        return this.model;
    }

    @Override    
    public void generate(final DataGenParams params) throws IOException {
        this.model.enterCriticalSection(Lock.WRITE);
        try {
            super.generate(params);
        } finally {
            this.model.leaveCriticalSection();
        }
    }

    @Override
    public void closeStorage() throws IOException {
        // no-op
    }

    @Override
    public int generateForInstitution(final String uri, final String label,
            final DataGenParams genParams) throws IOException {

        // create a resource for this institution
        final Resource institution = model.createResource(uri);
        // add the university type to the institution
        Statement stmt = model.createStatement(institution, RDF.type,
                model.createResource(UNIVERSITY_CLASS_URI));
        model.add(stmt);
        // add the name label
        stmt = model.createStatement(institution, RDFS.label, label);
        model.add(stmt);

        return super.generateForInstitution(uri, label, genParams);
    }

    @Override
    public int generateLab(final String institutionURI, final String institutionLabel,
            final String labLabel, final String labURI, final String labURL, final String labTypeURI,
            final DataGenParams genParams) throws IOException {

        // retrieve the resource for the institution
        final Resource institution = model.getResource(institutionURI);
        // create a resource for the lab
        final Resource lab = model.createResource(labURI);

        final OntClass labTypeClass = ((JenaEIOntModel) eagleiOntModel).getOntModel().getOntClass(labTypeURI);
        // add the direct type
        model.add(model.createStatement(lab, directType, labTypeClass));
        // add all types (potentially with materialization)   
        final String uri = labTypeClass.getURI();
        final EIURI eiURI = EIURI.create(uri);
        for (EIClass type: getTypes(eagleiOntModel, eiURI, genParams.getMaterializeHierarchy())) {
            model.add(model.createStatement(lab, RDF.type, model.getResource(type.getEntity().getURI().toString())));
        }

        // add the label
        model.add(model.createStatement(lab, RDFS.label, labLabel));
        // add located_in property from the lab to the university
        model.add(model.createStatement(lab,
                model.createProperty(EIOntConstants.LOCATED_IN_URI), institution));
        // add location_of property from the university to the lab
        model.add(model.createStatement(institution,
                model.createProperty(EIOntConstants.LOCATION_OF_URI), lab));

        return super.generateLab(institutionURI, institutionLabel, labLabel, labURI, labURL, labTypeURI, genParams);
    }

    @Override
    public void generateResource(final String institutionURI, final String institutionLabel,
            final String resourceLabel, final String resourceURI, final String resourceTypeURI,
            final String resourceURL,
            final String labName, final String labURI, final DataGenParams genParams)
            throws IOException {

        // retrieve the resource for the institution
        final Resource institution = model.getResource(institutionURI);
        // retrieve the resource for the lab
        final Resource lab = model.getResource(labURI);
        // retrieve the resource for resource type
        final Resource resourceType = model.getResource(resourceTypeURI);
        // create a resource for the resource
        final Resource resource = model.createResource(resourceURI);
        // add the label
        model.add(model.createStatement(resource, RDFS.label, resourceLabel));
        // add the type (potentially with materialization)
        // add the direct type
        model.add(model.createStatement(resource, directType, resourceType));
        for (EIClass type: getTypes(eagleiOntModel, EIURI.create(resourceTypeURI), genParams.getMaterializeHierarchy())) {
            final Statement stmt = model.createStatement(resource, RDF.type, model.getResource(type.getEntity().getURI().toString()));
            model.add(stmt);
        }
        
        // add located_in property from the resource to the lab 
        // and location_of property from the lab to the resource
        addLocatedInAndLocationOf(resource, lab);
        
        // add located_in property from the resource to the institution
        // and location_of property from the institution to the resource
        addLocatedInAndLocationOf(resource, institution);        
    }
    
    private void addLocatedInAndLocationOf(Resource child, Resource parent) {
        // add located_in property from the child to the parent 
        Statement stmt = model.createStatement(child,
                model.createProperty(EIOntConstants.LOCATED_IN_URI), parent);
        model.add(stmt);
        // add location_of property from the lab to the instrument
        model.add(model.createStatement(parent,
                model.createProperty(EIOntConstants.LOCATION_OF_URI), child));  
    }
}
