/******************************************************************
 * File:        SafeGraph.java
 * Created by:  Dave Reynolds
 * Created on:  5 Oct 2009
 * 
 * (c) Copyright 2009, Hewlett-Packard Development Company, LP
 * [See end of file]
 * $Id: SafeGraph.java,v 1.1 2009/10/05 17:12:33 der Exp $
 *****************************************************************/

package com.hp.hpl.jena.reasoner.rulesys.impl;

import com.hp.hpl.jena.graph.Graph;
import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;
import com.hp.hpl.jena.graph.TripleMatch;
import com.hp.hpl.jena.graph.impl.SimpleEventManager;
import com.hp.hpl.jena.graph.impl.WrappedGraph;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import com.hp.hpl.jena.util.iterator.Filter;

/**
 * A SafeGraph wraps a graph which might contain generalized RDF
 * triples and hides them from API queries so that consumers
 * of it are safe (but can use getRawGraph() to get back the unsafe graph.
 * 
 * @author <a href="mailto:der@hplb.hpl.hp.com">Dave Reynolds</a>
 * @version $Revision: 1.1 $
 */
public class SafeGraph extends WrappedGraph implements Graph {

    /** Wrap a graph to hide generalized triples */
    public SafeGraph(Graph base) {
        super(base);
    }

    @Override
    public ExtendedIterator<Triple> find( TripleMatch m ) {
        return find(m.getMatchSubject(), m.getMatchPredicate(), m.getMatchObject());
    }
    
    @Override
    public ExtendedIterator<Triple> find( Node s, Node p, Node o ) {
        return SimpleEventManager.notifyingRemove( this, 
                base.find( s, p, o ).filterDrop( new Filter<Triple>() {
                    @Override
                    public boolean accept(Triple t) {
                        if (t.getSubject().isLiteral()) return true;
                        if (t.getPredicate().isBlank() || t.getPredicate().isLiteral()) return true;
                        return false;
                    }
                } ) );
    }

    /**
     * Return the unfiltered version of the graph
     */
    public Graph getRawGraph() {
        return base;
    }
    
}



/*
    (c) Copyright 2009 Hewlett-Packard Development Company, LP
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.

    3. The name of the author may not be used to endorse or promote products
       derived from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
    OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
    NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
    DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
    THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
    THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
