package org.eaglei.services.repository;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.eaglei.services.NodeConfig;

/**
 * Configuration for connecting to the repository.
 */
public final class RepositoryHttpConfig {

    public static final String EAGLEI_REPOSITORY_USERNAME_PROPERTY = "org.eaglei.repository.username";
    public static final String EAGLEI_REPOSITORY_PASSWORD_PROPERTY = "org.eaglei.repository.password";
    public static final String PUBLISHED_NG = "http://eagle-i.org/ont/repo/1.0/NG_Published";
    public static final String DEFAULT_NG = "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace";
    public static final String PUBLISHED_VIEW = "published-resources";

    private String hostURL;
    private String whoamiUrl;
    private String logoutUrl;
    private String sparqlUrl;
    private String harvestUrl;
    private String instanceUrl;

    private String searchUsername;
    private String searchPassword;
    // Cached on first request
    private HttpClient searchHttpClient;

    /**
     * Constructs a RepositoryHttpConfig from the given
     * node configuration.
     * 
     * @param tier Optional tier component to be added to the repo URL
     */
    public RepositoryHttpConfig(NodeConfig nodeConfig, String tier) {
        this(computeHostURL(tier, nodeConfig.getRepositoryHostUrl()));
        setSearchUsername(nodeConfig.getSearchUsername());
        setSearchPassword(nodeConfig.getSearchPassword());
    }

    /**
     * Constructs a client using the given host URL.
     * The host URL must not be null and must end with a trailing /.
     * For example, "https://qa.harvard.eagle-i.net/"
     * 
     * @param hostURL may not be null
     */
    public RepositoryHttpConfig(String hostURL) {
    	assert(hostURL != null);
    	this.hostURL = hostURL;
        String baseUrl = hostURL + "repository/";
        this.whoamiUrl = baseUrl + "whoami";
        this.logoutUrl = baseUrl + "logout/";
        this.sparqlUrl = baseUrl + "sparql";
        this.harvestUrl = baseUrl + "harvest";
        this.instanceUrl = hostURL + "i";
    }
    
    public String getHostURL() {
    	return this.hostURL;
    }

    /**
     * Convenience utility method for constructing a default http client
     * suitable for connecting to the repository.
     * 
     * @param username
     * @param password
     * @return
     */
    public static HttpClient createHttpClient(String username, String password) {
        HttpClient client = new HttpClient();
        client.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
        client.getState().setCredentials(AuthScope.ANY,
                new UsernamePasswordCredentials(username, password));
        client.getParams().setParameter("accept", "application/sparql-results+xml");
        client.getParams().setAuthenticationPreemptive(true);
        return client;
    }

    /**
     * Generates a repository host url using a tier and institution subdomain.
     * Supports default behavior if either is null.
     * 
     * @param tier
     * @param institutionSubdomain
     * @return
     */
    public static String computeHostURL(String tier, String repositoryHostUrl) {
        if (repositoryHostUrl == null) {
        	return ("localhost/");
        } else {
	        StringBuilder buf = new StringBuilder();
	        String domainOnly;
	        if (repositoryHostUrl.startsWith("https://")) {
	            domainOnly = repositoryHostUrl.substring(repositoryHostUrl.indexOf("https://".length()));
	        } else if (repositoryHostUrl.startsWith("http://")) {
                domainOnly = repositoryHostUrl.substring(repositoryHostUrl.indexOf("http://".length()));
	        } else {
	            domainOnly = repositoryHostUrl;
	        }
	        buf.append("https://");
            if (tier != null) {
                // Inject tier between // and remainder 
                buf.append(tier.toLowerCase());
                buf.append('.');
            }
            buf.append(domainOnly);
	        return buf.toString();
        }
    }

    public String getWhoamiUrl() {
        return this.whoamiUrl;
    }
    
    public String getLogoutUrl() {
        return this.logoutUrl;
    }
    
    public String getHarvestUrl() {
        return this.harvestUrl;
    }
    
    public String getSparqlUrl() {
        return this.sparqlUrl;
    }
    
    public String getInstanceUrl() {
    	return this.instanceUrl;
    }

    public String getSearchUsername() {
        return searchUsername;
    }

    public void setSearchUsername(String searchUsername) {
        this.searchUsername = searchUsername;
    }

    public String getSearchPassword() {
        return searchPassword;
    }

    public void setSearchPassword(String searchPassword) {
        this.searchPassword = searchPassword;
    }
    
    public HttpClient getSearchHttpClient() {
        if (searchHttpClient == null) {
            searchHttpClient = createHttpClient(getSearchUsername(), getSearchPassword());
        }
        return searchHttpClient;
    }
}
